import { Product, ProductSite } from '@sage/x3-master-data-api';
import { dialogMessage } from '@sage/x3-master-data/lib/client-functions/dialogs';
import * as ui from '@sage/xtrem-ui';
import { PurchaseReceiptsDetailPage } from './purchase-receipt-details-initializer';
export class ProductSiteData {
    constructor(productCode: string, site: string, page: PurchaseReceiptsDetailPage) {
        this.productCode = productCode;
        this.site = site;
        this.page = page;
    }

    private page: PurchaseReceiptsDetailPage;
    private productCode: string;
    private site: string;
    private _productSite: ProductSite;
    private _product: Product;

    private async _lazyLoadProductSite(): Promise<ProductSite | undefined> {
        if (this._productSite) {
            return this._productSite;
        }

        // read product site record
        try {
            const productSiteToReceive = (await this.page.$.graph
                .node('@sage/x3-master-data/ProductSite')
                .read(
                    {
                        isLicensePlateNumberManaged: true,
                        isLocationManaged: true,
                        stockSite: {
                            code: true,
                        },
                        defaultLocations: {
                            query: {
                                edges: {
                                    node: {
                                        defaultLocationType: true,
                                        locationNumber: true,
                                        defaultLocation: true,
                                    },
                                },
                            },
                        },
                        product: {
                            code: true,
                            productCategory: { code: true },
                            serialNumberManagementMode: true,
                            serialSequenceNumber: true,
                            localizedDescription1: true,
                            lotManagementMode: true,
                            lotSequenceNumber: true,
                            stockVersionMode: true,
                            expirationManagementMode: true,
                            stockManagementMode: true,
                            defaultPotencyInPercentage: true,
                            purchaseUnitToStockUnitConversionFactor: true,
                            useByDateCoefficient: true,
                            expirationLeadTime: true,
                            expirationTimeUnit: true,
                            stockUnit: {
                                code: true,
                                numberOfDecimals: true,
                            },
                            purchaseUnit: {
                                code: true,
                                numberOfDecimals: true,
                            },
                            packingUnits: {
                                query: {
                                    edges: {
                                        node: {
                                            packingUnit: {
                                                code: true,
                                                numberOfDecimals: true,
                                            },
                                            packingUnitToStockUnitConversionFactor: true,
                                            isPackingFactorEntryAllowed: true,
                                        },
                                    },
                                },
                            },
                        },
                        defaultInternalContainer: {
                            code: true,
                        },
                    },
                    `${this.productCode}|${this.site}`, // TODO: Better way to specify id for a Sage X3 record?
                )
                .execute()) as ProductSite;
            this._productSite = productSiteToReceive;
        } catch (e) {
            await dialogMessage(
                this.page,
                'error',
                ui.localize('@sage/x3-purchasing/error-loading-product-site', 'Error loading product-site'),
                String(e),
            );
            return undefined;
        }
        return this._productSite;
    }

    private async _lazyLoadProduct(): Promise<Product | undefined> {
        if (this._product) {
            return this._product;
        }

        // read product site record
        try {
            const productToReceive = (await this.page.$.graph
                .node('@sage/x3-master-data/Product')
                .read(
                    {
                        code: true,
                        productCategory: { code: true },
                        serialNumberManagementMode: true,
                        serialSequenceNumber: true,
                        localizedDescription1: true,
                        lotManagementMode: true,
                        lotSequenceNumber: true,
                        stockVersionMode: true,
                        expirationManagementMode: true,
                        stockManagementMode: true,
                        defaultPotencyInPercentage: true,
                        purchaseUnitToStockUnitConversionFactor: true,
                        useByDateCoefficient: true,
                        expirationLeadTime: true,
                        expirationTimeUnit: true,
                        stockUnit: {
                            code: true,
                            numberOfDecimals: true,
                        },
                        purchaseUnit: {
                            code: true,
                            numberOfDecimals: true,
                        },
                        packingUnits: {
                            query: {
                                edges: {
                                    node: {
                                        packingUnit: {
                                            code: true,
                                            numberOfDecimals: true,
                                        },
                                        packingUnitToStockUnitConversionFactor: true,
                                        isPackingFactorEntryAllowed: true,
                                    },
                                },
                            },
                        },
                    },
                    `${this.productCode}`, // TODO: Better way to specify id for a Sage X3 record?
                )
                .execute()) as Product;
            this._product = productToReceive;
        } catch (e) {
            await dialogMessage(
                this.page,
                'error',
                ui.localize('@sage/x3-purchasing/error-loading-product', 'Error loading product.'),
                String(e),
            );
            return undefined;
        }
        return this._product;
    }

    /*
     * The GraphQL query that will be used to get values of the product-site node.
     * This default query can be modified or completely overridden.
     */

    get productSite() {
        return this._lazyLoadProductSite();
    }

    get product() {
        return this._lazyLoadProduct();
    }
}
