import { getSelectedStockSite } from '@sage/x3-master-data/lib/client-functions/get-selected-stock-site';
import { PurchaseOrder } from '@sage/x3-purchasing-api';
import { receipt } from '@sage/x3-stock-data/build/lib/menu-items/receipt';
import { Filter } from '@sage/xtrem-client';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';
@ui.decorators.page<MobileViewPurchaseOrder, PurchaseOrder>({
    module: 'purchasing',
    title: 'View purchase orders',
    isTitleHidden: true,
    menuItem: receipt,
    priority: 400,
    mode: 'default',
    access: { node: '@sage/x3-purchasing/PurchaseOrder' },
    authorizationCode: 'INQPOH',
    skipDirtyCheck: true,
    async onLoad() {
        this.siteField.value = await getSelectedStockSite(
            this,
            ui.localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error'),
            ui.localize(
                '@sage/x3-purchasing-sales/dialog-error-location-inquiry-set-site',
                'Define a default stock site on the user function profile.',
            ),
        );

        if (!this.siteField.value) {
            this._disablePage();
        }

        this._initializeDate();
    },
    businessActions() {
        return [this.searchButton];
    },
})
export class MobileViewPurchaseOrder extends ui.Page {
    @ui.decorators.textField<MobileViewPurchaseOrder>({
        isHidden: true,
    })
    siteField: ui.fields.Text;

    @ui.decorators.section<MobileViewPurchaseOrder>({
        isTitleHidden: true,
    })
    mainSection: ui.containers.Section;

    @ui.decorators.block<MobileViewPurchaseOrder>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock: ui.containers.Block;

    @ui.decorators.separatorField<MobileViewPurchaseOrder>({
        parent() {
            return this.mainBlock;
        },
        isInvisible: true,
    })
    separatorField: ui.fields.Separator;

    @ui.decorators.dateField<MobileViewPurchaseOrder>({
        parent() {
            return this.mainBlock;
        },
        title: 'Expected receipt date',
        isFullWidth: true,
    })
    date: ui.fields.Date;

    @ui.decorators.referenceField<MobileViewPurchaseOrder, PurchaseOrder>({
        parent() {
            return this.mainBlock;
        },
        title: 'Purchase order',
        placeholder: 'Scan or select...',
        node: '@sage/x3-purchasing/PurchaseOrder',
        valueField: 'id',
        helperTextField: { orderFromSupplier: { code: { code: true } } },
        isHelperTextHidden: true,
        canFilter: false,
        filter() {
            const filter: Filter<PurchaseOrder> = {
                isClosed: { _eq: false },
                // (X3-227355) TODO Issue: Cannot use the less verbose _in operator instead of individual _or filter criterion
                _and: [
                    {
                        _or: [
                            {
                                signatureStatus: 'inFull',
                            },
                            {
                                signatureStatus: 'notManaged',
                            },
                            {
                                signatureStatus: 'automatic',
                            },
                        ],
                    },
                    {
                        _or: [
                            {
                                receiptStatus: 'no',
                            },
                            {
                                receiptStatus: 'inPart',
                            },
                        ],
                    },
                ],
                // filtering on PO's lines to exclude PO that have nothing to receive
                purchaseOrderLines: {
                    _atLeast: 1,
                    isClosed: { _eq: false },
                    receiptSite: { code: this.siteField.value },
                    sourceRequest: { _ne: 'directOrder' },
                    workOrderInProgressStatus: { _eq: 'firm' },
                    productType: { _ne: 'subcontract' },
                    lineType: { _eq: 'normal' },
                },
            };

            if (this.date.value) {
                filter.purchaseOrderLines = {
                    _atLeast: 1,
                    isClosed: { _eq: false },
                    receiptSite: { code: this.siteField.value },
                    sourceRequest: { _ne: 'directOrder' },
                    workOrderInProgressStatus: { _eq: 'firm' },
                    productType: { _ne: 'subcontract' },
                    lineType: { _eq: 'normal' },
                    expectedReceiptDate: {
                        _lte: this.date.value,
                    },
                };
            } else {
                filter.purchaseOrderLines = {
                    _atLeast: 1,
                    isClosed: { _eq: false },
                    receiptSite: { code: this.siteField.value },
                    sourceRequest: { _ne: 'directOrder' },
                    workOrderInProgressStatus: { _eq: 'firm' },
                    productType: { _ne: 'subcontract' },
                    lineType: { _eq: 'normal' },
                };
            }
            return filter;
        },
        isTransient: true,
        isFullWidth: true,
        isAutoSelectEnabled: true,
        columns: [
            ui.nestedFields.text({
                bind: 'id',
                title: 'Order no.',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'purchaseSite',
                valueField: 'code',
                title: 'Order site',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/Supplier',
                bind: 'orderFromSupplier',
                valueField: { code: { code: true } },
                title: 'Supplier',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/Supplier',
                bind: 'orderFromSupplier',
                valueField: { code: { shortCompanyName: true } },
                title: 'Supplier short name',
                isReadOnly: true,
                isHidden: true,
            }),
            // ui.nestedFields.reference({
            //     node: '@sage/x3-master-data/BusinessPartner',
            //     bind: 'orderFromSupplier',
            //     valueField: 'companyName',
            //     title: 'Name',
            //     isReadOnly: true,
            // }),
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/BusinessPartnerCompanyNames',
                bind: 'orderFromSupplier',
                valueField: { code: { code: true } },
                title: 'Name',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'internalOrderReference',
                title: 'Internal or order reference',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'isIntersite',
                title: 'Intersite',
                isReadOnly: true,
                isHidden: true,
            }),
        ],
    })
    purchaseOrder: ui.fields.Reference;

    @ui.decorators.pageAction<MobileViewPurchaseOrder>({
        title: 'Search',
        buttonType: 'primary',
        shortcut: ['f3'],
        async onClick() {
            if (this.purchaseOrder.value) {
                this.$.router.goTo('@sage/x3-purchasing-sales/MobileViewPurchaseOrderSelectAPurchaseOrderLine', {
                    purchaseOrder: this.purchaseOrder.value.id,
                    orderFromSupplier: this.purchaseOrder.value.orderFromSupplier.code.code,
                    site: this.siteField.value,
                    receiptDate: this.date.value ? this.date.value : '',
                    isIntersite: this.purchaseOrder.value.isIntersite,
                    shortCompanyName: this.purchaseOrder.value.orderFromSupplier.code.shortCompanyName,
                });
            } else {
                if (!this.date.value) {
                    this.$.storage.set('receiptDate', '');
                } else {
                    this.$.storage.set('receiptDate', this.date.value);
                }
                this.$.router.goTo('@sage/x3-purchasing-sales/MobileViewPurchaseOrderSelectAPurchaseOrder', {});
            }
        },
    })
    searchButton: ui.PageAction;

    private _disablePage(): void {
        this.searchButton.isDisabled = true;
        this.purchaseOrder.isDisabled = true;
        this.date.isDisabled = true;
    }

    private _initializeDate(): void {
        this.date.value = DateValue.today().toString();
    }
}
