import { Product } from '@sage/x3-master-data-api';
import { dialogMessage } from '@sage/x3-master-data/lib/client-functions/dialogs';
import { PurchaseOrderLine } from '@sage/x3-purchasing-api';
import { Filter } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';

/** @internal */
function generateDetailFilter(
    currentSite: string,
    expectedDate: string,
    selectedProduct: string,
): Filter<PurchaseOrderLine> {
    const orderLineFilter: Filter<PurchaseOrderLine> = {
        receiptSite: {
            _id: {
                _eq: currentSite,
            },
        },
        purchaseOrder: {
            signatureStatus: {
                _in: ['inFull', 'notManaged', 'automatic'],
            },
        },
        workOrderInProgressStatus: {
            _eq: 'firm',
        },
        sourceRequest: {
            _ne: 'directOrder',
        },
        lineType: {
            _in: ['normal', 'parentProductBom'],
        },
        lineStatus: {
            _ne: 'closed',
        },
        isClosed: {
            _eq: false as any,
        },
    };
    if (selectedProduct === '') {
        orderLineFilter.product = {
            productStatus: {
                _eq: 'active',
            },
        };
    } else {
        orderLineFilter.product = {
            code: selectedProduct,
            productStatus: {
                _eq: 'active',
            },
        };
    }
    if (expectedDate !== '') {
        orderLineFilter.expectedReceiptDate = {
            _lte: expectedDate,
        };
    }

    return orderLineFilter;
}
@ui.decorators.page<MobileViewPurchaseOrderToReceiveSelectAProduct, PurchaseOrderLine>({
    module: 'purchasing',
    title: 'Purchase orders to receive',
    subtitle: 'Select a product',
    isTitleHidden: true,
    node: '@sage/x3-purchasing/PurchaseOrderLine',
    mode: 'default',
    isTransient: false,
    authorizationCode: 'INQPOR',
    skipDirtyCheck: true,
    async onLoad() {
        if (!this.$.storage.get('mobile-selected-stock-site')) {
            await dialogMessage(
                this,
                'error',
                ui.localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error'),
                ui.localize('@sage/x3-purchasing-sales/dialog-error-set-site', 'Stock site not defined'),
            );
        } else {
            this.site.value = this.$.storage.get('mobile-selected-stock-site').toString();
        }
    },
    navigationPanel: {
        canFilter: false,
        isHeaderHidden: true,
        isAutoSelectEnabled: true,
        listItem: {
            title: ui.nestedFields.reference({
                title: 'Product Code: ',
                bind: 'product',
                valueField: 'code',
                node: '@sage/x3-master-data/Product',
            }),
            titleRight: ui.nestedFields.reference({
                title: 'Supplier Product: ',
                bind: 'supplierProduct',
                valueField: 'supplierProduct',
                node: '@sage/x3-master-data/SupplierProduct',
            }),
            line2: ui.nestedFields.reference<
                MobileViewPurchaseOrderToReceiveSelectAProduct,
                PurchaseOrderLine,
                Product
            >({
                bind: 'product',
                valueField: 'localizedDescription1',
                node: '@sage/x3-master-data/Product',
            }),
            line2Right: ui.nestedFields.date({
                bind: 'expectedReceiptDate',
                canFilter: false,
            }),
            line3: ui.nestedFields.reference({
                bind: 'purchaseOrder',
                valueField: 'id',
                node: '@sage/x3-purchasing/PurchaseOrder',
            }),
            line4: ui.nestedFields.reference({
                bind: 'supplierProduct',
                valueField: 'supplierUpc',
                isHidden: true,
                node: '@sage/x3-master-data/SupplierProduct',
            }),
            line5: ui.nestedFields.reference({
                bind: 'product',
                valueField: 'upc',
                isHidden: true,
                node: '@sage/x3-master-data/Product',
            }),
            line6: ui.nestedFields.numeric({
                bind: 'lineNumber',
                isHidden: false,
            }),
            line7: ui.nestedFields.reference({
                bind: 'purchaseOrder',
                valueField: 'isIntersite',
                node: '@sage/x3-purchasing/PurchaseOrder',
                isHidden: true,
            }),
        },
        optionsMenu: [
            {
                title: '',
                graphQLFilter: (storage: any, queryParameters: any) =>
                    generateDetailFilter(
                        storage.get('mobile-selected-stock-site').toString(),
                        storage.get('date').toString(),
                        queryParameters.product as string,
                    ),
            },
        ],
        onSelect(listItemValue: any) {
            this.$.router.goTo('@sage/x3-purchasing-sales/MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail', {
                _id: listItemValue._id,
                productCode: listItemValue.product.code,
                purchaseOrderId: listItemValue.purchaseOrder.id,
                lineNumber: listItemValue.lineNumber,
                upcCode: listItemValue.product.upc,
                receiptSite: this.site.value,
                isIntersite: listItemValue.purchaseOrder.isIntersite,
            });
            return true;
        },
        orderBy: {
            product: {
                code: 1,
            },
        },
    },
})
export class MobileViewPurchaseOrderToReceiveSelectAProduct extends ui.Page {
    @ui.decorators.section<MobileViewPurchaseOrderToReceiveSelectAProduct>({
        isHidden: true,
    })
    mainSection: ui.containers.Section;

    @ui.decorators.block<MobileViewPurchaseOrderToReceiveSelectAProduct>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock: ui.containers.Block;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceiveSelectAProduct>({
        isTransient: true,
    })
    site: ui.fields.Text;
}
