import { dialogMessage } from '@sage/x3-master-data/lib/client-functions/dialogs';
import { GraphApi, PurchaseOrderLine } from '@sage/x3-purchasing-api';
import { PurchaseReceiptDetailsRecordManager } from '@sage/x3-purchasing/lib/client-functions/purchase-receipt-details-control';
import { Filter } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import { goToPurchaseOrderSelectADelivery } from '../client-functions/purchase-order-control';

type ParameterType = string | number | boolean;
@ui.decorators.page<MobileViewPurchaseOrderSelectAPurchaseOrderLine>({
    module: 'x3-purchasing',
    title: 'Purchase orders',
    subtitle: 'Select a line',
    isTitleHidden: true,
    isTransient: true,
    skipDirtyCheck: true,
    headerCard() {
        return {
            title: this.purchaseOrder,
            line2: this.orderFromSupplier,
        };
    },
    async onLoad() {
        // Requires a selected purchase order number in the query parameters
        const purchaseOrderParameter: ParameterType = this.$.queryParameters.purchaseOrder;
        const orderFromSupplierParameter: ParameterType = this.$.queryParameters.orderFromSupplier;
        const receiptDateParameter: ParameterType = this.$.queryParameters.receiptDate;
        const isIntersiteParameter = !!this.$.queryParameters.isIntersite;

        if (!purchaseOrderParameter) {
            this.$.showToast(
                ui.localize('@sage/x3-purchasing-sales/purchaseOrder-required', 'Selected purchase order is required'),
                { type: 'warning' },
            );
            this.$.router.goTo(`@sage/x3-purchasing-sales/MobileViewPurchaseOrder`);
            return;
        }
        this.site.value = String(this.$.queryParameters.site);
        this.purchaseOrder.value = String(purchaseOrderParameter);
        this.orderFromSupplier.value = String(orderFromSupplierParameter);
        this.receiptDate.value = String(receiptDateParameter);
        const orderLineFilter: Filter<PurchaseOrderLine> = {
            purchaseOrder: this.purchaseOrder.value,
            receiptSite: this.site.value,
            workOrderInProgressStatus: { _lte: 'firm' },
            sourceRequest: { _ne: 'directOrder' },
            lineType: {
                _in: ['normal', 'parentProductBom'],
            },
            lineStatus: { _ne: 'closed' },
            isClosed: { _eq: false },
            product: {
                productStatus: {
                    _eq: 'active',
                },
            },
        };
        if (this.receiptDate.value !== '') {
            orderLineFilter.expectedReceiptDate = {
                _lte: this.receiptDate.value,
            };
        }
        try {
            const response = await this.$.graph
                .node('@sage/x3-purchasing/PurchaseOrderLine')
                .query(
                    ui.queryUtils.edgesSelector<PurchaseOrderLine>(
                        {
                            purchaseOrder: { id: true },
                            lineNumber: true,
                            receiptSite: { code: true },
                            quantityInPurchaseUnitOrdered: true,
                            quantityInStockUnitOrdered: true,
                            quantityInStockUnitReceived: true,
                            quantityInPurchaseUnitReceived: true,
                            purchaseUnit: {
                                code: true,
                            },
                            stockUnit: {
                                code: true,
                            },
                            product: {
                                code: true,
                                localizedDescription1: true,
                            },
                            lineStatus: true,
                            isClosed: true,
                            lineType: true,
                            workOrderInProgressStatus: true,
                            sourceRequest: true,
                            expectedReceiptDate: true,
                        },
                        {
                            filter: orderLineFilter,
                            orderBy: {
                                lineNumber: 1,
                            },
                            first: 200,
                        },
                    ),
                )
                .execute();

            let idCounter = 0;
            this.orderLines.value = response.edges.map((currentRecord: any) => {
                idCounter++;
                return {
                    _id: this.orderLines.generateRecordId(),
                    lineNumber: currentRecord.node.lineNumber,
                    product: currentRecord.node.product.code,
                    localizedDescription1: currentRecord.node.product.localizedDescription1,
                    quantityInStockUnitRemaining: `${
                        currentRecord.node.quantityInStockUnitOrdered - currentRecord.node.quantityInStockUnitReceived
                    } ${currentRecord.node.stockUnit.code}`,
                    expectedReceiptDate: currentRecord.node.expectedReceiptDate,
                };
            });

            this.mainBlock.title =
                ui.localize('@sage/x3-purchasing-sales/purchase-order-block-title', 'Purchase order lines:') +
                idCounter;
            this.purchaseReceiptButton.isHidden = isIntersiteParameter;
            this.intersiteReceiptButton.isHidden = !isIntersiteParameter;
        } catch (e) {
            await dialogMessage(
                this,
                'error',
                ui.localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error'),
                ui.localize('@sage/x3-purchasing-sales/dialog-error-set-site', 'Stock site not defined') + String(e),
            );
            this.$.loader.isHidden = true;
            return;
        }
    },
    businessActions() {
        return [this.purchaseReceiptButton, this.intersiteReceiptButton];
    },
})
export class MobileViewPurchaseOrderSelectAPurchaseOrderLine extends ui.Page<GraphApi> {
    @ui.decorators.section<MobileViewPurchaseOrderSelectAPurchaseOrderLine>({
        isTitleHidden: true,
    })
    mainSection: ui.containers.Section;

    @ui.decorators.block<MobileViewPurchaseOrderSelectAPurchaseOrderLine>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock: ui.containers.Block;

    @ui.decorators.textField<MobileViewPurchaseOrderSelectAPurchaseOrderLine>({
        isTransient: true,
        isReadOnly: true,
    })
    site: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderSelectAPurchaseOrderLine>({
        isTransient: true,
        isReadOnly: true,
    })
    purchaseOrder: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderSelectAPurchaseOrderLine>({
        isTransient: true,
        isReadOnly: true,
    })
    orderFromSupplier: ui.fields.Text;

    @ui.decorators.dateField<MobileViewPurchaseOrderSelectAPurchaseOrderLine>({
        isTransient: true,
        isReadOnly: true,
    })
    receiptDate: ui.fields.Date;

    @ui.decorators.tableField<MobileViewPurchaseOrderSelectAPurchaseOrderLine>({
        parent() {
            return this.mainBlock;
        },
        canSelect: false,
        isTransient: true,
        canUserHideColumns: false,
        canFilter: false,
        displayMode: ui.fields.TableDisplayMode.comfortable,
        onRowClick(_id: string, rowItem) {
            this.$.router.goTo(`@sage/x3-purchasing-sales/MobileViewPurchaseOrderPurchaseOrderLineDetail`, {
                site: this.site.value ?? '',
                purchaseOrder: this.purchaseOrder.value ?? '',
                product: rowItem.product,
                lineNumber: rowItem.lineNumber,
                expectedReceiptDate: rowItem.expectedReceiptDate,
                remainingQuantityReceive: rowItem.quantityInStockUnitRemaining,
            });
        },
        columns: [
            ui.nestedFields.text({
                bind: 'product',
                isReadOnly: true,
                canFilter: true,
            }),
            ui.nestedFields.text({
                bind: 'quantityInStockUnitRemaining',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'lineNumber',
                prefix: 'Line number: ',
                isReadOnly: true,
                canFilter: true,
            }),
            ui.nestedFields.date({
                bind: 'expectedReceiptDate',
                isReadOnly: true,
                canFilter: false,
            }),
        ],
        cardView: true,
        mobileCard: {
            title: ui.nestedFields.text({
                bind: 'product',
                isReadOnly: true,
            }),
            titleRight: ui.nestedFields.text({
                bind: 'quantityInStockUnitRemaining',
                isReadOnly: true,
            }),
            line2: ui.nestedFields.text({
                bind: 'localizedDescription1',
                isReadOnly: true,
            }),
            line3: ui.nestedFields.text({
                bind: 'lineNumber',
                prefix: 'Line number: ',
                isReadOnly: true,
            }),
            // (X3-258981) TODO Issue: This is displayed below line3, rather than on the right of line3
            line3Right: ui.nestedFields.date({
                bind: 'expectedReceiptDate',
                isReadOnly: true,
            }),
        },
    })
    orderLines: ui.fields.Table<any>;

    @ui.decorators.pageAction<MobileViewPurchaseOrderSelectAPurchaseOrderLine>({
        title: 'Purchase receipt',
        buttonType: 'primary',
        onClick() {
            const purchaseReceiptRecordManager = new PurchaseReceiptDetailsRecordManager(this, true);
            if (purchaseReceiptRecordManager.purchaseSession) {
                this.$.removeToasts();
                setTimeout(
                    () =>
                        this.$.showToast(
                            ui.localize(
                                '@sage/x3-purchasing-sales/notification-preloading-verification-purchase-receipt',
                                'Purchase receipt already in progress.  Starting a new purchase receipt is not allowed.',
                            ),
                            { type: 'warning', timeout: 30000 },
                        ),
                    10,
                );
            } else {
                this.$.router.goTo('@sage/x3-purchasing/MobilePurchaseReceipt', {
                    purchaseOrderId: this.purchaseOrder.value ?? '',
                    supplierCode: this.orderFromSupplier.value ?? '',
                });
            }
        },
    })
    purchaseReceiptButton: ui.PageAction;

    @ui.decorators.pageAction<MobileViewPurchaseOrderSelectAPurchaseOrderLine>({
        title: 'Intersite receipt',
        buttonType: 'primary',
        async onClick() {
            await goToPurchaseOrderSelectADelivery(this);
        },
    })
    intersiteReceiptButton: ui.PageAction;
}
