import { dialogMessage } from '@sage/x3-master-data/lib/client-functions/dialogs';
import { GraphApi } from '@sage/x3-purchasing-api';
import * as ui from '@sage/xtrem-ui';

type ParameterType = string | number | boolean;
@ui.decorators.page<MobileViewPurchaseOrderPurchaseOrderLineDetail>({
    module: 'x3-purchasing',
    title: 'Purchase orders',
    subtitle: 'Line details',
    isTitleHidden: true,
    isTransient: true,
    skipDirtyCheck: true,
    headerCard() {
        return {
            title: this.product,
            titleRight: this.remainingQuantityReceive,
            line2: this.localizedDescription1,
            line3: this.purchaseOrder,
            line3Right: this.expectedReceiptDate,
        };
    },

    async onLoad() {
        // Requires a selected purchase order number in the query parameters
        const purchaseOrderParameter: ParameterType = this.$.queryParameters.purchaseOrder;
        const lineNumberParameter: ParameterType = this.$.queryParameters.lineNumber;
        const productParameter: ParameterType = this.$.queryParameters.product;
        const expectedReceiptDateParameter: ParameterType = this.$.queryParameters.expectedReceiptDate;
        const remainingQuantityReceiveParameter: ParameterType = this.$.queryParameters.remainingQuantityReceive;
        const mess: string = ui.localize(
            '@sage/x3-purchasing-sales/purchaseOrder-required',
            'Selected purchase order is required',
        );
        if (!purchaseOrderParameter) {
            this.$.showToast(mess, { type: 'warning' });
            this.$.router.goTo(`@sage/x3-purchasing-sales/MobileViewPurchaseOrder`);
            return;
        }

        this.site.value = String(this.$.queryParameters.site);
        this.purchaseOrder.value = String(purchaseOrderParameter);
        this.lineNumber.value = String(lineNumberParameter);
        this.product.value = String(productParameter);
        const tmpDate = ui.formatDateToCurrentLocale(expectedReceiptDateParameter as string);
        this.expectedReceiptDate.value = `Expected ${tmpDate}`;
        this.remainingQuantityReceive.value = String(remainingQuantityReceiveParameter);
        try {
            const response = await this.$.graph
                .node('@sage/x3-purchasing/PurchaseOrderLine')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            purchaseOrder: { id: true },
                            lineNumber: true,
                            receiptSite: { code: true },
                            orderFromSupplier: { code: { code: true } },
                            expectedReceiptDate: true,
                            quantityInPurchaseUnitOrdered: true,
                            quantityInStockUnitOrdered: true,
                            quantityInPurchaseUnitReceived: true,
                            quantityInStockUnitReceived: true,
                            quantityInOrderUnitOrdered: true,
                            purchaseUnit: {
                                code: true,
                            },
                            stockUnit: {
                                code: true,
                            },
                            orderUnit: {
                                code: true,
                            },
                            product: {
                                code: true,
                                localizedDescription1: true,
                                upc: true,
                                isVersionManaged: true,
                            },
                            majorProductVersion: { code: true },
                            minorProductVersion: true,
                            supplierProduct: {
                                supplierProduct: true,
                            },
                        },
                        {
                            filter: {
                                purchaseOrder: this.purchaseOrder.value,
                                lineNumber: Number(this.lineNumber.value),
                                lineStatus: { _ne: 'closed' },
                            },
                            orderBy: {
                                lineNumber: 1,
                            },
                        },
                    ),
                )
                .execute();

            this.localizedDescription1.value = response.edges[0].node.product.localizedDescription1; // records with the same PO & line number should always have the same product code + localized description
            if (response.edges[0].node.product.isVersionManaged) {
                this.isProductVersionManaged = true;
            } else {
                this.isProductVersionManaged = false;
            }

            this.orderLine.value = response.edges.map((currentRecord: any) => {
                let supplierProduct: string;
                supplierProduct = '';
                if (currentRecord.node.supplierProduct !== null) {
                    supplierProduct = currentRecord.node.supplierProduct.supplierProduct;
                }
                return {
                    lineNumber: currentRecord.node.lineNumber,
                    productDescription: currentRecord.node.product.localizedDescription1,
                    productUpc: currentRecord.node.product.upc,
                    orderFromSupplier: currentRecord.node.orderFromSupplier.code.code,
                    supplierProduct: supplierProduct,
                    quantityInOrderUnitOrdered: `${currentRecord.node.quantityInOrderUnitOrdered} ${currentRecord.node.orderUnit.code}`,
                    quantityInStockUnitReceived: `${currentRecord.node.quantityInStockUnitReceived} ${currentRecord.node.stockUnit.code}`,
                    quantityInStockUnitOrdered: `${currentRecord.node.quantityInStockUnitOrdered} ${currentRecord.node.stockUnit.code}`,
                    quantityInPurchaseUnitOrdered: `${currentRecord.node.quantityInPurchaseUnitOrdered} ${currentRecord.node.purchaseUnit.code}`,
                    quantityInPurchaseUnitReceived: `${currentRecord.node.quantityInPurchaseUnitReceived} ${currentRecord.node.purchaseUnit.code}`,
                    majorProductVersion: currentRecord.node.majorProductVersion?.code,
                    minorProductVersion: currentRecord.node.minorProductVersion,
                };
            });
        } catch (e) {
            await dialogMessage(
                this,
                'error',
                ui.localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error'),
                ui.localize('@sage/x3-purchasing-sales/dialog-error-set-site', 'Stock site not defined') + String(e),
            );
            this.$.loader.isHidden = true;
            return;
        }
    },
})
export class MobileViewPurchaseOrderPurchaseOrderLineDetail extends ui.Page<GraphApi> {
    private isProductVersionManaged: boolean;

    @ui.decorators.section<MobileViewPurchaseOrderPurchaseOrderLineDetail>({
        isTitleHidden: true,
    })
    mainSection: ui.containers.Section;

    @ui.decorators.block<MobileViewPurchaseOrderPurchaseOrderLineDetail>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock: ui.containers.Block;

    @ui.decorators.textField<MobileViewPurchaseOrderPurchaseOrderLineDetail>({
        isTransient: true,
        isReadOnly: true,
    })
    purchaseOrder: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderPurchaseOrderLineDetail>({
        isTransient: true,
        isReadOnly: true,
    })
    expectedReceiptDate: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderPurchaseOrderLineDetail>({
        isTransient: true,
        isReadOnly: true,
    })
    lineNumber: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderPurchaseOrderLineDetail>({
        isTransient: true,
        isReadOnly: true,
    })
    product: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderPurchaseOrderLineDetail>({
        isTransient: true,
        isReadOnly: true,
    })
    localizedDescription1: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderPurchaseOrderLineDetail>({
        isTransient: true,
        isReadOnly: true,
    })
    site: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderPurchaseOrderLineDetail>({
        isTransient: true,
        isReadOnly: true,
    })
    remainingQuantityReceive: ui.fields.Text;

    @ui.decorators.detailListField<MobileViewPurchaseOrderPurchaseOrderLineDetail>({
        parent() {
            return this.mainBlock;
        },
        isTransient: true,
        fields: [
            ui.nestedFields.text({
                title: 'Line number ',
                bind: 'lineNumber',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Description',
                bind: 'productDescription',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Product UPC',
                bind: 'productUpc',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Supplier',
                bind: 'orderFromSupplier',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Supplier product',
                bind: 'supplierProduct',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Order qty.',
                bind: 'quantityInOrderUnitOrdered',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'STK order qty.',
                bind: 'quantityInStockUnitOrdered',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'STK rec. qty.',
                bind: 'quantityInStockUnitReceived',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'PUR order qty.',
                bind: 'quantityInPurchaseUnitOrdered',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'PUR receipt qty.',
                bind: 'quantityInPurchaseUnitReceived',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Major version',
                bind: 'majorProductVersion',
                isReadOnly: true,
                isHidden() {
                    return !this.isProductVersionManaged; // return true to hide the field and return false to not hide the field
                },
            }),
            ui.nestedFields.text({
                title: 'Minor version',
                bind: 'minorProductVersion',
                isReadOnly: true,
                isHidden() {
                    return !this.isProductVersionManaged;
                },
            }),
        ],
    })
    orderLine: ui.fields.DetailList;
}
