import { Container, ProductSite } from '@sage/x3-master-data-api';
import { dialogConfirmation, dialogMessage } from '@sage/x3-master-data/lib/client-functions/dialogs';
import { GraphApi } from '@sage/x3-purchasing-api';
import { LicensePlateNumber, Location } from '@sage/x3-stock-data-api';
import { Filter } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import {
    IntersiteReceiptDetailsRecordManager,
    IntersiteReceiptSession,
} from '../client-functions/intersite-receipt-details-control';
import {
    IntersiteReceiptDetailsInitializer,
    StockJournalNodeResponse,
} from '../client-functions/intersite-receipt-details-initializer';
import { readParameterValue } from '../client-functions/read-parameter';

export type packingUnit = {
    node: {
        packingUnit: {
            code: string;
            numberOfDecimals: number;
        };
        packingUnitToStockUnitConversionFactor: string;
        isPackingFactorEntryAllowed: boolean;
    };
};

@ui.decorators.page<MobileIntersiteReceiptEnterReceiptDetail>({
    title: 'Intersite receipt',
    subtitle: 'Enter details',
    module: 'x3-purchasing-sales',
    mode: 'default',
    isTransient: true,
    isTitleHidden: true,
    headerCard() {
        return {
            title: this.headerTitleLeft,
            titleRight: this.headerTitleRight,
            line2: this.headerLine2Left,
            line2Right: this.headerLine2Right,
            line3: this.headerLine3Left,
            line3Right: this.headerLine3Right,
        };
    },
    businessActions() {
        return [this.nextButton, this.submitButton];
    },
    async onLoad() {
        this.isLocationPreloaded = this.$.queryParameters?.isLocationPreloaded === '1';
        this.salesDeliveryId = this.$.queryParameters._id as string;
        this.salesDeliveryLineNumber = this.$.queryParameters.deliveryLineNumber as number;
        this.salesDeliveryLines = JSON.parse(this.$.queryParameters.deliveryLines as string);
        this._intersiteReceiptRecordManager = new IntersiteReceiptDetailsRecordManager(this);
        this.intersiteSession = this._intersiteReceiptRecordManager.intersiteSession;
        if (!this.intersiteSession) {
            this.submitButton.isHidden = true;
            this.nextButton.isHidden = true;
            this.blockProduct.isHidden = true;
        } else if (!(await IntersiteReceiptDetailsInitializer.initializePage(this as any))) {
            this.$.router.goTo('@sage/x3-purchasing-sales/MobileIntersiteReceipt', {
                _id: this.salesDeliveryId,
            });
        } else {
            this._displayStockJournal(this.stockJournalNodeResponses[this.currStockJournal]);
            this.nextButton.isHidden = this.stockJournalNodeResponses.length === 1;
            this._isOverReceiptAllowed =
                (await readParameterValue('RCPORD', this.$.userCode ?? '', this.site.value ?? '', this)) === 2;
            this._initDirty();
            await this._focusToNextField(undefined);
        }
    },
})
export class MobileIntersiteReceiptEnterReceiptDetail extends ui.Page<GraphApi> {
    /*
     *
     *  Technical properties
     *
     */

    private _intersiteReceiptRecordManager: IntersiteReceiptDetailsRecordManager; // to store the receipt to create from session data
    public intersiteSession: IntersiteReceiptSession;
    private _isOverReceiptAllowed: boolean | undefined;

    receiptUnits: packingUnit[];
    stockJournalNodeResponses: StockJournalNodeResponse[];
    currStockJournal: number;
    productSite: ProductSite;
    salesDeliveryId: string;
    salesDeliveryLineNumber: string | number | boolean;
    salesDeliveryLines: string[];
    sequence: number;
    isExpirationManaged: boolean;
    salesUnitToStockUnitConversionFactor: string;
    previousSalesUnitToStockUnitConversionFactor: string;
    quantityToReceiveNoRounded: number;
    quantityToReceiveRounded: number;
    remainingQuantityNoRounded: number;
    remainingQuantityRounded: number;
    remainingQuantityInitial: number;
    isLocationPreloaded: boolean;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerTitleLeft: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerTitleRight: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerLine2Left: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerLine2Right: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerLine3Left: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerLine3Right: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        title: 'Product',
        isReadOnly: true,
        size: 'large',
    })
    product: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        title: 'Description',
        isReadOnly: true,
        size: 'small',
    })
    description: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        title: 'Site',
        isReadOnly: true,
        isTransient: true,
    })
    site: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        title: 'Supplier',
        isReadOnly: true,
        isTransient: true,
    })
    supplier: ui.fields.Text;

    /*
     *
     *  Page Actions
     *
     */

    @ui.decorators.pageAction<MobileIntersiteReceiptEnterReceiptDetail>({
        title: 'Submit',
        shortcut: ['f3'],
        buttonType: 'primary',
        async onClick() {
            await this.$.commitValueAndPropertyChanges();
            if (await this._validate()) {
                if (this.intersiteSession?.purchaseReceipt?.lines) {
                    this.intersiteSession.purchaseReceipt.lines[
                        this.intersiteSession.purchaseReceipt.lines.length - 1
                    ].stockDetails?.push({
                        product: this.product.value ?? undefined,
                        quantityInPackingUnit: this.quantityToReceive.value ?? undefined,
                        packingUnit: this.receiptUnit.value ?? undefined,
                        packingUnitToStockUnitConversionFactor:
                            this.packingUnitToStockUnitConversionFactor.value ?? undefined,
                        location: this.location.value?.code,
                        licensePlateNumber: this.licensePlateNumber.value?.code ?? undefined,
                        status: this.status.value ?? undefined,
                        serialNumber: this._getPayloadFieldString(this.serialNumber.value),
                        supplierLot: this._getPayloadFieldString(this.supplierLot.value),
                        lot: this._getPayloadFieldString(this.lot.value),
                        sublot: this._getPayloadFieldString(this.sublot.value),
                        lotCustomField1: this._getPayloadFieldString(this.lotCustomField1.value),
                        lotCustomField2: this._getPayloadFieldString(this.lotCustomField2.value),
                        lotCustomField3: this._getPayloadFieldNumber(this.lotCustomField3.value),
                        lotCustomField4: this._getPayloadFieldDate(this.lotCustomField4.value),
                        majorVersion: this._getPayloadFieldString(this.majorVersion.value),
                        minorVersion: this._getPayloadFieldString(this.minorVersion.value),
                        identifier1: this._getPayloadFieldString(this.identifier1.value),
                        identifier2: this._getPayloadFieldString(this.identifier2.value),
                        expirationDate: this._getPayloadFieldDate(this.expirationDate.value),
                        useByDate: this._getPayloadFieldDate(this.useByDate.value),
                        sequence: this.sequence,
                    });
                }
                this.remainingQuantity.value = this._calculateRemainingQuantity(
                    this.stockJournalNodeResponses[this.currStockJournal],
                );
                if (this.remainingQuantity.value === 0) {
                    if (this._getNextStockJournal()) {
                        this._displayStockJournal(this.stockJournalNodeResponses[this.currStockJournal]);
                        this.receiptUnit.isDisabled = true;
                        this.container.isDisabled = true;
                        this.licensePlateNumber.isDisabled = true;
                        this.nextButton.isHidden = this._getNextButtonHidden();
                        this.quantityToReceive.focus();
                    } else {
                        await this._updatePurchaseReceiptLine();
                    }
                } else {
                    this._displayStockJournal(this.stockJournalNodeResponses[this.currStockJournal]);
                    this.receiptUnit.isDisabled = true;
                    this.container.isDisabled = true;
                    this.licensePlateNumber.isDisabled = true;
                    this.quantityToReceive.focus();
                }
            }
        },
    })
    submitButton: ui.PageAction;

    @ui.decorators.pageAction<MobileIntersiteReceiptEnterReceiptDetail>({
        title: 'Next',
        shortcut: ['f2'],
        buttonType: 'secondary',
        async onClick() {
            if (this._getNextStockJournal()) {
                this._displayStockJournal(this.stockJournalNodeResponses[this.currStockJournal]);
                this.nextButton.isHidden = this._getNextButtonHidden();
                this.quantityToReceive.focus();
            } else {
                if (
                    this.intersiteSession?.purchaseReceipt?.lines &&
                    this.intersiteSession?.purchaseReceipt?.lines[
                        this.intersiteSession?.purchaseReceipt?.lines?.length - 1
                    ].stockDetails?.length !== 0
                ) {
                    await this._updatePurchaseReceiptLine();
                } else {
                    this.intersiteSession?.purchaseReceipt?.lines?.splice(
                        this.intersiteSession?.purchaseReceipt?.lines.length - 1,
                    );
                    this._intersiteReceiptRecordManager.intersiteSession = this.intersiteSession;
                    this.$.setPageClean();
                    this.$.router.goTo('@sage/x3-purchasing-sales/MobileIntersiteReceiptSelectFromDelivery', {
                        _id: this.salesDeliveryId,
                        deliveryLineNumber: this.salesDeliveryLineNumber,
                        deliveryLines: JSON.stringify(this.salesDeliveryLines),
                        isLocationPreloaded: `${this.isLocationPreloaded ? '1' : '0'}`,
                    });
                }
            }
        },
    })
    nextButton: ui.PageAction;

    private _getNextStockJournal(): StockJournalNodeResponse | null {
        const startingCurrStockJournal = this.currStockJournal;
        let lookForStockJournals: boolean;
        lookForStockJournals = true;
        let nextStockJournal: StockJournalNodeResponse | null;
        nextStockJournal = null;
        while (lookForStockJournals) {
            if (this.currStockJournal < this.stockJournalNodeResponses.length - 1) {
                this.currStockJournal += 1;
            } else {
                this.currStockJournal = 0;
            }
            if (this._calculateRemainingQuantity(this.stockJournalNodeResponses[this.currStockJournal]) !== 0) {
                lookForStockJournals = false;
                nextStockJournal = this.stockJournalNodeResponses[this.currStockJournal];
            }
            if (this.currStockJournal === startingCurrStockJournal) {
                lookForStockJournals = false;
            }
        }
        if (!nextStockJournal) {
            this.currStockJournal = startingCurrStockJournal;
        }
        return nextStockJournal;
    }

    private _getNextButtonHidden(): boolean {
        let stockJournalWithRemainingQuantityNotNullCount: number;
        stockJournalWithRemainingQuantityNotNullCount = 0;
        this.stockJournalNodeResponses.forEach(stockJournal => {
            if (this._calculateRemainingQuantity(stockJournal) !== 0) {
                stockJournalWithRemainingQuantityNotNullCount += 1;
            }
        });
        return stockJournalWithRemainingQuantityNotNullCount <= 1;
    }

    private async _updatePurchaseReceiptLine() {
        if (
            this.intersiteSession &&
            this.intersiteSession.purchaseReceipt &&
            this.intersiteSession.purchaseReceipt.lines
        ) {
            this.intersiteSession.purchaseReceipt.lines[this.intersiteSession.purchaseReceipt.lines.length - 1] = {
                ...this.intersiteSession.purchaseReceipt.lines[this.intersiteSession.purchaseReceipt.lines.length - 1],
                lineNumber: Number(this.intersiteSession.purchaseReceipt.lines.length * 1000),
                receiptUnit: this.receiptUnit.value ?? '',
                receiptUnitToStockUnitConversionFactor: this.salesUnitToStockUnitConversionFactor,
                // container: this.container.value?.code,
            };
            let quantityInReceiveUnit: number;
            quantityInReceiveUnit = 0;
            this.intersiteSession.purchaseReceipt.lines[
                this.intersiteSession.purchaseReceipt.lines.length - 1
            ].stockDetails?.forEach(item => {
                quantityInReceiveUnit += Number(item.quantityInPackingUnit);
            });
            this.intersiteSession.purchaseReceipt.lines[
                this.intersiteSession.purchaseReceipt.lines.length - 1
            ].quantityInReceiptUnitReceived = quantityInReceiveUnit;
            this._intersiteReceiptRecordManager.intersiteSession = this.intersiteSession;

            const lineIndex = this.salesDeliveryLines.findIndex(
                lineNumber => lineNumber === this.salesDeliveryLineNumber,
            );
            if (lineIndex >= 0) {
                this.salesDeliveryLines.splice(lineIndex, 1);
            }
            if (this.salesDeliveryLines.length > 0) {
                await this._goToPageTwo();
            } else {
                await this._goToPageOne();
            }
        }
    }

    private async _goToPageTwo(): Promise<void> {
        const errors = await this.$.page.validate();
        if (errors.length === 0) {
            this.$.setPageClean();
            this.$.router.goTo('@sage/x3-purchasing-sales/MobileIntersiteReceiptSelectFromDelivery', {
                _id: this.salesDeliveryId,
                deliveryLineNumber: this.salesDeliveryLineNumber,
                deliveryLines: JSON.stringify(this.salesDeliveryLines),
                isLocationPreloaded: `${this.isLocationPreloaded ? '1' : '0'}`,
            });
        } else {
            this.$.showToast(`${ui.localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error')}: ${errors[0]}`, {
                type: 'error',
                timeout: 30000,
            });
        }
    }

    private async _goToPageOne(): Promise<void> {
        const errors = await this.$.page.validate();
        if (errors.length === 0) {
            this.$.setPageClean();
            this.$.router.goTo('@sage/x3-purchasing-sales/MobileIntersiteReceipt', {
                _id: this.salesDeliveryId,
            });
        } else {
            this.$.showToast(`${ui.localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error')}: ${errors[0]}`, {
                type: 'error',
                timeout: 30000,
            });
        }
    }

    private _displayStockJournal(stockJournal: StockJournalNodeResponse) {
        this.sequence = stockJournal.sequence;
        this.remainingQuantity.value = this._calculateRemainingQuantity(stockJournal);
        this.quantityToReceive.value = this.remainingQuantity.value;
        this.packingUnitToStockUnitConversionFactor.value = Number(this.salesUnitToStockUnitConversionFactor);
        if (stockJournal.startingSerialNumber) {
            this.serialNumber.value = stockJournal.startingSerialNumber;
        } else {
            this.serialNumber.value = stockJournal.serialNumber;
        }
        this.supplierLot.value = stockJournal.supplierLot;
        this.lot.value = stockJournal.lot;
        this.sublot.value = stockJournal.sublot;
        this.majorVersion.value = stockJournal.majorVersion;
        this.minorVersion.value = stockJournal.minorVersion;
        this.lotCustomField1.value = stockJournal.lotCustomField1;
        this.lotCustomField2.value = stockJournal.lotCustomField2;
        this.lotCustomField3.value = Number(stockJournal.lotCustomField3);
        this.lotCustomField4.value = stockJournal.lotCustomField4?.startsWith('1599')
            ? null
            : stockJournal.lotCustomField4;
        this.identifier1.value = stockJournal.identifier1;
        this.identifier2.value = stockJournal.identifier2;
        this.expirationDate.value = stockJournal.expirationDate;
        this.useByDate.value = stockJournal.useByDate;
        this._showSubmit();
    }

    private _calculateRemainingQuantity(stockJournal: StockJournalNodeResponse): number {
        let pendingQuantityInReceiptUnitReceived = 0;
        if (this.intersiteSession?.purchaseReceipt?.lines) {
            this.intersiteSession.purchaseReceipt.lines.forEach(line => {
                if (
                    line.salesDelivery === this.salesDeliveryId &&
                    line.salesDeliveryLine === this.salesDeliveryLineNumber
                ) {
                    line.stockDetails?.forEach(stockDetail => {
                        if (stockDetail.sequence === stockJournal.sequence) {
                            pendingQuantityInReceiptUnitReceived += Number(stockDetail.quantityInPackingUnit);
                        }
                    });
                }
            });
        }
        let _remainingQuantity: number;
        _remainingQuantity = Math.max(
            0,
            -(Number(this.salesUnitToStockUnitConversionFactor) !== 0
                ? stockJournal.quantityInStockUnit / Number(this.salesUnitToStockUnitConversionFactor)
                : 0) - pendingQuantityInReceiptUnitReceived,
        );
        const selectedUnit = this.receiptUnits.find(line => line.node.packingUnit.code === this.receiptUnit.value);
        const tmp = Math.pow(10, Number(selectedUnit?.node?.packingUnit?.numberOfDecimals));
        if (tmp !== 0) {
            _remainingQuantity = Math.round(_remainingQuantity * tmp) / tmp;
        }
        return _remainingQuantity;
    }

    /*
     *
     *  Sections
     *
     */

    @ui.decorators.section<MobileIntersiteReceiptEnterReceiptDetail>({
        isTitleHidden: true,
    })
    sectionHeader: ui.containers.Section;

    /*
     *
     *  Blocks
     *
     */

    @ui.decorators.block<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.sectionHeader;
        },
        isTitleHidden: false,
    })
    blockProduct: ui.containers.Block;

    /*
     *
     *  Fields
     *
     */

    @ui.decorators.selectField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Unit',
        placeholder: 'Enter unit',
        options: ['UN'],
        isMandatory: true,
        onChange() {
            if (this.receiptUnit.value) {
                const selectedUnit = this.receiptUnits.find(
                    packingUnit => packingUnit.node.packingUnit.code === this.receiptUnit.value,
                );
                if (selectedUnit) {
                    this.packingUnitToStockUnitConversionFactor.value = Number(
                        selectedUnit.node.packingUnitToStockUnitConversionFactor,
                    );
                    this.packingUnitToStockUnitConversionFactor.isDisabled =
                        !selectedUnit.node.isPackingFactorEntryAllowed;
                } else {
                    this.packingUnitToStockUnitConversionFactor.value = 1;
                    this.packingUnitToStockUnitConversionFactor.isDisabled = true;
                }
                this.quantityToReceive.scale = selectedUnit?.node.packingUnit.numberOfDecimals;
                this.salesUnitToStockUnitConversionFactor = String(this.packingUnitToStockUnitConversionFactor.value);

                this.quantityToReceiveNoRounded =
                    (Number(this.quantityToReceive.value) * Number(this.previousSalesUnitToStockUnitConversionFactor)) /
                    this.packingUnitToStockUnitConversionFactor.value;
                const tmp = Math.pow(10, Number(selectedUnit?.node?.packingUnit?.numberOfDecimals));
                if (tmp !== 0) {
                    this.quantityToReceiveRounded = Math.round(this.quantityToReceiveNoRounded * tmp) / tmp;
                } else {
                    this.quantityToReceiveRounded = this.quantityToReceiveNoRounded;
                }
                this.quantityToReceive.value = this.quantityToReceiveRounded;
                this.quantityToReceive.focus();
                this.previousReceiptUnit.value = this.receiptUnit.value;
                this.previousSalesUnitToStockUnitConversionFactor = this.salesUnitToStockUnitConversionFactor;
            }
            this._showSubmit();
        },
    })
    receiptUnit: ui.fields.Select;

    @ui.decorators.labelField<MobileIntersiteReceiptEnterReceiptDetail>({
        isHidden: true,
        isTransient: true,
    })
    receiptUnitCode: ui.fields.Label;

    @ui.decorators.labelField<MobileIntersiteReceiptEnterReceiptDetail>({
        isHidden: true,
        isTransient: true,
    })
    previousReceiptUnit: ui.fields.Label;

    @ui.decorators.numericField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Quantity',
        placeholder: 'Enter quantity',
        isMandatory: true,
        validation(value: number) {
            const regex = /^([1-9][0-9]*(\.[0-9]+)?|[0]+\.[0-9]*[1-9][0-9]*)$/; // reg ex for any positive numbers (integers or decimals) excluding 0
            if ((value.toString().match(regex)?.length ?? 0) === 0) {
                return ui.localize('@sage/x3-purchasing-sales/validate-error-invalid-value', 'invalid value');
            }
            if (
                this.salesDeliveryId &&
                this.salesDeliveryLineNumber &&
                (this.quantityToReceive.value ?? 0) > (this.remainingQuantity.value ?? 0) &&
                !this._isOverReceiptAllowed
            ) {
                return ui.localize(
                    '@sage/x3-purchasing-sales/validation-error-intersite-receipt-details-quantity-received-is-greater-than-the-ordered-quantity',
                    'The quantity received is greater than the ordered quantity.',
                );
            }
            return undefined;
        },
        scale() {
            const selectedUnit = this.receiptUnits.find(unit => unit.node.packingUnit.code === this.receiptUnit.value);
            return selectedUnit?.node.packingUnit.numberOfDecimals ?? 0;
        },
        min: 0,
        async onChange() {
            if (this.salesDeliveryId && this.salesDeliveryLineNumber) {
                if ((this.quantityToReceive.value ?? 0) > (this.remainingQuantity.value ?? 0)) {
                    if (!this._isOverReceiptAllowed) {
                        await dialogMessage(
                            this,
                            'error',
                            ui.localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error'),
                            ui.localize(
                                '@sage/x3-purchasing-sales/dialog-error-intersite-receipt-details-quantity-received-is-greater-than-the-ordered-quantity',
                                `The quantity received is greater than the ordered quantity.`,
                            ),
                            {
                                fullScreen: false,
                                acceptButton: {
                                    text: ui.localize('@sage/x3-purchasing-sales/button-accept-ok', 'OK'),
                                },
                            },
                        );
                        this.quantityToReceive.focus();
                        return;
                    } else {
                        if (
                            !(await dialogConfirmation(
                                this,
                                'warn',
                                ui.localize('@sage/x3-purchasing-sales/dialog-warning-title', 'Warning'),
                                ui.localize(
                                    '@sage/x3-purchasing-sales/dialog-warn-intersite-receipt-details-quantity-message',
                                    'Quantity larger than expected. Validate ?',
                                ),
                                {
                                    fullScreen: false,
                                    acceptButton: {
                                        text: ui.localize('@sage/x3-purchasing-sales/button-accept-yes', 'Yes'),
                                    },
                                    cancelButton: {
                                        text: ui.localize('@sage/x3-purchasing-sales/button-cancel-no', 'No'),
                                    },
                                }, //
                            ))
                        ) {
                            this.quantityToReceive.value = 0;
                            await this.$.commitValueAndPropertyChanges();
                            this.quantityToReceive.focus();
                        }
                    }
                }
            }
            if (!this.quantityToReceive.value) {
                this.quantityToReceive.value = 0;
            }
            if (this.quantityToReceive.value !== this.quantityToReceiveRounded) {
                this.quantityToReceiveNoRounded = this.quantityToReceive.value;
            }
            this.quantityToReceiveRounded = this.quantityToReceive.value;
            this._showSubmit();
        },
    })
    quantityToReceive: ui.fields.Numeric;

    @ui.decorators.numericField<MobileIntersiteReceiptEnterReceiptDetail>({
        isReadOnly: true,
        isHidden: true,
    })
    remainingQuantity: ui.fields.Numeric;

    @ui.decorators.numericField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Conversion factor',
        isDisabled: true,
        isHidden: false,

        onChange() {
            if (this.intersiteSession?.purchaseReceipt?.lines) {
                this.intersiteSession.purchaseReceipt.lines.forEach(line => {
                    if (
                        line.salesDelivery === this.salesDeliveryId &&
                        line.salesDeliveryLine === this.salesDeliveryLineNumber
                    ) {
                        line.stockDetails?.forEach(stockDetail => {
                            if (
                                this.packingUnitToStockUnitConversionFactor.value !==
                                stockDetail.packingUnitToStockUnitConversionFactor
                            ) {
                                this.packingUnitToStockUnitConversionFactor.value = Number(
                                    stockDetail.packingUnitToStockUnitConversionFactor,
                                );
                                this.packingUnitToStockUnitConversionFactor.focus();
                                this.$.showToast(
                                    `${ui.localize(
                                        '@sage/x3-purchasing-sales/notification-The-conversion-factor-can-t-be-different-as-the-one-entered-previously-on-the-same-stock-line',
                                        "The conversion factor can't be different as the one entered previously on the same stock line.",
                                    )}`,
                                    { type: 'error', timeout: 5000 },
                                );
                            }
                        });
                    }
                });
            }

            this.salesUnitToStockUnitConversionFactor = String(this.packingUnitToStockUnitConversionFactor.value);
        },
    })
    packingUnitToStockUnitConversionFactor: ui.fields.Numeric;

    @ui.decorators.selectField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Status',
        placeholder: 'Enter status',
        isMandatory: true,
        onChange() {
            this.status.getNextField(true)?.focus();
        },
    })
    status: ui.fields.Select;

    @ui.decorators.referenceField<MobileIntersiteReceiptEnterReceiptDetail, Container>({
        parent() {
            return this.blockProduct;
        },
        title: 'Container',
        node: '@sage/x3-master-data/Container',
        valueField: 'code',
        isAutoSelectEnabled: true,
        placeholder: 'Scan or select...',
        width: 'large',
        isTransient: true,
        canFilter: false,
        filter() {
            return {
                isInternal: { _eq: true },
                isActive: { _eq: true },
            };
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Code',
            }),
            ui.nestedFields.text({
                bind: 'containerType',
                title: 'Type',
            }),
            ui.nestedFields.text({
                bind: 'description',
                title: 'Description',
            }),
        ],
    })
    container: ui.fields.Reference;

    @ui.decorators.referenceField<MobileIntersiteReceiptEnterReceiptDetail, LicensePlateNumber>({
        parent() {
            return this.blockProduct;
        },
        title: 'License plate number',
        placeholder: 'Scan or select...',
        isFullWidth: true,
        node: '@sage/x3-stock-data/LicensePlateNumber',
        isAutoSelectEnabled: true,
        valueField: 'code',
        canFilter: false,
        filter() {
            let licensePlateNumberFilter: Filter<LicensePlateNumber> = {
                stockSite: { code: this.site.value ?? undefined },
                isActive: { _eq: true },
                _or: [
                    {
                        isSingleProduct: { _eq: true },
                        stock: { _atLeast: 1, product: { product: { code: this.product.value ?? undefined } } },
                    },
                    { isSingleProduct: { _eq: true }, stock: { _none: true } },
                    { isSingleProduct: { _eq: false } },
                ],
            };

            if (this.location.value?.code) {
                licensePlateNumberFilter._or = [
                    {
                        location: { code: this.location.value.code },
                    },
                    {
                        _and: [{ location: { code: undefined } }, { status: 'free' }],
                    },
                ];
            }

            if (this.container.value) {
                licensePlateNumberFilter = {
                    ...licensePlateNumberFilter,
                    container: { code: (this.container.value as Container)?.code },
                };
            }

            return licensePlateNumberFilter;
        },
        async onChange() {
            if (!this.licensePlateNumber.value) {
                this.location.isDisabled = false;
                this.location.value = null;
                this.container.isDisabled = false;
                await this.$.commitValueAndPropertyChanges(); // without this, when you clear out LPN and then, without tabbing out, click Location's lookup button directly, nothing will happen
                return;
            }

            if (!!this.licensePlateNumber?.value?.location?.code) {
                this.location.isDisabled = true;
                this.location.value = this.licensePlateNumber.value.location;
                if ((this.licensePlateNumber.value as LicensePlateNumber).container) {
                    this.container.value = (this.licensePlateNumber.value as LicensePlateNumber).container;
                    this.container.isDisabled = true;
                } else {
                    this.container.isDisabled = false;
                }
            } else {
                if (this.location.isDisabled) {
                    this.location.value = null;
                    this.location.isDisabled = false;
                }
                this.container.isDisabled = false;
            }

            await this.$.commitValueAndPropertyChanges();
            this._showSubmit();
            this.licensePlateNumber.getNextField(true)?.focus();
        },
        isTransient: true,
        isMandatory: false,
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'License Plate Number',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-stock-data/Location',
                bind: 'location',
                valueField: 'code',
                title: 'Location',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/Container',
                bind: 'container',
                valueField: 'code',
                title: 'Container',
                isReadOnly: true,
            }),
            ui.nestedFields.label({
                bind: 'status',
                title: 'Status',
                map(value: any) {
                    switch (value) {
                        case 'free':
                            return 'Free';
                        case 'inStock':
                            return 'In Stock';
                        default:
                            return '';
                    }
                },
                borderColor: ui.tokens.colorsYang100,
                optionType: '@sage/x3-stock-data/ContainerStatus',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'stockSite',
                valueField: 'code',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isActive',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isSingleProduct',
                isHidden: true,
            }),
        ],
    })
    licensePlateNumber: ui.fields.Reference;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Suggested Location',
        //node: '@sage/xtrem-x3-stock/Location',
        //valueField: 'code',
        isReadOnly: true,
        width: 'large',
        isTransient: true,
    })
    suggestedLocation: ui.fields.Text;

    @ui.decorators.referenceField<MobileIntersiteReceiptEnterReceiptDetail, Location>({
        parent() {
            return this.blockProduct;
        },
        title: 'Location',
        placeholder: 'Scan or select...',
        node: '@sage/x3-stock-data/Location',
        isAutoSelectEnabled: true,
        valueField: 'code',
        canFilter: false,
        filter() {
            return {
                stockSite: { code: this.site.value ?? undefined },
            };
        },
        onChange() {
            this._showSubmit();
            if (this.location.value) this.location.getNextField(true)?.focus();
            this.warehouse.value = !this.location.value?.warehouse ? '' : this.location.value.warehouse.code;
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Code',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'type',
                title: 'Type',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'stockSite',
                valueField: 'code',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-stock-data/Warehouse',
                bind: 'warehouse',
                valueField: 'code',
                isHidden: true,
            }),
        ],
        minLookupCharacters: 1,
        isFullWidth: true,
        isMandatory: true,
    })
    location: ui.fields.Reference;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        isHidden: true,
        isReadOnly: true,
    })
    warehouse: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Serial number',
        placeholder: 'Scan...',
        isFullWidth: true,
        validation: /^$|^[^|]+$/,
        isDisabled: true,
    })
    serialNumber: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Supplier lot',
        placeholder: 'Scan...',
        isFullWidth: true,
        validation: /^$|^[^|]+$/,
        isDisabled: true,
    })
    supplierLot: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Lot',
        placeholder: 'Scan...',
        isFullWidth: true,
        validation: /^$|^[^|]+$/,
        isTransient: true,
        isDisabled: true,
    })
    lot: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Sublot',
        placeholder: 'Scan...',
        isFullWidth: true,
        validation: /^$|^[^|]+$/,
        isDisabled: true,
    })
    sublot: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Major version',
        placeholder: 'Scan...',
        isDisabled: true,
    })
    majorVersion: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Minor version',
        placeholder: 'Scan...',
        isDisabled: true,
    })
    minorVersion: ui.fields.Text;

    @ui.decorators.dateField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Expiration date',
        placeholder: 'Enter an expiration date',
        isDisabled: true,
    })
    expirationDate: ui.fields.Date;

    @ui.decorators.dateField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        placeholder: 'Enter a use-by date',
        title: 'Use-by date',
        isDisabled: true,
    })
    useByDate: ui.fields.Date;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        placeholder: 'Scan...',
        title: 'Lot custom text 1',
        validation: /^$|^[^|]+$/,
        maxLength: 20,
        isDisabled: true,
    })
    lotCustomField1: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        placeholder: 'Scan...',
        title: 'Lot custom text 2',
        validation: /^$|^[^|]+$/,
        maxLength: 10,
        isDisabled: true,
    })
    lotCustomField2: ui.fields.Text;

    @ui.decorators.numericField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        placeholder: 'Scan...',
        title: 'Lot custom number',
        isDisabled: true,
    })
    lotCustomField3: ui.fields.Numeric;

    @ui.decorators.dateField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        placeholder: 'Enter a date',
        title: 'Lot custom date',
        isDisabled: true,
    })
    lotCustomField4: ui.fields.Date;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Identifier 1',
        placeholder: 'Scan...',
        isMandatory: false,
        validation: /^$|^[^|]+$/,
        maxLength: 10,
        isDisabled: true,
    })
    identifier1: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Identifier 2',
        placeholder: 'Scan...',
        validation: /^$|^[^|]+$/,
        isMandatory: false,
        maxLength: 10,
        isDisabled: true,
    })
    identifier2: ui.fields.Text;

    private async _focusToNextField(field: ui.fields.FilterSelect | ui.fields.Text | ui.fields.Reference | undefined) {
        switch (field?.id) {
            case this.quantityToReceive.id:
                if (!this.status.isHidden && !this.status.isDisabled && !this.status.isReadOnly) {
                    this.status.focus();
                }
                break;
            case this.status.id:
                if (!this.container.isHidden && !this.container.isDisabled && !this.container.isReadOnly) {
                    this.container.focus();
                }
                break;
            case this.container.id:
                if (
                    !this.licensePlateNumber.isHidden &&
                    !this.licensePlateNumber.isDisabled &&
                    !this.licensePlateNumber.isReadOnly
                ) {
                    this.licensePlateNumber.focus();
                }
                break;
            case this.licensePlateNumber.id:
                if (!this.location.isHidden && !this.location.isDisabled && !this.location.isReadOnly) {
                    this.location.focus();
                }
                break;
            case this.location.id:
                break;
            default:
                if (
                    !this.quantityToReceive.isHidden &&
                    !this.quantityToReceive.isDisabled &&
                    !this.quantityToReceive.isReadOnly
                ) {
                    this.quantityToReceive.focus();
                }
                break;
        }
    }

    private async _validate(): Promise<boolean> {
        const fieldErrors: string[] = [];
        if ((!this.receiptUnit.isHidden && !this.receiptUnit.value) || !!(await this.receiptUnit.validate()))
            fieldErrors.push(this.receiptUnit.title ?? '');
        if (!this.quantityToReceive.value || !!(await this.quantityToReceive.validate()))
            fieldErrors.push(this.quantityToReceive.title ?? '');
        const validationStatus = await this.status.validate();
        if (validationStatus) {
            fieldErrors.push(this.status.title ?? '');
        }
        const validationLicensePlateNumber = await this.licensePlateNumber.validate();
        if (validationLicensePlateNumber) {
            fieldErrors.push(this.licensePlateNumber.title ?? '');
        }
        const validationLocation = await this.location.validate();
        if (validationLocation) {
            fieldErrors.push(this.location.title ?? '');
        }
        const validationContainer = await this.container.validate();
        if (validationContainer) {
            fieldErrors.push(this.container.title ?? '');
        }
        if (fieldErrors.length > 0) {
            this.$.removeToasts();
            this.$.showToast(
                ui.localize(
                    '@sage/x3-purchasing-sales/pages__utils__notification__invalid_inputs_error',
                    `Please check your inputs for {{#each fieldNames}}\n - {{this}}{{/each}}`,
                    { fieldNames: fieldErrors },
                ),
                { type: 'error', timeout: 5000 },
            );
            await this.$.page.validate();
        }

        return fieldErrors.length === 0;
    }

    private _showSubmit() {
        this.submitButton.isDisabled =
            !this.receiptUnit.value ||
            Number(this.quantityToReceive.value) <= 0 ||
            (!this.container.isHidden &&
                !this.licensePlateNumber.isHidden &&
                (!this.licensePlateNumber.value || this.licensePlateNumber.value?.code === '') &&
                (!this.container.value || this.container.value?.code === ''));
    }

    private _initDirty(): void {
        if ((this.intersiteSession?.purchaseReceipt?.lines ?? []).length > 1) {
            this.product.isDirty = true;
        }
    }

    private _getPayloadFieldString(value: string | null): string | undefined {
        if (value && value !== '') {
            return value;
        } else {
            return undefined;
        }
    }

    private _getPayloadFieldDate(value: string | null): string | undefined {
        if (value && value !== '' && !value.startsWith('1599') && !value.startsWith('2999')) {
            return value;
        } else {
            return undefined;
        }
    }

    private _getPayloadFieldNumber(value: number | null): number | undefined {
        if (value && value !== 0) {
            return value;
        } else {
            return undefined;
        }
    }
}
