import {
    GraphApi,
    PurchaseEntryTransactionInput,
    PurchaseReceiptInput,
    PurchaseReceiptLineInput,
} from '@sage/x3-purchasing-api';
import { StockManagementRules } from '@sage/x3-stock-data-api';
import { extractEdges } from '@sage/xtrem-client';
import { Page, localize, queryUtils } from '@sage/xtrem-ui';

export async function validate(page: Page): Promise<boolean> {
    // (X3-201046, X3-203885) TODO Issue: Have better validation error messages that specifically tell which fields have issues
    const errors = await page.$.page.validate();
    if (errors.length === 0) {
        return true;
    }

    page.$.showToast(`${localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error')}: ${errors[0]}`, {
        type: 'error',
        timeout: 30000,
    });

    return false;
}
export async function controlLotReceipt(
    pageInstance: Page,
    lot: string,
    product: string,
    entryType: string,
    site: string,
): Promise<boolean> {
    if (!lot || !product || !entryType) throw new Error('Invalid arguments');
    const stockJournalFilter = {
        stockSite: site,
        documentType: entryType,
        product: { code: product },
        isUpdated: true,
        lot: lot,
    };

    const response = extractEdges(
        await pageInstance.$.graph
            .node('@sage/x3-stock-data/StockJournal')
            .query(
                queryUtils.edgesSelector(
                    {
                        lot: true,
                        sublot: true,
                    },
                    {
                        filter: stockJournalFilter,
                    },
                ),
            )
            .execute(),
    );

    if (response.length > 0) {
        pageInstance.$.showToast(
            localize(
                '@sage/x3-purchasing-sales/notification-error-receipt-lot',
                'the lot number {{ lot }} already exists for this product',
                { lot: lot },
            ),
            { type: 'error' },
        );
        return false;
    }
    return true;
}

enum YesNoEnum {
    no = 1,
    yes = 2,
}

export type IntersiteReceiptSession = {
    purchaseReceipt: PurchaseReceiptInput;
    purchaseEntryTransaction: PurchaseEntryTransactionInput;
    purchaseStockManagementRules?: StockManagementRules;
    salesUnitToStockUnitConversionFactor: number;
} | undefined;

export class IntersiteReceiptDetailsRecordManager {
    constructor(
        private page: Page<GraphApi>,
        cleanupIncompleteSessionLines = false,
    ) {
        if (cleanupIncompleteSessionLines) this._cleanupSession();
    }

    private static readonly INTERSITE_RECEIPT_KEY: string = 'IntersiteReceipt';

    private _intersiteSession: IntersiteReceiptSession;

    public get intersiteSession(): IntersiteReceiptSession {
        if (this._intersiteSession) {
            return this._intersiteSession;
        }

        const storedSession = this.page.$.storage.get(
            IntersiteReceiptDetailsRecordManager.INTERSITE_RECEIPT_KEY,
        ) as string;
        if (storedSession) {
            const parsedSession = JSON.parse(storedSession) as IntersiteReceiptSession;
            if (this._checkStorage(parsedSession)) {
                this._intersiteSession = parsedSession;
            } else {
                this.clearSession();
            }
        }
        return this._intersiteSession;
    }

    public set intersiteSession(intersiteSession: IntersiteReceiptSession) {
        if (!intersiteSession) {
            return;
        }

        if (!this._checkStorage(intersiteSession)) {
            throw new Error('Invalid Intersite Receipt session');
        }

        this.page.$.storage.set(
            IntersiteReceiptDetailsRecordManager.INTERSITE_RECEIPT_KEY,
            JSON.stringify(intersiteSession),
        );
        this._intersiteSession = intersiteSession;
    }

    private _checkStorage(sessionData: IntersiteReceiptSession): boolean {
        if (!sessionData || !sessionData.purchaseEntryTransaction.code || !sessionData.purchaseReceipt.receiptDate) {
            return false;
        }
        return true;
    }

    private _cleanupSession() {
        this._intersiteSession = this.intersiteSession; // get the latest from session
        if (!this._intersiteSession || !this._intersiteSession.purchaseReceipt.lines) return;

        const prevLineCount = this._intersiteSession.purchaseReceipt.lines.length;
        this._intersiteSession.purchaseReceipt.lines = this._intersiteSession.purchaseReceipt.lines.filter(
            (line: PurchaseReceiptLineInput) => {
                if (
                    !line.receiptUnit ||
                    line.quantityInReceiptUnitReceived === null ||
                    line.quantityInReceiptUnitReceived === undefined ||
                    !line.stockDetails
                ) {
                    return false;
                }

                return !line.stockDetails.some(detail => {
                    // check for each stock detail on this line, the packing unit & quantity are defined
                    return (
                        !detail.packingUnit ||
                        detail.quantityInPackingUnit === null ||
                        detail.quantityInPackingUnit === undefined
                    );
                });
            },
        );

        if (this._intersiteSession.purchaseReceipt.lines.length === 0) {
            this.clearSession();
        } else if (prevLineCount !== this._intersiteSession.purchaseReceipt.lines.length) {
            this.intersiteSession = this._intersiteSession;
        }
    }

    public clearSession(): void {
        this.page.$.storage.remove(IntersiteReceiptDetailsRecordManager.INTERSITE_RECEIPT_KEY);
        this._intersiteSession = undefined;
    }
}
