"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntersiteReceiptDetailsRecordManager = exports.controlLotReceipt = exports.validate = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const xtrem_client_1 = require("@sage/xtrem-client");
const xtrem_ui_1 = require("@sage/xtrem-ui");
async function validate(page) {
    // (X3-201046, X3-203885) TODO Issue: Have better validation error messages that specifically tell which fields have issues
    const errors = await page.$.page.validate();
    if (typesLib.strictEq(errors.length, 0)) {
        return true;
    }
    page.$.showToast(`${(0, xtrem_ui_1.localize)('@sage/x3-purchasing-sales/dialog-error-title', 'Error')}: ${errors[0]}`, {
        type: 'error',
        timeout: 30000,
    });
    return false;
}
exports.validate = validate;
async function controlLotReceipt(pageInstance, lot, product, entryType, site) {
    if (!lot || !product || !entryType)
        throw new Error('Invalid arguments');
    const stockJournalFilter = {
        stockSite: site,
        documentType: entryType,
        product: { code: product },
        isUpdated: true,
        lot: lot,
    };
    const response = (0, xtrem_client_1.extractEdges)(await pageInstance.$.graph
        .node('@sage/x3-stock-data/StockJournal')
        .query(xtrem_ui_1.queryUtils.edgesSelector({
        lot: true,
        sublot: true,
    }, {
        filter: stockJournalFilter,
    }))
        .execute());
    if (typesLib.gt(response.length, 0)) {
        pageInstance.$.showToast((0, xtrem_ui_1.localize)('@sage/x3-purchasing-sales/notification-error-receipt-lot', 'the lot number {{ lot }} already exists for this product', { lot: lot }), { type: 'error' });
        return false;
    }
    return true;
}
exports.controlLotReceipt = controlLotReceipt;
var YesNoEnum;
(function (YesNoEnum) {
    YesNoEnum[YesNoEnum["no"] = 1] = "no";
    YesNoEnum[YesNoEnum["yes"] = 2] = "yes";
})(YesNoEnum || (YesNoEnum = {}));
class IntersiteReceiptDetailsRecordManager {
    constructor(page, cleanupIncompleteSessionLines = false) {
        this.page = page;
        if (cleanupIncompleteSessionLines)
            this._cleanupSession();
    }
    static { this.INTERSITE_RECEIPT_KEY = 'IntersiteReceipt'; }
    get intersiteSession() {
        if (this._intersiteSession) {
            return this._intersiteSession;
        }
        const storedSession = this.page.$.storage.get(IntersiteReceiptDetailsRecordManager.INTERSITE_RECEIPT_KEY);
        if (storedSession) {
            const parsedSession = JSON.parse(storedSession);
            if (this._checkStorage(parsedSession)) {
                this._intersiteSession = parsedSession;
            }
            else {
                this.clearSession();
            }
        }
        return this._intersiteSession;
    }
    set intersiteSession(intersiteSession) {
        if (!intersiteSession) {
            return;
        }
        if (!this._checkStorage(intersiteSession)) {
            throw new Error('Invalid Intersite Receipt session');
        }
        this.page.$.storage.set(IntersiteReceiptDetailsRecordManager.INTERSITE_RECEIPT_KEY, JSON.stringify(intersiteSession));
        this._intersiteSession = intersiteSession;
    }
    _checkStorage(sessionData) {
        if (!sessionData || !sessionData.purchaseEntryTransaction.code || !sessionData.purchaseReceipt.receiptDate) {
            return false;
        }
        return true;
    }
    _cleanupSession() {
        this._intersiteSession = this.intersiteSession; // get the latest from session
        if (!this._intersiteSession || !this._intersiteSession.purchaseReceipt.lines)
            return;
        const prevLineCount = this._intersiteSession.purchaseReceipt.lines.length;
        this._intersiteSession.purchaseReceipt.lines = this._intersiteSession.purchaseReceipt.lines.filter((line) => {
            if (!line.receiptUnit || typesLib.strictEq(line.quantityInReceiptUnitReceived, null) || typesLib.strictEq(line.quantityInReceiptUnitReceived, undefined) ||
                !line.stockDetails) {
                return false;
            }
            return !line.stockDetails.some(detail => {
                // check for each stock detail on this line, the packing unit & quantity are defined
                return (!detail.packingUnit || typesLib.strictEq(detail.quantityInPackingUnit, null) || typesLib.strictEq(detail.quantityInPackingUnit, undefined));
            });
        });
        if (typesLib.strictEq(this._intersiteSession.purchaseReceipt.lines.length, 0)) {
            this.clearSession();
        }
        else if (typesLib.strictNe(prevLineCount, this._intersiteSession.purchaseReceipt.lines.length)) {
            this.intersiteSession = this._intersiteSession;
        }
    }
    clearSession() {
        this.page.$.storage.remove(IntersiteReceiptDetailsRecordManager.INTERSITE_RECEIPT_KEY);
        this._intersiteSession = undefined;
    }
}
exports.IntersiteReceiptDetailsRecordManager = IntersiteReceiptDetailsRecordManager;
//# sourceMappingURL=intersite-receipt-details-control.js.map