"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const SECRET_CONFIG_PREFIX = "@secret/";
class AWSSecretWrapper {
    constructor(config, secretsManagerMock) {
        if (!config) {
            throw new Error("There is no configuration defined to the wrapper constructor.");
        }
        if (!config.xtremEnv) {
            throw new Error("No environment defined in the wrapper constructor.");
        }
        if (!config.clusterId) {
            throw new Error("No cluster id defined to the wrapper constructor.");
        }
        this.xtremEnv = config.xtremEnv;
        this.clusterId = config.clusterId;
        this.secretCache = {};
        this.secretCacheTimeoutInMinutes = config.secretCacheTimeoutInMinutes || 5;
        if (secretsManagerMock) {
            this.SecretsManager = secretsManagerMock;
            console.log("SecretHandler is mocking secretsManager library");
        }
        else {
            this.SecretsManager = new client_secrets_manager_1.SecretsManager({});
        }
    }
    async getTenantEncryptionKey(tenantId) {
        const secretPath = this.getTenantSecretPath(this.clusterId, tenantId);
        return this.readSecret(secretPath);
    }
    /***
     * Read a configuration file value, if it's a secret it will return the secret value.
     * If it's not it will return the original config value.
     * we expect the secret configValue to have full path ex : @secret/xtreem-dev-eu/cluster-xxx/path
     * @param configValue
     */
    async readConfigValue(configValue) {
        if (this.isConfigValueASecret(configValue)) {
            const secretPath = configValue.substring(SECRET_CONFIG_PREFIX.length);
            return this.readSecret(secretPath);
        }
        return configValue;
    }
    isConfigValueASecret(configValue) {
        return configValue && configValue.startsWith(SECRET_CONFIG_PREFIX);
    }
    /***
     * Read a secret from cache if exists, add to cache after read if existed
     * @param secretPath
     */
    async readSecret(secretPath) {
        try {
            let secret = this.cacheRetrieveSecret(secretPath);
            if (!secret) {
                console.log("Secret read from secrets manager - " + secretPath);
                const params = {
                    SecretId: secretPath
                };
                secret = await this.SecretsManager.getSecretValue(params);
                this.cacheStoreSecret(secretPath, secret.SecretString);
            }
            else {
                console.log("Secret read from cache - " + secretPath);
            }
            return secret.SecretString;
        }
        catch (err) {
            throw new Error("Error while getting secret : " + err);
        }
    }
    // Cache for values stored in secret manager as API calls are relatively expensive
    cacheExpired(timeStamp) {
        return (Date.now() - timeStamp) > this.secretCacheTimeoutInMinutes * 60 * 1000;
    }
    cacheStoreSecret(secretPath, secretString) {
        if (!this.secretCache[secretPath]) {
            this.secretCache[secretPath] = {
                cacheLastCleaned: Date.now()
            };
        }
        this.secretCache[secretPath] = {
            SecretString: secretString,
            TimeStamp: Date.now()
        };
    }
    cacheRetrieveSecret(secretPath) {
        if (this.secretCache[secretPath]) {
            console.log(`Secret read from cache - ${secretPath}`);
            if (!this.cacheExpired(this.secretCache[secretPath].TimeStamp)) {
                // update the timestamp on the secret as it's been accessed
                this.secretCache[secretPath].TimeStamp = Date.now();
                return this.secretCache[secretPath];
            }
            else {
                console.log(`Cached secret has expired - ${secretPath}`);
            }
        }
        // return null if not in the cache or expired
        return null;
    }
    getTenantSecretPath(clusterId, tenantId) {
        return `xtreem-${this.xtremEnv}/${clusterId}/${tenantId}/encryptionKey`;
    }
}
exports.default = AWSSecretWrapper;
