import { Command } from '@ckeditor/ckeditor5-core';
/**
 * The table cell attribute command.
 *
 * This command is a base command for other table property commands.
 */
export default class QueryTablePropertyCommand extends Command {
    /**
     * Creates a new `TablePropertyCommand` instance.
     *
     * @param editor An editor in which this command will be used.
     * @param attributeName Table cell attribute name.
     * @param defaultValue The default value of the attribute.
     */
    constructor(editor, attributeName, defaultValue) {
        super(editor);
        this.attributeName = attributeName;
        this._defaultValue = defaultValue;
    }
    /**
     * @inheritDoc
     */
    refresh() {
        const model = this.editor.model;
        const selection = model.document.selection;
        const selectedElement = selection.getSelectedElement();
        this.isEnabled = selectedElement?.name === 'queryTable';
        this.value = selectedElement ? this._getValue(selectedElement) : undefined;
    }
    /**
     * Executes the command.
     *
     * @fires execute
     * @param options.value If set, the command will set the attribute on the selected table.
     * If not set, the command will remove the attribute from the selected table.
     * @param options.batch Pass the model batch instance to the command to aggregate changes,
     * for example, to allow a single undo step for multiple executions.
     */
    execute(options = {}) {
        const { value, batch } = options;
        const model = this.editor.model;
        const selection = model.document.selection;
        const queryTable = selection.getSelectedElement();
        if (!queryTable) {
            return;
        }
        const valueToSet = this._getValueToSet(value);
        model.enqueueChange(batch, writer => {
            if (valueToSet) {
                writer.setAttribute(this.attributeName, valueToSet, queryTable);
            }
            else {
                writer.removeAttribute(this.attributeName, queryTable);
            }
        });
    }
    /**
     * Returns the attribute value for a table.
     */
    _getValue(table) {
        if (!table) {
            return undefined;
        }
        const value = table.getAttribute(this.attributeName);
        return value;
    }
    /**
     * Returns the proper model value. It can be used to add a default unit to numeric values.
     */
    _getValueToSet(value) {
        return value;
    }
}
//# sourceMappingURL=query-table-property-command.js.map