import Button from 'carbon-react/esm/components/button';
import Dialog from 'carbon-react/esm/components/dialog';
import Form from 'carbon-react/esm/components/form';
import * as React from 'react';
import { NodeBrowserTree } from '../node-browser-tree/node-browser-tree';
import { TableContentWithGroups, getMainPresentation, enforceContiguousOrder } from '@sage/xtrem-ui-components';
import { FiltersComponent } from '../filter/filters-component';
import { GraphQLTypes, FieldKey, Sortings } from '@sage/xtrem-shared';
import { StepSequence, StepSequenceItem } from 'carbon-react/esm/components/step-sequence';
import Typography from 'carbon-react/esm/components/typography/typography.component';
import { SelectionCard } from '../selection-card/selection-card';
import Loader from 'carbon-react/esm/components/loader';
import { SortingComponent } from '../sorting/sorting-component';
import { getSelectedItemsFromProperties } from '../../plugins/utils';
import { chain, isEmpty, isEqual, isNil, omit } from 'lodash';
import { getAggregationsFromItems, getOrderByFromItems, getSelectedFieldsFromItems, } from './insert-list-property-dialog-utils';
const noListItems = (nodes) => nodes.filter(n => n.kind !== 'LIST');
const requestSteps = {
    recordContext: ['modeSelection', 'filterDefinition', 'orderDefinition'],
    list: ['modeSelection', 'propertySelection', 'contentDefinition', 'filterDefinition'],
};
export function InsertListPropertyDialog({ contextProvider, locale, localize, localizeEnumMember, lookupRequest, onClose, onDisplayNotification, }) {
    const stepTranslations = {
        contentDefinition: localize('@sage/xtrem-document-editor/property-dialog-step-define-content', 'Define content'),
        filterDefinition: localize('@sage/xtrem-document-editor/property-dialog-step-define-filters', 'Define filters'),
        modeSelection: localize('@sage/xtrem-document-editor/property-dialog-step-insert-mode', 'Insert mode'),
        orderDefinition: localize('@sage/xtrem-document-editor/property-dialog-step-define-sorting-order', 'Define sorting order'),
        propertySelection: localize('@sage/xtrem-document-editor/property-dialog-step-column-selection', 'Column selection'),
    };
    const [properties, setProperties] = React.useState(null);
    const [selectedItems, setSelectedItems] = React.useState({});
    const [filters, setFilters] = React.useState([]);
    const [orderBy, setOrderBy] = React.useState({});
    const [groupOrderError, setGroupOrderError] = React.useState(false);
    const onGroupOrderError = React.useCallback((value) => {
        setGroupOrderError(value);
    }, []);
    const [currentStep, setCurrentStep] = React.useState(0);
    const [steps, setSteps] = React.useState(['modeSelection']);
    const [selectedMode, setSelectedMode] = React.useState(null);
    const getTableContentFromSelectedItems = React.useCallback((currentTableContentValue) => {
        const basicTypes = Object.values(GraphQLTypes);
        const newTableContent = Object.keys(selectedItems).reduce((acc, curr) => {
            if (basicTypes.includes(selectedItems[curr].data.type) ||
                selectedItems[curr].data.kind === 'ENUM') {
                const property = selectedItems[curr];
                acc.push({
                    presentation: selectedItems[curr].data.kind === 'ENUM'
                        ? FieldKey.Text
                        : getMainPresentation({
                            localize,
                            type: property.data.type,
                        }),
                    title: property.title ?? property.label,
                    formatting: undefined,
                    divisor: undefined,
                    property,
                    path: property.id ?? property.path,
                    labelPath: property.id ?? property.path,
                    group: property.group ?? 0,
                    operation: property.operation,
                    sorting: property.sorting ?? Sortings.ascending,
                });
            }
            return acc;
        }, []);
        const existingRows = currentTableContentValue.filter(v => newTableContent.find(t => t.path === v.path));
        const maxGroup = existingRows.reduce((max, curr) => (curr?.group > max ? curr.group : max), 0);
        const newTableContentRows = newTableContent
            .filter(t => existingRows.find(s => s.path === t.path) === undefined)
            .map(e => ({
            ...e,
            group: maxGroup,
        }));
        const orderedData = [...existingRows, ...newTableContentRows].map((e, idx) => ({
            ...e,
            _id: String(idx + 1),
        }));
        return enforceContiguousOrder({ orderedData, key: 'group' }).map(row => omit(row, ['_id']));
    }, [localize, selectedItems]);
    const [tableContentValue, setTableContentValue] = React.useState(getTableContentFromSelectedItems([]));
    const isFormValid = React.useCallback(() => {
        const currentStepName = steps[currentStep];
        if (groupOrderError) {
            return false;
        }
        if (currentStepName === 'modeSelection' && selectedMode === null) {
            // The user has to select the mode (record context or table) to proceed
            return false;
        }
        if (currentStepName === 'propertySelection' && isEmpty(selectedItems)) {
            // At least one property must be selected in order to insert a table.
            return false;
        }
        if (currentStepName === 'contentDefinition' &&
            !chain(tableContentValue)
                .groupBy(s => s.group)
                .toPairs()
                .map(s => s[1][0])
                .value()
                .every(s => Boolean(s.sorting))) {
            // The first row of each group must be sortable
            return false;
        }
        return true;
    }, [steps, currentStep, groupOrderError, selectedMode, selectedItems, tableContentValue]);
    const onPrimaryButtonClicked = React.useCallback(() => {
        if (currentStep === steps.length - 1 && selectedMode) {
            lookupRequest?.onSelected({
                filters,
                orderBy: selectedMode === 'recordContext' ? orderBy : getOrderByFromItems(tableContentValue),
                selectedMode,
                aggregations: getAggregationsFromItems(tableContentValue),
                selectedFields: getSelectedFieldsFromItems(tableContentValue),
            });
            onClose();
        }
        else {
            setCurrentStep(currentStep + 1);
        }
    }, [currentStep, steps.length, selectedMode, lookupRequest, filters, orderBy, tableContentValue, onClose]);
    const onPreviousButtonClicked = React.useCallback(() => {
        setCurrentStep(currentStep - 1);
        setGroupOrderError(false);
    }, [currentStep]);
    const getStepStatus = React.useCallback((index) => {
        if (currentStep < index) {
            return 'incomplete';
        }
        if (currentStep === index) {
            return 'current';
        }
        return 'complete';
    }, [currentStep]);
    React.useEffect(() => {
        setTableContentValue(getTableContentFromSelectedItems);
    }, [getTableContentFromSelectedItems]);
    React.useEffect(() => {
        setSelectedItems({});
        setCurrentStep(0);
        setSelectedMode(null);
        setProperties(null);
        setFilters([]);
        setOrderBy({});
        if (lookupRequest?.contextObjectType) {
            contextProvider
                .getObjectDetails({
                objectType: lookupRequest?.contextObjectType,
                contextType: 'body',
                canParentFilter: true,
                canParentSort: true,
            })
                .then(setProperties);
        }
    }, [lookupRequest, contextProvider]);
    const selectAllNodeProperties = React.useCallback(() => {
        setSelectedItems(getSelectedItemsFromProperties(properties));
    }, [properties]);
    React.useEffect(() => {
        setSteps(selectedMode ? requestSteps[selectedMode] : ['modeSelection']);
        // prepopulate selected properties for data context so that filters can be added
        if (selectedMode === 'recordContext') {
            selectAllNodeProperties();
        }
    }, [selectAllNodeProperties, selectedMode]);
    const onContentChange = React.useCallback((newData) => {
        const newTableContent = newData.filter(newItem => !isNil(newItem.property));
        if (!isEqual(newTableContent, tableContentValue)) {
            setTableContentValue(newTableContent);
        }
        // make sure filters are still compatible with selected items
        setFilters(currentFilters => currentFilters?.filter(f => newData.find(d => d._id === f.id)));
    }, [tableContentValue]);
    const onTableModeSelected = React.useCallback(() => {
        setSelectedItems({});
        setSelectedMode('list');
    }, []);
    const onDataContextSelected = React.useCallback(() => {
        selectAllNodeProperties();
        setSelectedMode('recordContext');
    }, [selectAllNodeProperties]);
    const handleOnClose = React.useCallback(() => {
        setGroupOrderError(false);
        onClose();
    }, [onClose]);
    return (React.createElement(Dialog, { open: !!lookupRequest, onCancel: handleOnClose, title: localize('@sage/xtrem-document-editor/insert-object-type-dialog-title', 'Insert an object type'), className: "document-editor-property-selection-dialog", size: "extra-large" },
        React.createElement(Form, { onSubmit: (ev) => {
                ev.preventDefault();
            }, stickyFooter: true, height: "500px", leftSideButtons: currentStep !== 0 ? React.createElement(Button, { onClick: onPreviousButtonClicked }, "Previous") : undefined, saveButton: React.createElement(Button, { buttonType: "primary", type: "submit", onClick: onPrimaryButtonClicked, disabled: !isFormValid() }, steps.length > 1 && currentStep === steps.length - 1
                ? localize('@sage/xtrem-document-editor/dialog-button-confirm', 'Confirm')
                : localize('@sage/xtrem-document-editor/dialog-button-next', 'Next')) },
            !properties && React.createElement(Loader, null),
            properties && (React.createElement(React.Fragment, null,
                React.createElement(StepSequence, { pb: "20px" }, steps.map((s, index) => {
                    const currentStepNumber = String(index + 1);
                    return (React.createElement(StepSequenceItem, { key: s, "aria-label": localize('@sage/xtrem-document-editor/step-status-count', 'Step {{currentStepNumber}} of {{totalStepCount}}', {
                            currentStepNumber,
                            totalStepCount: steps.length,
                        }), hiddenCompleteLabel: localize('@sage/xtrem-document-editor/step-status-complete', 'Complete'), hiddenCurrentLabel: localize('@sage/xtrem-document-editor/step-status-current', 'Current'), indicator: String(index + 1), status: getStepStatus(index) }, stepTranslations[s]));
                })),
                React.createElement(Typography, { variant: "h2", mb: "16px" },
                    React.createElement("span", null, currentStep + 1),
                    React.createElement("span", null, ". "),
                    React.createElement("span", null, stepTranslations[steps[currentStep]])),
                steps[currentStep] === 'modeSelection' && (React.createElement("div", { className: "document-editor-mode-selection-container" },
                    React.createElement(SelectionCard, { title: localize('@sage/xtrem-document-editor/list-insertion-mode-table', 'Table'), description: localize('@sage/xtrem-document-editor/list-insertion-mode-table-description', 'Inserts a table with configurable columns'), _id: "list", icon: "csv", isSelected: selectedMode === 'list', onClick: onTableModeSelected }),
                    React.createElement(SelectionCard, { title: localize('@sage/xtrem-document-editor/list-insertion-mode-data-container', 'Data container'), description: localize('@sage/xtrem-document-editor/list-insertion-mode-data-context-description', 'Inserts a single record for detailed documents'), _id: "recordContext", icon: "document_right_align", isSelected: selectedMode === 'recordContext', onClick: onDataContextSelected }))),
                steps[currentStep] === 'propertySelection' && lookupRequest && (React.createElement(NodeBrowserTree, { getObjectDetails: contextProvider.getObjectDetails, onCheckedItemsUpdated: setSelectedItems, checkedItems: selectedItems, filter: noListItems, objectType: lookupRequest?.contextObjectType, selectionMode: "multiple", localize: localize, root: lookupRequest.root })),
                steps[currentStep] === 'contentDefinition' && (React.createElement(TableContentWithGroups, { isAddButtonHidden: true, localize: localize, onChange: onContentChange, onDisplayNotification: onDisplayNotification, selectedItems: selectedItems, value: tableContentValue, isPropertySelectionDisabled: true, setGroupOrderError: onGroupOrderError })),
                steps[currentStep] === 'filterDefinition' && (React.createElement(FiltersComponent, { contextProvider: contextProvider, filters: filters, locale: locale, localize: localize, localizeEnumMember: localizeEnumMember, objectType: lookupRequest?.contextObjectType, onChange: setFilters, selectedItems: selectedItems })),
                steps[currentStep] === 'orderDefinition' && (React.createElement(SortingComponent, { localize: localize, value: orderBy, properties: properties, onChange: setOrderBy })))))));
}
//# sourceMappingURL=insert-list-property-dialog.js.map