import * as ui from '@sage/xtrem-ui';
import {
    AsyncCompositeAllowed,
    BarcodeManagerSupportedFields,
    compositeCodePrefix,
    DictionaryDataComposite,
    DictionaryFieldSupported,
} from './screen-management-gs-1';

import { ManagementServiceGs1 } from './management-service-gs-1';

const defaultManagementServiceKey = 'DEFAULT_MOBILE_COMPOSITE_GS_1_DATA_KEY';

export class SupportServiceManagementGs1Page<TGraphqlApi = any, TNodeType extends ui.ClientNode = any> extends ui.Page<
    TGraphqlApi,
    TNodeType
> {
    /**
     * Internal only
     */

    constructor() {
        super();
        this._controlManagerGs1 = new ManagementServiceGs1(defaultManagementServiceKey);
    }

    /**
     * Manager services for General Specifications One (GS 1).
     */

    /**
     * Current instance of management service GS 1
     * A temporary instance is created here to prevent any preliminary function calls
     * before the real initialization.
     *
     */
    /** @internal */
    private _controlManagerGs1: ManagementServiceGs1;

    /**
     * Used only for disable scan operation (service must stay active)
     */
    /** @internal */
    private _disableServiceGs1 = false;

    /**
     * Initialize ControlManagerGs1
     *
     * @param site current site
     * @param compositePageKey key for storage operation
     * @param dictionaryFieldSupported fields screen to manage and control
     * @param dictionaryDataComposites? optional composite data
     * @param checkCompositeDataAllowed? optional client callback to check composite data code before to dispatch them
     * @returns true when ControlManagerGs1 has usable
     */
    /** @internal */
    private async _createAndInitControlManagerGs1(
        site: string,
        compositePageKey: string,
        dictionaryFieldSupported: DictionaryFieldSupported,
        dictionaryDataComposites?: DictionaryDataComposite,
        checkCompositeDataAllowed?: AsyncCompositeAllowed,
    ): Promise<boolean> {
        this._controlManagerGs1 = new ManagementServiceGs1(compositePageKey);
        // Including erroneous field misspelled dateTestFake instead dateTest
        if (site && compositePageKey) {
            if (
                (await this._controlManagerGs1
                    .initialize(
                        this as ui.Page,
                        dictionaryFieldSupported,
                        dictionaryDataComposites,
                        checkCompositeDataAllowed,
                    )
                    .catch(() => false)) as boolean
            ) {
                return true;
            } else {
                // Error detected, abort controller
                ui.console.error(`Service initialization failure for site ${site}`);
            }
        }
        // Initialization failure : reset to default instance
        this._controlManagerGs1 = new ManagementServiceGs1(defaultManagementServiceKey);
        return false;
    }

    /**
     * Protected only
     */
    /**
     * Return current state of service
     */
    /** @protected */
    protected get isServiceGs1Disabled(): boolean {
        return this._disableServiceGs1;
    }

    /**
     * Disable or enable service (default : enable) :
     * Disabling clearing all composite data
     */
    /** @protected */
    protected set disableServiceGs1(disableServiceGs1: boolean) {
        this._disableServiceGs1 = disableServiceGs1;
        if (this._disableServiceGs1) {
            this._controlManagerGs1.clearCompositeData();
        }
    }

    /**
     * Get current control manager instance
     */
    /** @protected */
    protected get controlManagerGs1() {
        return this._controlManagerGs1;
    }

    /**
     * Remove composite data
     */
    /** @protected */
    protected clearCompositeData(): void {
        this._controlManagerGs1.clearCompositeData();
    }

    /**
     * Remove composite data from storage
     * @param storageKey optional, for overriding current service key (use with caution)
     */
    /** @protected */
    protected clearCompositeDataStorage(storageKey?: string): void {
        this._controlManagerGs1.clearCompositeDataStorage(this as ui.Page, storageKey);
    }

    /**
     * Remove all composite and storage data
     * @param storageKey optional, for overriding current service key (use with caution)
     */
    /** @protected */
    protected clearAllCompositeDataAndStorage(storageKey?: string): void {
        this._controlManagerGs1.clearAllCompositeDataAndStorage(this as ui.Page, storageKey);
    }

    /**
     * Create and initialize manager for a given service :
     * manager stay available only when site has defined and initialization is done.
     *
     * @param site current site
     * @param compositePageKey key for storage operation
     * @param dictionaryFieldSupported fields screen to manage and control
     * @param dictionaryDataComposites? optional composite data
     * @param checkCompositeDataAllowed? optional client callback to check composite data code before to dispatch them
     * @return false when service has not properly initialized (unavailable)
     */
    /** @protected */
    protected async createAndInitServiceGs1(
        site: string,
        compositePageKey: string,
        dictionaryFieldSupported: DictionaryFieldSupported,
        dictionaryDataComposites?: DictionaryDataComposite,
        checkCompositeDataAllowed?: AsyncCompositeAllowed,
    ): Promise<boolean> {
        if (
            !site ||
            !(await this._createAndInitControlManagerGs1(
                site,
                compositePageKey,
                dictionaryFieldSupported,
                dictionaryDataComposites,
                checkCompositeDataAllowed,
            ))
        ) {
            ui.console.error(`Unable to initialize service control manager Gs 1 for site ${site}`);
            return false;
        }
        return true;
    }

    /**
     * Load composite data from storage and erase them
     * @returns true when some data has been loaded
     */
    /** @protected */
    protected loadCompositeData(): boolean {
        return this._controlManagerGs1.loadCompositeData(this as ui.Page);
    }

    /**
     * Save composite data, zero elements allowed
     * @returns true when operation has performed
     */
    /** @protected */
    protected saveCompositeData(): boolean {
        return this._controlManagerGs1.saveCompositeData(this as ui.Page);
    }

    /**
     * call scan evaluator only when manager available and rawData has minimum requirement
     * @param mainField current screen field reference
     * @param rawData data to parse
     * @return false when is not a bar code composite
     */
    /** @protected */
    protected async scanBarCode(mainField: BarcodeManagerSupportedFields, rawData: string): Promise<boolean> {
        // class must be created, initialized and not busy, data must be started by a numeric value,
        // minimum size 2 to 4 numerics or prefix ]<alphabetic><numeric>
        if (
            !(this._disableServiceGs1 ?? false) &&
            rawData.length > 4 &&
            Number(rawData.match(compositeCodePrefix)?.length)
        ) {
            return (await this._controlManagerGs1
                .scan(this as ui.Page, mainField, rawData)
                .catch(() => false)) as boolean;
        }
        return false;
    }

    /**
     * Update screen fields supported (only when exists Gs1 parameters).
     * mapping field has been performed after
     * @param dictionaryFieldSupported dictionary of page fields submitted
     * @returns true when done, false when no any field to process with data
     */
    protected async setScreenFieldSupported(dictionaryFieldSupported: DictionaryFieldSupported): Promise<boolean> {
        return await this._controlManagerGs1.setScreenFieldSupported(this as ui.Page, dictionaryFieldSupported);
    }
}
