"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ManagementServiceGs1 = exports.errorMessageError = exports.errorMessageCheckYourEntry = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const xtrem_date_time_1 = require("@sage/xtrem-date-time");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const ui = require("@sage/xtrem-ui");
const barcode_parser_1 = require("../shared-functions/barcode-parser");
const screen_management_gs_1_1 = require("./screen-management-gs-1");
exports.errorMessageCheckYourEntry = `Check your entry for {{#each fieldNames}}\n - {{this}}{{/each}}`;
exports.errorMessageError = 'Error';
class ManagementServiceGs1 {
    /**
     * Initialize permanent data for page
     * @param _storageKey       : storage key used for this page
     */
    constructor(_storageKey) {
        this._storageKey = _storageKey;
        // Private data area
        /**
         *  Store all codes GS 1 supported by access code (for speedup decoding)
         */
        /** @internal */
        this._screenFieldsSupported = [];
        /** @internal */
        this._dictionarySegment = {};
        /** @internal */
        this._screenFieldsMapped = [];
        /** @internal */
        this._dictionaryDataComposites = {};
        /** @internal */
        this._abortDispatch = false;
        /** @internal */
        this._dispatchInProgress = false;
        /** @internal */
        this._initializationInProgress = false;
        this._parseBarCode = new barcode_parser_1.ParseBarCode();
    }
    /**
     * This Method build instance with retrieving all data
     * @param pageInstance : current page
     * @param dictionaryFieldSupported : page fields supported
     * @param dictionaryDataComposites : extracted composite data, may be send to storage
     * @param checkCompositeDataAllowed? : optional client callback to check composite data code before to dispatch them
     */
    async initialize(pageInstance, dictionaryFieldSupported = {}, dictionaryDataComposites = {}, checkCompositeDataAllowed) {
        if (!pageInstance || !this._storageKey) {
            throw new xtrem_shared_1.SystemError('Invalid arguments');
        }
        try {
            this._dispatchInProgress = false;
            this._initializationInProgress = true;
            this._checkCompositeDataAllowed = checkCompositeDataAllowed;
            // ui.console.warn(`Field(s) supported :\n${JSON.stringify(dictionaryFieldSupported)}`);
            // ui.console.warn(`Composite data :\n${JSON.stringify(dictionaryDataComposites)}`);
            // Process composite data before to continue
            if (typesLib.gt(Object.keys(dictionaryDataComposites ?? {}).length, 0)) {
                this.clearCompositeData();
                // Erase storage
                this._removeCompositeDataStorage(pageInstance);
                this._dictionaryDataComposites = dictionaryDataComposites ?? {};
            }
            else {
                this.loadCompositeData(pageInstance);
            }
            if (typesLib.gt(Object.keys(dictionaryFieldSupported ?? {}).length, 0)) {
                await this.setScreenFieldSupported(pageInstance, dictionaryFieldSupported ?? {});
                // mapped data has not deleted after dispatching
                if (this.isExistsDataComposite && this.isExistsFieldsMapped) {
                    if (this._beginDispatching()) {
                        try {
                            // use existing first segment field, temporary ignoring result
                            const results = (await this._dispatchDataToFields(pageInstance, this._screenFieldsMapped[0].mainField, false).catch(() => false));
                        }
                        catch (error) {
                            ui.console.error(`Error dispatching initial data :\n${error}`);
                        }
                        finally {
                            this._endingDispatching();
                        }
                    }
                }
            }
        }
        catch (error) {
            ui.console.error(`Unexpected initialization error :\n${error}`);
        }
        finally {
            this._initializationInProgress = false;
        }
        return true;
    }
    /**
     * Return if dispatch session is in progress
     */
    get isDispatchInProgress() {
        return this._dispatchInProgress;
    }
    /**
     * Return if initialization is in progress
     */
    get isInitializationInProgress() {
        return this._initializationInProgress;
    }
    /**
     * This function start dispatching
     * @returns true when a new session has started, false when already in progress
     */
    /** @internal */
    _beginDispatching() {
        const oldDispatchInProgress = this._dispatchInProgress;
        this._dispatchInProgress = true;
        this._abortDispatch = false;
        this._postActionAbortDispatch = undefined;
        return !oldDispatchInProgress;
    }
    /**
     * Terminate current dispatch session
     * @returns true when session has been terminated, false when non session exist
     */
    /** @internal */
    _endingDispatching() {
        const oldDispatchInProgress = this._dispatchInProgress;
        this._dispatchInProgress = false;
        this._abortDispatch = false;
        this._postActionAbortDispatch = undefined;
        return oldDispatchInProgress;
    }
    /**
     * Request to abort dispatch without error (end of scan) :
     * This is used only when onChange() has performed some operations,
     * ie : switching page... except when initialization is in progress
     * @param {function}    optional callback to be executed approx. 100ms after end of dispatching.
     * @returns {boolean}   true when request has be set
     */
    abortDispatch(postActionAbortDispatch) {
        this._abortDispatch = false;
        this._postActionAbortDispatch = undefined;
        if (this._dispatchInProgress) {
            this._abortDispatch = true;
            if (!this.isInitializationInProgress) {
                this._postActionAbortDispatch = postActionAbortDispatch;
            }
        }
        return this.isAbortDispatchInProgress;
    }
    /**
     * Return if abort dispatch session is requested
     */
    get isAbortDispatchInProgress() {
        return this._abortDispatch;
    }
    /**
     * Called when field update is in progress
     * @param pageInstance current page
     * @param currentField reference of field to updating
     * @param rawData received raw data
     * @param appendCompositeData true when is necessary to appending current date to previous set
     * @return true when is not necessary to continue and affect a value : composite value has been processed
     */
    async scan(pageInstance, currentField, rawData, appendCompositeData = false) {
        if (this._beginDispatching()) {
            try {
                let result = true;
                // when is not a composite data, is not necessary to process
                if (!rawData || typesLib.strictNe(typeof rawData, 'string') || typesLib.lt(rawData.length, 4) ||
                    !Number(rawData.match(screen_management_gs_1_1.compositeCodePrefix)?.length) ||
                    !(await this._decodeCompositeData(rawData, appendCompositeData))) {
                    this._endingDispatching();
                    return false;
                }
                // When composite data have skipped, nothing to do, report no errors
                if (this.isExistsDataComposite) {
                    let _postActionToDo;
                    const _soundOriginalValue = await this._setSoundField(pageInstance, currentField, true);
                    try {
                        result = (await this._dispatchDataToFields(pageInstance, currentField).catch(() => false));
                        // Special action to do after exiting a function
                        if (result && this.isAbortDispatchInProgress && this._postActionAbortDispatch) {
                            _postActionToDo = this._postActionAbortDispatch;
                        }
                    }
                    finally {
                        setTimeout(async () => {
                            try {
                                await this._setSoundField(pageInstance, currentField, _soundOriginalValue);
                                if (_postActionToDo)
                                    await _postActionToDo();
                            }
                            catch (error) {
                                ui.console.error(`Dispatching post done action error :\n${error}`);
                            }
                        }, 100);
                    }
                }
                else {
                    await this._clearField(pageInstance, currentField);
                }
                return result;
            }
            catch (error) {
                ui.console.error(`Dispatching error :${error}`);
            }
            finally {
                this._endingDispatching();
            }
        }
        ui.console.error(`Dispatching not available`);
        return false;
    }
    /**
     * returns if the field support sound management
     * @param originalField
     * @returns
     */
    /** @internal */
    _getIsSupportedSoundField(currentField) {
        return currentField instanceof ui.fields.Reference || currentField instanceof ui.fields.FilterSelect;
    }
    /**
     * When possible, changes the sound mode activation state and returns the old value
     * @param currentField current field or undefined
     * @param disableSound true for disable sound for this field
     * @returns old value
     */
    /** @internal */
    async _setSoundField(pageInstance, currentField, disableSound, refreshField = true) {
        const originalIsSoundDisabled = currentField?.isSoundDisabled ?? false;
        if (currentField && typesLib.strictNe(disableSound, originalIsSoundDisabled) && this._getIsSupportedSoundField(currentField)) {
            currentField.isSoundDisabled = disableSound;
            if (refreshField) {
                await pageInstance.$.commitValueAndPropertyChanges();
            }
        }
        return originalIsSoundDisabled;
    }
    /**
     * Clear current UI field with null value and update page.
     * @param pageInstance current page
     * @param currentField current field to clear
     * @param setFocus optional value must be true when focus must be performed in reInitializing
     */
    /** @internal */
    async _clearField(pageInstance, currentField, setFocus = true) {
        currentField.value = null;
        // Request all changes to page (optional field and unit) before any action
        await pageInstance.$.commitValueAndPropertyChanges();
        if (setFocus) {
            currentField.focus();
        }
        // Force validation
        await currentField.validate().catch(() => false);
    }
    /**
     * Decode raw data to one or more data fields
     * @param rawData raw composite data before decoding
     * @param appendCompositeData may apply only when valid received data must be replace or append
     * @returns true when data has valid and operation fully performed
     */
    /** @internal */
    async _decodeCompositeData(rawData, appendCompositeData = false) {
        const extractedComposite = {};
        ui.console.debug(`initial data raw : ${JSON.stringify(rawData)}, size : ${rawData.length}`);
        let parsedBarcode;
        try {
            parsedBarcode = this._parseBarCode.parseBarcode(rawData);
            ui.console.debug(`RawData :\n${rawData}\nParsed code(s) :\n${JSON.stringify(parsedBarcode)}`);
        }
        catch (error) {
            ui.console.debug(`ERROR in parseBarcode :\n${error}`);
            return false;
        }
        /**
         * When the block does not include a prefix or when it is not recognized by the parser,
         * the GS1 composite code block requires in this case more than one AI code to be considered,
         * as a single value could be incorrectly considered composite.
         */
        if (!parsedBarcode.parsedCodeItems.length ||
            (typesLib.lt(parsedBarcode.parsedCodeItems.length, 2) && typesLib.strictEq(parsedBarcode.codeName, ''))) {
            return false;
        }
        // Store data into composite data storage
        for (const parsedCodeItem of parsedBarcode.parsedCodeItems) {
            // Add a new data entry
            extractedComposite[parsedCodeItem.elementDataTitle] = {
                DataTitle: parsedCodeItem.elementDataTitle,
                data: parsedCodeItem.data,
                unit: parsedCodeItem.unit,
            };
        }
        // ui.console.warn(`composite data existing : ${JSON.stringify(this._dictionaryDataComposites)}`);
        // ui.console.warn(`composite data loaded : ${JSON.stringify(extractedComposite)}`);
        // Dictionary update
        if (!appendCompositeData) {
            this.clearCompositeData();
            this._dictionaryDataComposites = extractedComposite;
        }
        else {
            for (const [key, value] of Object.entries(extractedComposite)) {
                this._dictionaryDataComposites[key] = value;
            }
        }
        // ui.console.debug(`composite data updated : ${JSON.stringify(this._dictionaryDataComposites)}`);
        // before to map, check if allowed
        if (!(await this._isDataCompositeAllowed())) {
            this.clearCompositeData();
            return true;
        }
        // mapping composite data to dataField without checking error condition
        this._mappingFields();
        // Sometimes no data is stored due to inactive code, this is not an error
        return true;
    }
    /**
     * Check if exists some fields loaded into service
     */
    get isExistsFieldsSupported() {
        return typesLib.gt(Object.keys(this._screenFieldsSupported).length, 0);
    }
    /**
     * Check if exists some fields mapped into service,
     * ready to dispatch
     */
    get isExistsFieldsMapped() {
        return typesLib.gt(this._screenFieldsMapped.length, 0);
    }
    /**
     *
     * @returns true when some composite data have been stored.
     */
    get isExistsDataComposite() {
        return typesLib.gt(Object.keys(this._dictionaryDataComposites).length, 0);
    }
    /**
     *
     * @param field check current field is available
     * @returns true if not available
     */
    /** @internal */
    _isFieldNotAvailable(field) {
        return (field?.isHidden || field?.isDisabled || field?.isReadOnly) ?? true;
    }
    /**
     * @returns cloned dictionary data composite (must be empty)
     */
    /** @internal */
    _cloneDictionaryDataComposite() {
        const dictionaryDataComposite = {};
        for (const [key, value] of Object.entries(this._dictionaryDataComposites)) {
            dictionaryDataComposite[key] = value;
        }
        return dictionaryDataComposite;
    }
    /**
     * Process client callback when exists to check data
     * @returns Checks if the composite dataset can be processed
     */
    /** @internal */
    async _isDataCompositeAllowed() {
        // If any error occurs, the composite data block will be processed
        try {
            if (this._checkCompositeDataAllowed &&
                this.isExistsDataComposite &&
                !(await this._checkCompositeDataAllowed(this._cloneDictionaryDataComposite()))) {
                return false;
            }
        }
        catch (error) {
            ui.console.error(`Error when evaluating data composite condition is allowed :\n${error}`);
        }
        return true;
    }
    /**
     * Update screen fields supported.
     * field mapping was done after
     * @param pageInstance current page
     * @param dictionaryFieldSupported dictionary of page fields submitted
     * @returns true when done, false when no any field to process with data
     */
    async setScreenFieldSupported(pageInstance, dictionaryFieldSupported) {
        // Get only fields matching with active parameters
        let index = 0;
        const lastInitializationInProgress = this.isInitializationInProgress;
        try {
            this._initializationInProgress = true;
            this._screenFieldsSupported =
                this._existingFields(pageInstance, dictionaryFieldSupported).map(field => {
                    const segmentsSupported = [];
                    // Create segments allowed
                    field.segments.forEach(segment => {
                        const currentField = segment.currentField;
                        segmentsSupported.push({
                            currentField: currentField,
                            onChange: segment.onChange,
                        });
                    });
                    return {
                        DataTitle: field.DataTitle,
                        sequence: (index = typesLib.add(index, 1)),
                        segments: segmentsSupported,
                    };
                }) ?? [];
            // ui.console.debug(`Fields supported into page : ${JSON.stringify(this._screenFieldsSupported)}`);
            // Map fields with existing composite data
            return this._mappingFields();
        }
        catch (error) {
            ui.console.error(`Error in setScreenFieldSupported :\n${error}`);
            return false;
        }
        finally {
            this._initializationInProgress = lastInitializationInProgress;
        }
    }
    /**
     * This function map current fields with set of data
     * @returns true when exists some fields with data mapped
     */
    /** @internal */
    _mappingFields() {
        // ui.console.debug(`Starting mapping, composite data : ${JSON.stringify(this._dictionaryDataComposites)}`);
        // ui.console.debug(`Screen field(s) supported : ${JSON.stringify(this._screenFieldsSupported)}`);
        // Extract active field
        this._screenFieldsMapped = [];
        let index = 0;
        const mappingFields = [];
        // No process only hidden fields
        this._screenFieldsSupported
            .filter(field => this._dictionaryDataComposites[field.DataTitle] &&
            !field.segments.some(segment => segment.currentField.isHidden))
            .sort((field1, field2) => typesLib.sub(field1.sequence, field2.sequence))
            .forEach(item => {
            const dataFieldMapped = {
                DataTitle: item.DataTitle,
                sequence: (index = typesLib.add(index, 1)),
                mainField: item.segments[0].currentField,
                onChangeMainField: item.segments[0].onChange,
                data: this._dictionaryDataComposites[item.DataTitle]?.data,
            };
            // Add optional part
            if (typesLib.gt(item.segments.length, 1) && item.segments[1]?.currentField) {
                dataFieldMapped.unitField = item.segments[1].currentField;
                dataFieldMapped.onChangeUnitField = item.segments[1].onChange;
                dataFieldMapped.unit = this._dictionaryDataComposites[item.DataTitle]?.unit;
            }
            mappingFields.push(dataFieldMapped);
        });
        this._screenFieldsMapped = mappingFields;
        // ui.console.debug(`Field(s) Mapped : ${JSON.stringify(this._screenFieldsMapped)}`);
        return typesLib.gt(this._screenFieldsMapped.length, 0);
    }
    /**
     * Remove fields definition
     * @param removeCompositeData  : optionally remove composite data too
     */
    clearScreenFields(removeCompositeData = false) {
        this._screenFieldsSupported = [];
        this._screenFieldsMapped = [];
        if (removeCompositeData) {
            this.clearCompositeData();
        }
    }
    /**
     * Remove composite data
     */
    clearCompositeData() {
        this._dictionaryDataComposites = {};
    }
    /**
     * Remove composite data from storage
     * @param pageInstance current page
     * @param storageKey optional, for overriding current service key (use with caution)
     */
    clearCompositeDataStorage(pageInstance, storageKey) {
        this._removeCompositeDataStorage(pageInstance, storageKey);
    }
    /**
     * Remove all composite and storage data
     * @param pageInstance current page
     * @param storageKey optional, for overriding current service key (use with caution)
     */
    clearAllCompositeDataAndStorage(pageInstance, storageKey) {
        this.clearCompositeData();
        this.clearCompositeDataStorage(pageInstance, storageKey);
    }
    /**
     * Load composite data from storage and then delete it there.
     * @returns true when some data has been loaded
     */
    loadCompositeData(pageInstance) {
        this.clearCompositeData();
        try {
            const dictionaryDataComposites = JSON.parse((pageInstance.$.storage.get(this._storageKey) ?? '{}'));
            // ui.console.warn(`Composite data loaded :\n${JSON.stringify(dictionaryDataComposites)}`);
            // Erase storage
            this._removeCompositeDataStorage(pageInstance);
            if (typesLib.gt(Object.keys(dictionaryDataComposites).length, 0)) {
                this._dictionaryDataComposites = dictionaryDataComposites;
                // ui.console.warn(`Assigned :\n${JSON.stringify(this._dictionaryDataComposites)}`);
                return true;
            }
        }
        catch (error) {
            this.clearCompositeData();
            this._removeCompositeDataStorage(pageInstance);
            ui.console.error(`Error loading composite data from storage :\n${error}`);
        }
        // ui.console.error(`No composite data`);
        return this.isExistsDataComposite;
    }
    /**
     * Save composite data, zero elements allowed
     * @param pageInstance current page
     * @returns true when operation has performed
     */
    saveCompositeData(pageInstance) {
        if (pageInstance && this._storageKey && this.isExistsDataComposite) {
            const dictionaryDataComposites = this._dictionaryDataComposites;
            pageInstance.$.storage.set(this._storageKey, JSON.stringify(dictionaryDataComposites));
            return true;
        }
        if (pageInstance) {
            this._removeCompositeDataStorage(pageInstance);
        }
        return false;
    }
    /**
     * Remove data from the storage
     * @param pageInstance current page
     */
    /** @internal */
    _removeCompositeDataStorage(pageInstance, storageKey) {
        const isDataStorage = !!pageInstance && (!!this._storageKey || !!storageKey);
        if (isDataStorage) {
            pageInstance.$.storage.remove(storageKey ?? this._storageKey);
        }
        return isDataStorage;
    }
    /**
     * assign date to page fields when existing and available
     * @param pageInstance current page
     * @param originalField original field (reserved)
     * @param setFocus optional value must be true when focus must be performed in reInitializing
     * @returns false when dispatching failure occurs
     */
    /** @internal */
    async _dispatchDataToFields(pageInstance, originalField, setFocus = false) {
        const $items = pageInstance?._pageMetadata.layout.$items ?? [];
        let isFailure = false;
        let isOneFieldProcessed = false;
        // ui.console.debug('Starting dispatching all fields');
        if (typesLib.gt($items?.length, 0) && this.isExistsDataComposite) {
            for (const field of this._screenFieldsMapped) {
                if (field?.data) {
                    if ((isFailure = !(await this._processField(pageInstance, field, setFocus).catch(() => false)))) {
                        // ui.console.error(`Dispatching error, composite data has cleared`);
                        this.clearCompositeData();
                        break;
                    }
                    else {
                        isOneFieldProcessed = true;
                    }
                }
            }
        }
        // If no composite data could be assigned, it is an error
        if (!isFailure && !isOneFieldProcessed) {
            isFailure = true;
            this.clearCompositeData();
            this._clearField(pageInstance, originalField, setFocus);
            ui.console.error(`Composite data was removed because none could be assigned, and field ${originalField?.id} was reset`);
        }
        // ui.console.debug(`Ending dispatching all fields : ${!isFailure}`);
        return !isFailure;
    }
    /**
     *
     * @param pageInstance current page
     * @param field current field already evaluated
     * @param setFocus optional value must be true when focus must be performed in reInitializing
     * @returns true when item has found, but updated only when available
     */
    /** @internal */
    async _processField(pageInstance, field, setFocus = false) {
        const fieldsToRevalidate = [];
        const onChangeCallbacks = [];
        const mainField = field?.mainField;
        const mainFieldId = mainField?.id ?? '';
        const unitField = field?.unitField;
        const unitFieldId = unitField?.id ?? '';
        let validation = true;
        let mainSoundOriginalValue = false;
        let unitSoundOriginalValue = undefined;
        try {
            // process all subFields (segments) requested (main and optional unit)
            // ui.console.debug(`_ProcessField (mapped fields) :\n${JSON.stringify(field)}`);
            // current field not currently available : don't process this field
            if (!mainField || this._isFieldNotAvailable(mainField)) {
                // ui.console.error(`Field ${mainFieldId} has unavailable`);
                return true;
            }
            if (field?.data) {
                if (unitField && field?.unit && this._isFieldNotAvailable(unitField)) {
                    // ui.console.error(`Unit field ${unitFieldId} has unavailable`);
                    return true;
                }
                mainSoundOriginalValue = await this._setSoundField(pageInstance, mainField, true);
                // If not possible to change a value, composite block is invalid
                switch (await this._setFieldValue(pageInstance, mainField, field?.data, setFocus)) {
                    case null:
                        // ui.console.warn(`Field ${mainFieldId} has no data available`);
                        return false;
                    case true:
                        fieldsToRevalidate.push(mainField);
                        if (field?.onChangeMainField) {
                            onChangeCallbacks.push([mainFieldId, field.onChangeMainField]);
                        }
                        break;
                }
                // Process optional unit available with data
                if (unitField && field?.unit && !this._isFieldNotAvailable(unitField)) {
                    unitSoundOriginalValue = await this._setSoundField(pageInstance, unitField, true);
                    switch (await this._setFieldValue(pageInstance, unitField, field.unit)) {
                        case null:
                            // ui.console.warn(`Unit field ${unitFieldId} has no data available`);
                            return false;
                        case true:
                            fieldsToRevalidate.push(unitField);
                            if (field?.onChangeUnitField) {
                                onChangeCallbacks.push([unitFieldId, field.onChangeUnitField]);
                            }
                            break;
                    }
                }
                // Request all changes to page (optional field and unit) before any action
                await pageInstance.$.commitValueAndPropertyChanges();
                // We only process with the case of modified fields
                if (fieldsToRevalidate.length) {
                    // Now process change / validation field segments
                    for (const onChangeCallback of onChangeCallbacks) {
                        const fieldId = onChangeCallback[0];
                        const onChange = onChangeCallback[1];
                        try {
                            await onChange();
                        }
                        catch (error) {
                            throw new xtrem_shared_1.SystemError(`onChange failure on field ${fieldId} :\n${JSON.stringify(error)}`, error);
                        }
                        if (this.isAbortDispatchInProgress) {
                            // ui.console.warn(`Aborting dispatching in progress for field ${fieldId}`);
                            break;
                        }
                    }
                    // CAUTION: when aborting is in progress, context may be invalid :
                    // This behavior is taken into account by the possibility of abandoning dispatching,
                    // and possibly executing a postDone action at the end. If possible, this action should
                    // avoid being asynchronous for this reason.
                    if (!this.isAbortDispatchInProgress) {
                        // If we have performed at least one change action, we redo a commit of the page
                        if (onChangeCallbacks.length) {
                            await pageInstance.$.commitValueAndPropertyChanges();
                        }
                        // ui.console.debug(`Starting Validation`);
                        for (const fieldToValidate of fieldsToRevalidate) {
                            validation &&= (await fieldToValidate
                                .validate()
                                .then(data => typesLib.strictEq(data, undefined))
                                .catch(() => false));
                            if (!validation) {
                                ui.console.warn(`Validation failure on field : ${fieldToValidate.id}`);
                                break;
                            }
                        }
                    }
                }
            }
        }
        catch (error) {
            ui.console.error(`Error dispatching ${mainFieldId} :\n${error?.message}`);
            return false;
        }
        finally {
            let isChanged = typesLib.strictNe((await this._setSoundField(pageInstance, mainField, mainSoundOriginalValue, false)), mainSoundOriginalValue);
            if (typesLib.strictNe(unitSoundOriginalValue, undefined)) {
                isChanged ||= typesLib.strictNe((await this._setSoundField(pageInstance, unitField, unitSoundOriginalValue, false)), unitSoundOriginalValue);
            }
            if (isChanged) {
                await pageInstance.$.commitValueAndPropertyChanges();
            }
        }
        return validation;
    }
    /**
     * Assign value to field
     * @param field current field
     * @param value value to assign in the field
     * @param setFocus optional value must be true when focus must be performed in reInitializing
     * @return false if field has reinitialized, null when no change has performed (to more results)
     */
    /** @internal */
    async _setFieldValue(pageInstance, field, value, setFocus = true) {
        const originalValue = field.value;
        let newValue;
        if (field instanceof ui.fields.Reference) {
            const result = await field.fetchSuggestions(value);
            // We only update if there is a result otherwise the value will remain unchanged
            if (!result.length) {
                await pageInstance.$.sound.error();
                await this._clearField(pageInstance, field, setFocus);
                return false;
            }
            else {
                newValue = result[0];
                await pageInstance.$.sound.success();
                if (typesLib.strictEq(result.length, 1)) {
                    await pageInstance.$.sound.success();
                }
                else {
                    /* Duplicate not supported, keep first result */
                    await pageInstance.$.sound.success();
                }
            }
        }
        else if (field instanceof ui.fields.FilterSelect) {
            newValue = value;
        }
        else if (field instanceof ui.fields.Text) {
            newValue = String(value);
        }
        else if (field instanceof ui.fields.Numeric) {
            newValue = Number(value);
        }
        else if (field instanceof ui.fields.Date) {
            newValue = value instanceof xtrem_date_time_1.DateValue ? value.toString() : value;
        }
        else {
            newValue = value;
        }
        field.value = newValue;
        const isChanged = typesLib.strictNe(originalValue, newValue);
        if (isChanged && !field.isDirty) {
            field.isDirty = true;
        }
        return isChanged;
    }
    /**
     * Convert a user field submitted by full segment reference before evaluation.
     * @param dictionaryFieldsToPrepare fields to check to put in dictionary
     * @returns Screen fields supported for internal use
     */
    /** @internal */
    _prepareSegment(dictionaryFieldsToPrepare) {
        // Reset internal dictionary
        this._dictionarySegment = {};
        const screenFieldsSupported = [];
        for (const [key, field] of Object.entries(dictionaryFieldsToPrepare)) {
            if (field) {
                const screenFieldSupported = {
                    DataTitle: key,
                    sequence: -1,
                    segments: [],
                };
                const mainField = field.mainField;
                // Create directory entry now
                screenFieldSupported.segments.push((this._dictionarySegment[mainField.id] = {
                    currentField: mainField,
                    onChange: field.onChangeMainField,
                    sequence: -2,
                }));
                // Add unit too
                const unitField = field.unitField;
                if (unitField) {
                    screenFieldSupported.segments.push((this._dictionarySegment[unitField.id] = {
                        currentField: unitField,
                        onChange: field.onChangeUnitField,
                        sequence: -3,
                    }));
                }
                screenFieldsSupported.push(screenFieldSupported);
            }
        }
        return screenFieldsSupported;
    }
    /**
     *  Analyze all fields existing
     * @param pageInstance current page
     * @param dictionaryFieldToCheck
     * @returns
     */
    /** @internal */
    _existingFields(pageInstance, dictionaryFieldToCheck) {
        const $items = pageInstance?._pageMetadata.layout.$items ?? [];
        const preparedFields = this._prepareSegment(dictionaryFieldToCheck);
        const selectedFields = [];
        const dictionarySegment = this._dictionarySegment;
        let sequence = 0;
        // ui.console.warn(`Dictionary prepared :\n${JSON.stringify(dictionarySegment)}`);
        // ui.console.warn(`Field prepared :\n${JSON.stringify(preparedFields)}`);
        // For now, the check does not test if the current field is available or not
        if (typesLib.gt($items.length, 0) && typesLib.gt(preparedFields.length, 0)) {
            // Start analysis
            this._existFields(pageInstance, $items, dictionarySegment, (segments, field) => true, (segments, field) => {
                field.sequence = (sequence = typesLib.add(sequence, 1));
                return !Object.entries(segments).some(([key, value]) => typesLib.lte(value.sequence, 0));
            });
            // Remove invalid dictionary entries : Each field must be have all segments available
            for (const [key, value] of Object.entries(dictionarySegment)) {
                if (typesLib.lte(value.sequence, 0)) {
                    delete dictionarySegment[key];
                }
            }
        }
        // Update internal dictionary now
        this._dictionarySegment = dictionarySegment;
        // Retain entry only with full segments
        for (const key in preparedFields) {
            const fields = preparedFields[key];
            if (fields.segments.every(thisSegment => typesLib.strictEq(dictionarySegment[thisSegment.currentField.id]?.currentField.id, thisSegment.currentField.id))) {
                selectedFields.push(fields);
            }
            else {
                // Remove partial entries
                for (const segment of fields.segments) {
                    delete this._dictionarySegment[segment.currentField.id];
                }
            }
        }
        // ui.console.debug(`Dictionary updated :\n${JSON.stringify(this._dictionarySegment)}`);
        // ui.console.debug(`Prepared fields :\n${JSON.stringify(preparedFields)}`);
        // ui.console.debug(`Selected fields :\n${JSON.stringify(selectedFields)}`);
        return selectedFields;
    }
    /**
     * Analyze list of fields submitted (recursive)
     * @param pageInstance current page
     * @param $items
     * @param segments current segments list for field
     * @param check verify is possible to use the field for action
     * @param action perform action
     * @returns
     */
    /** @internal */
    _existFields(pageInstance, $items, segments, check, action) {
        // Index from array
        for (let i in $items) {
            let $bind = undefined;
            let pageField = undefined;
            let field = undefined;
            if (($bind = $items[i].$bind) && (pageField = pageInstance[$bind]) && (field = segments[pageField?.id])) {
                // ui.console.debug(`Bind field : ${JSON.stringify($bind)}`);
                if (check(segments, field) && action(segments, field)) {
                    return true;
                }
                continue;
            }
            else {
                // ui.console.debug(`Skipping binded field : ${JSON.stringify($bind)}`);
                const $items2 = $items[i].$layout?.$items ?? [];
                // recursively search for 'binded' components inside the layout's containers like section & blocks
                if (typesLib.gt($items2.length, 0) && this._existFields(pageInstance, $items2, segments, check, action)) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * Section page validation
     * */
    /**
     * Perform validation of all segments managed
     * @param pageInstance current page
     * @param errorMessage translated message error errorMessageCheckYourEntry
     * @returns
     */
    async validate(pageInstance, errorMessage = exports.errorMessageCheckYourEntry) {
        const fieldErrors = new Array();
        const isPageValid = await pageInstance.$.page.isValid;
        let areAllFieldsValid = true;
        for (const field of this._screenFieldsSupported) {
            for (const segment of field.segments) {
                if (!(await this._validateField(segment.currentField))) {
                    areAllFieldsValid = false;
                    fieldErrors.push(segment.currentField.id);
                }
            }
        }
        if (typesLib.gt(fieldErrors.length, 0)) {
            this._notifyFieldNotValid(pageInstance, fieldErrors, errorMessage);
        }
        return isPageValid && areAllFieldsValid;
    }
    /**
     * Perform field validation
     * @param field current field
     * @returns true when field is validated
     */
    /** @internal */
    async _validateField(field) {
        return !(await field.validate());
    }
    /**
     * Notify all fields invalids
     * @param pageInstance current page
     * @param errorMessage translated message error errorMessageCheckYourEntry
     * @param fields
     */
    /** @internal */
    async _notifyFieldNotValid(pageInstance, fields, errorMessage = exports.errorMessageCheckYourEntry) {
        pageInstance.$.removeToasts();
        pageInstance.$.showToast((0, xtrem_shared_1.format)(errorMessage, 'base', {
            fieldNames: fields.map(field => pageInstance[field]?.title),
        }), { type: 'error', timeout: 5000 });
    }
    /**
     * Validate current page
     * @param pageInstance current page
     * @param errorMessage translated message error errorMessageError
     * @returns false when error occur
     */
    async validatePage(pageInstance, errorMessage = exports.errorMessageError) {
        const errors = await pageInstance.$.page.validate();
        if (!errors.length) {
            return true;
        }
        const zz = pageInstance.$.removeToasts();
        pageInstance.$.showToast(`${(0, xtrem_shared_1.format)(errorMessage, 'base')}: ${errors[0]}`, {
            type: 'error',
            timeout: 30000,
        });
        return false;
    }
    /**
     * perform fields validation with detailed errors lists.
     * @param pageInstance current page
     * @param errorMessage errorMessageCheckYourEntry
     * @returns false when error occur
     */
    async validateWithDetails(pageInstance, errorMessage = exports.errorMessageCheckYourEntry) {
        // Request change to page
        // to handle such edgy cases as user clearing an input from a field and then directly clicking on a button without blurring that field
        await pageInstance.$.commitValueAndPropertyChanges();
        const errors = await pageInstance.$.page.validateWithDetails();
        if (!errors.length) {
            return true;
        }
        pageInstance.$.removeToasts();
        pageInstance.$.showToast((0, xtrem_shared_1.format)(errorMessage, 'base', {
            fieldNames: errors.map((error) => `${pageInstance[error.elementId]?.title ?? error.elementId}`),
        }), { type: 'error', timeout: 5000 });
        return false;
    }
}
exports.ManagementServiceGs1 = ManagementServiceGs1;
//# sourceMappingURL=management-service-gs-1.js.map