"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setOrderOfPageTableHeaderBusinessActions = exports.setOrderOfPageTableFieldActions = exports.setOrderOfPageHeaderDropDownActions = exports.setOrderOfPageHeaderQuickActions = exports.setOrderOfPageBusinessActions = exports.setDisplayOfCommonPageActions = exports.setApplicativePageCrudActions = exports.saveAndClosePanel = exports.cancelAndClosePanel = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const ui = require("@sage/xtrem-ui");
/**
 * Run the standard cancel action and close the page if it is opened in the sidebar
 * @param pageInstance the page where save is triggered
 */
async function cancelAndClosePanel(pageInstance) {
    try {
        // If the page is in the sidebar, in case of error, the message is not well managed by the framework
        // => it's better to manage it here
        if (pageInstance.$.queryParameters.called) {
            pageInstance.$.finish(pageInstance.$.values);
        }
        else {
            await pageInstance.$standardCancelAction.execute(!pageInstance.$.queryParameters.called);
        }
    }
    catch (err) {
        pageInstance.$.showToast(err.message, { type: 'error' });
        pageInstance.$.loader.isHidden = true;
    }
}
exports.cancelAndClosePanel = cancelAndClosePanel;
/**
 * Run the standard save action and close the page if it is opened in the sidebar
 * @param pageInstance the page where save is triggered
 */
async function saveAndClosePanel(pageInstance) {
    try {
        // If the page is in the sidebar, in case of error, the message is not well managed by the framework
        // => it's better to manage it here
        await pageInstance.$standardSaveAction.execute(!pageInstance.$.queryParameters.called);
        if (pageInstance.$.queryParameters.called) {
            pageInstance.$.finish(pageInstance.$.values);
        }
    }
    catch (err) {
        pageInstance.$.showToast(err.message, { type: 'error' });
        pageInstance.$.loader.isHidden = true;
    }
}
exports.saveAndClosePanel = saveAndClosePanel;
/**
Manage activation of CRUD buttons
If the page is opened in the sidebar, the buttons Save and New are hidden
@param page: the page to manage
@param values: default value = page.$.values
@param isDirty: default value = page.$.isDirty,
@param cb: callback
*/
function setApplicativePageCrudActions(
/*
_page: ui.Page,
_isDirty: boolean = _page.$.isDirty,
*/
{ page, isDirty = page.$.isDirty, isNewRecord = false, cancel = [], remove = [], duplicate = [], save = [], actions = [], }) {
    const cancelActions = Array.isArray(cancel) ? cancel : [cancel];
    const removeActions = Array.isArray(remove) ? remove : [remove];
    const duplicateActions = Array.isArray(duplicate) ? duplicate : [duplicate];
    const saveActions = Array.isArray(save) ? save : [save];
    const otherActions = actions;
    const recordId = page.$.recordId;
    const pageAsPanel = Boolean(page.$.queryParameters?.called);
    // The cancel and save actions are only visible when the page is dirty
    if (!page.$.isInDialog) {
        [...cancelActions, ...saveActions].forEach(action => {
            if (isNewRecord) {
                action.isDisabled = !isDirty;
            }
            else {
                action.isHidden = !isDirty;
            }
        });
    }
    else {
        // In dialogs the cancel action is always available
        saveActions.forEach(action => {
            action.isDisabled = !isDirty;
        });
    }
    // The footer business actions are only visible when the page is not dirty
    actions.forEach(action => {
        if (!cancelActions.includes(action) && !saveActions.includes(action)) {
            action.isHidden = isDirty;
        }
    });
    // Duplicate and delete action are only available if the page is NOT dirty, the user should only duplicate or delete clean records
    [...duplicateActions, ...removeActions].forEach(action => {
        if (!recordId) {
            // Moreover only records that have already been saved to the database can be duplicated or deleted
            action.isHidden = true;
        }
        else {
            action.isDisabled = !!isDirty;
            action.isHidden = false;
        }
    });
    // Hide all actions except Save in panels, overrides previous logic from above, this condition must be at the end
    if (pageAsPanel) {
        [...removeActions, ...duplicateActions, ...cancelActions].forEach(action => {
            action.isHidden = true;
        });
    }
}
exports.setApplicativePageCrudActions = setApplicativePageCrudActions;
// note : any is needed as we can have a ui.MENU_SEPARATOR in the array, which is not a pageAction
function getActions(actions) {
    return Array.isArray(actions) ? actions : [actions];
}
/**
Manage default display of most page common actions
    @param page:  page
    @param isDirty: dirty state
    @param cancel: the cancel action
    @param remove: the remove action
    @param duplicate: the duplicate action
    @param save: the save action
    @param businessActions: all other business actions
*/
function setDisplayOfCommonPageActions({ page, isDirty = page.$.isDirty, cancel = [], remove = [], duplicate = [], save = [], businessActions = [], }) {
    const cancelActions = getActions(cancel);
    const removeActions = getActions(remove);
    const duplicateActions = getActions(duplicate);
    const saveActions = getActions(save);
    const otherActions = getActions(businessActions);
    const isPageAsPanel = Boolean(page.$.queryParameters?.called);
    const isNewRecord = page.$.recordId ? false : true;
    // Cancel & Save management
    [...cancelActions, ...saveActions].forEach(action => {
        if (isDirty) {
            action.isHidden = false;
            action.isDisabled = false;
        }
        else if (isNewRecord) {
            action.isHidden = false;
            action.isDisabled = true;
        }
        else {
            action.isHidden = true;
            action.isDisabled = true;
        }
    });
    // Other footer BusinessActions management
    if (isDirty || isNewRecord) {
        otherActions.forEach(action => {
            action.isHidden = true;
            // action.isDisabled = true; // never manage isDisabled for other business actions: they are only displayed/hidden
        });
    }
    // Duplicate and delete action are only available if the page is NOT dirty, the user should only duplicate or delete clean records
    [...duplicateActions, ...removeActions].forEach(action => {
        if (isNewRecord) {
            // Moreover only records that have already been saved to the database can be duplicated or deleted
            action.isHidden = true;
        }
        else {
            action.isDisabled = !!isDirty;
            action.isHidden = false;
        }
    });
    // Hide all actions except Save in panels, overrides previous logic from above, this condition must be at the end
    if (isPageAsPanel) {
        [...removeActions, ...duplicateActions, ...cancelActions].forEach(action => {
            action.isHidden = true;
        });
    }
}
exports.setDisplayOfCommonPageActions = setDisplayOfCommonPageActions;
/**
Manage order of business action buttons
@param cancel: the cancel action
@param save: the save action
@param businessActions: all other business actions
*/
function setOrderOfPageBusinessActions({ cancel = [], save = [], businessActions = [], }) {
    const cancelActions = getActions(cancel);
    const saveActions = getActions(save);
    const otherActions = getActions(businessActions);
    return [...otherActions.reverse(), ...cancelActions, ...saveActions];
    // intentionally left commented
    // const uxOrderRule: number = 1; // a rule defined by the UX team, default one is the 1st
    // switch (uxOrderRule) {
    //     default:
    //     case 1:
    //         return [...otherActions.reverse(), ...cancelActions, ...saveActions];
    //     case 2:
    //         return [...otherActions, ...cancelActions, ...saveActions];
    //     case 3:
    //         return [...saveActions, ...cancelActions, ...otherActions.reverse()];
    //     case 4:
    //         return [...saveActions, ...cancelActions, ...otherActions];
    // }
}
exports.setOrderOfPageBusinessActions = setOrderOfPageBusinessActions;
/**
Manage order of header quick action buttons
@param duplicate: the duplicate action
@param quickActions: all other quick actions
*/
function setOrderOfPageHeaderQuickActions({ duplicate = [], quickActions = [], }) {
    const duplicateActions = getActions(duplicate);
    const listActions = getActions(quickActions);
    const returnedActions = [...duplicateActions, ...listActions];
    return [...returnedActions.reverse()];
}
exports.setOrderOfPageHeaderQuickActions = setOrderOfPageHeaderQuickActions;
/**
Manage order of header dropdown action buttons
@param remove: the delete action
@param dropDownBusinessActions: all other dropdown actions and separators
*/
function setOrderOfPageHeaderDropDownActions({ remove = [], dropDownBusinessActions = [], }) {
    const removeActions = getActions(remove);
    const listActions = getActions(dropDownBusinessActions);
    if (typesLib.gt(removeActions.length, 0)) {
        return [...removeActions, ui.MENU_SEPARATOR, ...listActions].reverse();
    }
    return [...listActions].reverse();
}
exports.setOrderOfPageHeaderDropDownActions = setOrderOfPageHeaderDropDownActions;
/**
Manage order of table action buttons
@param actions: all table actions
*/
function setOrderOfPageTableFieldActions({ actions = [], }) {
    const returnedActions = getActions(actions);
    return [...returnedActions];
    // intentionally left commented
    // return [...returnedActions.reverse()];
}
exports.setOrderOfPageTableFieldActions = setOrderOfPageTableFieldActions;
/**
Manage order of table header business action buttons
@param actions: all table actions
*/
function setOrderOfPageTableHeaderBusinessActions({ actions = [], }) {
    const returnedActions = getActions(actions);
    return [...returnedActions.reverse()];
}
exports.setOrderOfPageTableHeaderBusinessActions = setOrderOfPageTableHeaderBusinessActions;
//# sourceMappingURL=applicative-crud-actions.js.map