"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const util_stream_1 = require("@smithy/util-stream");
const xtrem_deployment_1 = require("@sage/xtrem-deployment");
const fs = require("fs");
const path_1 = require("path");
const stream_1 = require("stream");
const consumers_1 = require("stream/consumers");
const utils_1 = require("./utils");
const ERROR_TENANT_ID = "error";
class LocalWrapper {
    basePath;
    app;
    setUnitTestMode(_value) {
        // nothing to do here
    }
    constructor(basePath, app) {
        this.basePath = basePath ? basePath : process.cwd();
        this.app = app;
    }
    getFullPath = (p) => {
        return (0, path_1.join)(this.basePath, this.app || "", p);
    };
    createLocalDir = (dirPath) => {
        fs.mkdirSync(dirPath, { recursive: true });
    };
    createLocalFile = (filePath, fileContent) => {
        const fullPath = this.getFullPath(filePath);
        // Make sure the folder exists (filePath may contain some '/')
        this.createLocalDir((0, path_1.dirname)(fullPath));
        fs.writeFileSync(fullPath, fileContent);
    };
    readLocalFile = (filePath) => {
        return fs.readFileSync(this.getFullPath(filePath));
    };
    listLocalFiles = (dirPath) => {
        return fs.readdirSync(this.getFullPath(dirPath));
    };
    async createFile(tenantId, objectKey, description, byteStream, expiration) {
        try {
            let tagging = "";
            if (description) {
                tagging = (0, utils_1.concatenateStr)(tagging, "description", description);
            }
            if (expiration) {
                tagging = (0, utils_1.concatenateStr)(tagging, "expiration", expiration);
            }
            if (tenantId !== ERROR_TENANT_ID) {
                const data = await (0, consumers_1.buffer)(stream_1.Readable.from(byteStream));
                this.createLocalFile((0, path_1.join)(tenantId, objectKey), data);
                this.createLocalFile((0, path_1.join)(tenantId, `${objectKey}.tags`), tagging);
                return { ETag: "dummyEtag" };
            }
        }
        catch (err) {
            throw new Error(`Error while creating file in local directory: ${err}`);
        }
        return undefined;
    }
    async updateExpiration(tenantId, objectKey, expiration) {
        return this.updateTagging(tenantId, objectKey, { TagSet: [{ Key: "expiration", Value: expiration }] });
    }
    async updateTagging(tenantId, objectKey, tagging) {
        const pathTags = (0, path_1.join)(tenantId, `${objectKey}.tags`);
        const data = this.readLocalFile(pathTags);
        const tagsSplitted = data.toString().split("&");
        const tagsFromFile = {
            TagSet: tagsSplitted.map((rawTag) => {
                const tagsplit = rawTag.split("=");
                return { Key: tagsplit[0], Value: tagsplit[1] };
            })
        };
        const updated = (0, utils_1.updateTagging)(tagsFromFile, tagging);
        if (updated && tagsFromFile.TagSet) {
            const newData = tagsFromFile.TagSet.map((tag) => `${tag.Key}=${tag.Value}`).join("&");
            this.createLocalFile(pathTags, newData);
        }
    }
    getFormattedTag = (expression) => {
        const results = [];
        const tags = expression.split("&");
        tags.forEach(tag => {
            const formattedTag = tag.split("=");
            results.push({
                Key: formattedTag[0],
                Value: formattedTag[1],
            });
        });
        return results;
    };
    async readFile(tenantId, objectKey) {
        try {
            if (tenantId !== ERROR_TENANT_ID) {
                return {
                    body: (0, util_stream_1.sdkStreamMixin)(stream_1.Readable.from(this.readLocalFile((0, path_1.join)(tenantId, objectKey)).toString())),
                    tagData: {
                        TagSet: this.getFormattedTag(this.readLocalFile((0, path_1.join)(tenantId, `${objectKey}.tags`)).toString()),
                    },
                };
            }
        }
        catch (err) {
            throw new Error(`Error while reading file in local directory: ${err}`);
        }
        return undefined;
    }
    getFormattedFiles(files) {
        const formattedFiles = [];
        files.forEach(file => {
            formattedFiles.push({
                Key: file,
                LastModified: "2020-06-08T08:52:12.000Z",
                ETag: "dummyETag",
            });
        });
        return formattedFiles;
    }
    async listFiles(tenantId, continuationToken) {
        return this.list(xtrem_deployment_1.TenantDataType.DATA, tenantId, continuationToken);
    }
    async list(tenantDataType, tenantId, _continuationToken) {
        try {
            if (tenantId !== ERROR_TENANT_ID) {
                const pathFiles = tenantDataType === xtrem_deployment_1.TenantDataType.DATA ? tenantId : (0, path_1.join)(tenantDataType, tenantId);
                const files = this.listLocalFiles(pathFiles);
                return {
                    IsTruncated: false,
                    Contents: this.getFormattedFiles(files),
                    Prefix: tenantId,
                    Name: this.basePath,
                    KeyCount: this.listLocalFiles(pathFiles).length,
                };
            }
        }
        catch (err) {
            throw new Error("Error while listing files in local directory: " + err);
        }
        return undefined;
    }
    async listExports(tenantId, continuationToken) {
        return this.list(xtrem_deployment_1.TenantDataType.EXPORT, tenantId, continuationToken);
    }
    async listImports(tenantId, continuationToken) {
        return this.list(xtrem_deployment_1.TenantDataType.IMPORT, tenantId, continuationToken);
    }
    async listTenants() {
        const tenants = new Set();
        const tenantsFiles = {};
        for (const tdt of Object.values(xtrem_deployment_1.TenantDataType)) {
            const files = this.listLocalFiles(tdt);
            for (const file of files) {
                this.addFilesFromFolder(tdt, file, tenants, tenantsFiles);
            }
        }
        tenants.clear();
        return tenantsFiles;
    }
    addFilesFromFolder(tdt, file, tenants, tenantsFiles) {
        const fileInfo = fs.statSync((0, path_1.join)(tdt, file));
        if (fileInfo.isDirectory()) {
            const filesTenant = this.listLocalFiles((0, path_1.join)(tdt, file));
            if (!tenants.has(file)) {
                tenants.add(file);
                tenantsFiles[file] = this.getFormattedFiles(filesTenant);
            }
            else {
                const tenantFilesArray = tenantsFiles[file];
                for (const tenantFile of this.getFormattedFiles(filesTenant)) {
                    tenantFilesArray.push(tenantFile);
                }
            }
        }
    }
    /**
     * For local implementation we just return the path of the file
     * @param tenantId
     * @param objectKey
     * @param _urlTimeToLiveInSec
     */
    async generateDownloadPresignedUrl(tenantId, objectKey, _urlTimeToLiveInSec, filename) {
        let newFilePath = `/${xtrem_deployment_1.TenantDataType.DATA}/${tenantId}/uploads/${objectKey}`;
        if (filename) {
            newFilePath += `?${filename}`;
        }
        return this.getFullPath(newFilePath);
    }
    /**
     * For local implementation we just return the path of the file
     * @param tenantId
     * @param contextId
     * @param urlTimeToLiveInSec
     */
    generateUploadPresignedUrl(tenantId, contextId, urlTimeToLiveInSec) {
        return this.generateDownloadPresignedUrl(tenantId, contextId, urlTimeToLiveInSec);
    }
    /**
     * For local implementation we just return the path of the file
     * @param tenantId
     * @param contextId
     * @param urlTimeToLiveInSec
     */
    async generateAttachmentUploadPresignedUrl(tenantId, contextId, urlTimeToLiveInSec) {
        return this.generateDownloadPresignedUrl(tenantId, contextId, urlTimeToLiveInSec);
    }
    readAsyncContextResult(tenantId, contextId, contextKind) {
        return this.readFile(tenantId, `/${contextKind === xtrem_deployment_1.ASYNC_CONTEXT_CONST.CONTEXT_ERP_ADAPTER_REQUEST ? xtrem_deployment_1.S3_ERP_ADAPTER_UPLOAD_PATH : xtrem_deployment_1.S3_UPLOAD_PATH}/${contextId}`);
    }
    async generateAttachmentDownloadPresignedUrl(tenantId, attachmentId, _urlTimeToLiveInSec) {
        const newFilePath = `/${xtrem_deployment_1.TenantDataType.DATA}/${tenantId}/attachments/${attachmentId}`;
        return this.getFullPath(newFilePath);
    }
}
exports.default = LocalWrapper;
