"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const client_s3_1 = require("@aws-sdk/client-s3");
const credential_providers_1 = require("@aws-sdk/credential-providers");
const s3_request_presigner_1 = require("@aws-sdk/s3-request-presigner");
const xtrem_deployment_1 = require("@sage/xtrem-deployment");
const utils_1 = require("./utils");
const DEFAULT_UPLOAD_TTL_IN_SEC = 60 * 60;
class S3Wrapper {
    s3;
    bucket;
    app;
    maxKeys;
    signedUrlGetter;
    setMaxKeys(value) {
        this.maxKeys = value;
    }
    setBucket(value) {
        this.bucket = value;
    }
    setUnitTestMode(_value) {
        // nothing to do here
    }
    constructor(configFS, logger, s3Mock) {
        this.maxKeys = (configFS && configFS.maxListFiles) || 1000;
        this.bucket = configFS.bucket || "";
        this.app = configFS.app;
        if (!configFS.awsConfigOverride) {
            configFS.awsConfigOverride = (0, credential_providers_1.fromNodeProviderChain)();
        }
        if (s3Mock) {
            this.s3 = s3Mock;
            this.signedUrlGetter = s3Mock.signedUrlGetter;
            logger("FileStorageHandler is mocking S3 library");
        }
        else {
            this.s3 = new client_s3_1.S3(configFS.awsConfigOverride);
            this.signedUrlGetter = s3_request_presigner_1.getSignedUrl;
        }
    }
    getApp = () => this.app ? `${this.app}/` : "";
    getPath = (tenantId, objectKey) => `${xtrem_deployment_1.TenantDataType.DATA}/${this.getApp()}${tenantId}/${objectKey}`;
    async createFile(tenantId, objectKey, description, body, expiration) {
        try {
            const params = {
                Body: body,
                Bucket: this.bucket,
                Key: this.getPath(tenantId, objectKey),
                ContentType: "application/string",
            };
            let tagging = "";
            if (description) {
                tagging = (0, utils_1.concatenateStr)(tagging, "description", description);
            }
            if (expiration) {
                tagging = (0, utils_1.concatenateStr)(tagging, "expiration", expiration);
            }
            if (tagging.length > 0) {
                params.Tagging = tagging;
            }
            return await this.s3.putObject(params);
        }
        catch (err) {
            throw new Error(`Error while creating file in bucket: ${err}`);
        }
    }
    async updateTagging(tenantId, objectKey, tagging) {
        try {
            const paramsGet = {
                Bucket: this.bucket,
                Key: this.getPath(tenantId, objectKey)
            };
            const objectTagging = await this.s3.getObjectTagging(paramsGet);
            const updated = (0, utils_1.updateTagging)(objectTagging, tagging);
            if (updated) {
                const params = {
                    Bucket: this.bucket,
                    Key: this.getPath(tenantId, objectKey),
                    Tagging: objectTagging
                };
                return await this.s3.putObjectTagging(params);
            }
            else {
                return { $metadata: {} };
            }
        }
        catch (err) {
            throw new Error(`Error while creating file in bucket: ${err}`);
        }
    }
    async updateExpiration(tenantId, objectKey, expiration) {
        return this.updateTagging(tenantId, objectKey, { TagSet: [{ Key: "expiration", Value: expiration }] });
    }
    async readFile(tenantId, objectKey) {
        try {
            const params = {
                Bucket: this.bucket,
                Key: (0, xtrem_deployment_1.getDownloadUrlPathForTenant)(tenantId, objectKey, this.app)
            };
            const data = await this.s3.getObject(params);
            if (!data.Body) {
                throw new Error("no data");
            }
            const body = data.Body;
            const tagData = await this.s3.getObjectTagging(params);
            return {
                body,
                tagData,
            };
        }
        catch (err) {
            throw new Error(`Error while reading file in bucket: ${err}`);
        }
    }
    async list(tenantDataType, tenantId, continuationToken) {
        try {
            const params = {
                Bucket: this.bucket,
                Prefix: `${tenantDataType}/${this.getApp()}${tenantId}`,
                MaxKeys: this.maxKeys
            };
            if (continuationToken && continuationToken.length > 0) {
                params.ContinuationToken = continuationToken;
            }
            return await this.s3.listObjectsV2(params);
        }
        catch (err) {
            throw new Error(`Error while listing ${tenantDataType} files in bucket: ${err.message}`);
        }
    }
    async listFiles(tenantId, continuationToken) {
        return this.list(xtrem_deployment_1.TenantDataType.DATA, tenantId, continuationToken);
    }
    async listExports(tenantId, continuationToken) {
        return this.list(xtrem_deployment_1.TenantDataType.EXPORT, tenantId, continuationToken);
    }
    async listImports(tenantId, continuationToken) {
        return this.list(xtrem_deployment_1.TenantDataType.IMPORT, tenantId, continuationToken);
    }
    async listTenants() {
        const tenants = new Set();
        const tenantsFiles = {};
        try {
            for (const dataType of Object.values(xtrem_deployment_1.TenantDataType)) {
                const params = {
                    Bucket: this.bucket,
                    Prefix: `${dataType}/${this.getApp()}`,
                    MaxKeys: this.maxKeys
                };
                let files;
                do {
                    files = await this.s3.listObjectsV2(params);
                    if (files && files.IsTruncated) {
                        params.ContinuationToken = files.NextContinuationToken;
                    }
                    this.addFiles(files, tenants, tenantsFiles);
                } while (files && files.IsTruncated);
            }
        }
        catch (err) {
            throw new Error(`Error while listing files in bucket: ${err.message} => ${err.stack}`);
        }
        tenants.clear();
        return tenantsFiles;
    }
    addFiles(files, tenants, tenantsFiles) {
        const keyCount = files && files.KeyCount || 0;
        if (keyCount > 0) {
            for (const s3Object of files.Contents || []) {
                const tenantSplit = s3Object.Key?.split("/") || [];
                if (tenantSplit.length > (this.app ? 3 : 2)) {
                    const tenant = tenantSplit[this.app ? 2 : 1];
                    if (!tenants.has(tenant)) {
                        tenants.add(tenant);
                        tenantsFiles[tenant] = [s3Object];
                    }
                    else {
                        tenantsFiles[tenant].push(s3Object);
                    }
                }
            }
        }
    }
    generateDownloadPresignedUrl(tenantId, objectKey, urlTimeToLiveInSec, filename) {
        const s3PresignedParam = {
            Bucket: this.bucket,
            Key: (0, xtrem_deployment_1.getDownloadUrlPathForTenant)(tenantId, objectKey, this.app),
        };
        if (filename) {
            s3PresignedParam.ResponseContentDisposition = `attachment;filename="${encodeURIComponent(filename)}"`;
        }
        const cmd = new client_s3_1.GetObjectCommand(s3PresignedParam);
        return this.signedUrlGetter(this.s3, cmd, { expiresIn: this.getSanitizedTTL(urlTimeToLiveInSec) });
    }
    generateAttachmentUploadPresignedUrl(tenantId, contextId, urlTimeToLiveInSec) {
        const s3PresignedParam = {
            Bucket: this.bucket,
            Key: (0, xtrem_deployment_1.getAttachmentDirtyPathForTenant)(tenantId, contextId, this.app),
            ContentType: "application/octet-stream"
        };
        // One day aws s3 parsing of tags will be fixed
        /* s3PresignedParam.Tagging = `expiration=${expiration}`; */
        const cmd = new client_s3_1.PutObjectCommand(s3PresignedParam);
        return this.signedUrlGetter(this.s3, cmd, { expiresIn: this.getSanitizedTTL(urlTimeToLiveInSec) });
    }
    async generateAttachmentDownloadPresignedUrl(tenantId, attachmentId, urlTimeToLiveInSec, filename) {
        const s3PresignedParam = {
            Bucket: this.bucket,
            Key: (0, xtrem_deployment_1.getAttachmentPathForTenant)(tenantId, attachmentId, this.app),
        };
        if (filename) {
            s3PresignedParam.ResponseContentDisposition = `attachment;filename="${encodeURIComponent(filename)}"`;
        }
        const cmd = new client_s3_1.GetObjectCommand(s3PresignedParam);
        return this.signedUrlGetter(this.s3, cmd, { expiresIn: this.getSanitizedTTL(urlTimeToLiveInSec) });
    }
    generateUploadPresignedUrl(tenantId, contextId, /* expiration: FileTimeToLive,*/ urlTimeToLiveInSec) {
        const s3PresignedParam = {
            Bucket: this.bucket,
            Key: (0, xtrem_deployment_1.getUploadDirtyUrlPathForTenant)(tenantId, contextId, this.app),
            ContentType: "application/octet-stream"
        };
        // One day aws s3 parsing of tags will be fixed
        /* s3PresignedParam.Tagging = `expiration=${expiration}`; */
        const cmd = new client_s3_1.PutObjectCommand(s3PresignedParam);
        return this.signedUrlGetter(this.s3, cmd, { expiresIn: this.getSanitizedTTL(urlTimeToLiveInSec) });
    }
    readAsyncContextResult(tenantId, contextId, contextKind) {
        return this.readFile(tenantId, `${contextKind === xtrem_deployment_1.ASYNC_CONTEXT_CONST.CONTEXT_ERP_ADAPTER_REQUEST ? xtrem_deployment_1.S3_ERP_ADAPTER_UPLOAD_PATH : xtrem_deployment_1.S3_UPLOAD_PATH}/${contextId}`);
    }
    getSanitizedTTL(timeToLiveInSec) {
        if (!timeToLiveInSec || timeToLiveInSec <= 1) {
            return DEFAULT_UPLOAD_TTL_IN_SEC;
        }
        return timeToLiveInSec;
    }
}
exports.default = S3Wrapper;
