"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileStorageHandler = void 0;
const localWrapper_1 = require("./localWrapper");
const S3Wrapper_1 = require("./S3Wrapper");
const OBJECT_KEY = "object key";
/**
 *  That handler can be run locally or by using S3
 *
 *  - To run locally: you can create a new instance of that handler without passing any params.
 *    The files will be created directly in the current directory
 *    ex: const handler = new FileStorageHandler()
 *
 *  - Using S3: you must pass your bucket name in the constructor.
 *    ex:
 *      const config = {
 *          bucket: "your-bucket-name"
 *      }
 *
 *      const handler = new FileStorageHandler(config)
 */
class FileStorageHandler {
    fileStorageFunctions;
    UNIT_TEST;
    logger;
    constructor(config, s3Mock) {
        // Unless an override is provided, we are using the 'XTREM_ENV' environment variable to detect whether the
        // local wrapper should be used.
        const useLocalWrapper = config?.useLocal != null
            ? config.useLocal
            : process.env.XTREM_ENV === "local" || (config && config.bucket && config.bucket.length === 0);
        this.logger = config?.logger || console.debug;
        if (useLocalWrapper) {
            this.logger("FileStorageHandler module is running in local dev mode");
            this.fileStorageFunctions = new localWrapper_1.default(config?.localBasePath, config?.app);
        }
        else {
            if (!config) {
                throw new Error("No config provided");
            }
            this.logger("FileStorageHandler module is running with S3");
            this.fileStorageFunctions = new S3Wrapper_1.default(config, this.logger, s3Mock);
        }
        this.UNIT_TEST = false;
    }
    setUnitTestMode(value) {
        this.UNIT_TEST = value;
        this.fileStorageFunctions.setUnitTestMode(value);
    }
    static checkParametersForNullOrBlankOrInvalid(param, paramName) {
        if (!param || param === "") {
            throw new Error(paramName + " cannot be empty");
        }
        if (param.endsWith("/")) {
            throw new Error(paramName + " cannot end with /");
        }
    }
    async createFile(tenantId, objectKey, description, byteStream, expiration) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(objectKey, OBJECT_KEY);
        return this.fileStorageFunctions.createFile(tenantId, objectKey, description, byteStream, expiration);
    }
    async updateExpiration(tenantId, objectKey, expiration) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(objectKey, OBJECT_KEY);
        return this.fileStorageFunctions.updateExpiration(tenantId, objectKey, expiration);
    }
    async updateTagging(tenantId, objectKey, tagging) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(objectKey, OBJECT_KEY);
        return this.fileStorageFunctions.updateTagging(tenantId, objectKey, tagging);
    }
    /***
     * Return A stream you can use with .pipe(file) to store on a temp location.
     * @param tenantId
     * @param objectKey
     */
    async readFile(tenantId, objectKey) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(objectKey, OBJECT_KEY);
        return this.fileStorageFunctions.readFile(tenantId, objectKey);
    }
    async list(tenantDataType, tenantId, continuationToken) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        return this.fileStorageFunctions.list(tenantDataType, tenantId, continuationToken);
    }
    async listFiles(tenantId, continuationToken) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        return this.fileStorageFunctions.listFiles(tenantId, continuationToken);
    }
    async listExports(tenantId, continuationToken) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        return this.fileStorageFunctions.listExports(tenantId, continuationToken);
    }
    async listImports(tenantId, continuationToken) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        return this.fileStorageFunctions.listImports(tenantId, continuationToken);
    }
    async listTenants() {
        return this.fileStorageFunctions.listTenants();
    }
    async generateDownloadPresignedUrl(tenantId, objectKey, urlTimeToLiveInSec, filename) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(objectKey, OBJECT_KEY);
        return this.fileStorageFunctions.generateDownloadPresignedUrl(tenantId, objectKey, urlTimeToLiveInSec, filename);
    }
    async generateUploadPresignedUrl(tenantId, contextId, urlTimeToLiveInSec) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(contextId, "context id");
        return this.fileStorageFunctions.generateUploadPresignedUrl(tenantId, contextId, urlTimeToLiveInSec);
    }
    async readAsyncContextResult(tenantId, contextId, contextKind) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(contextId, OBJECT_KEY);
        return this.fileStorageFunctions.readAsyncContextResult(tenantId, contextId, contextKind);
    }
    /***
     * Provide a presigned URL to be shared to the client to upload an attachment.
     * It will be uploaded in a dirty bucket and after scan is done it will end in the tenant's "attachments" subfolder
     * See https://confluence.sage.com/display/XTREEM/Tenant+S3+file+system+organization
     * @param tenantId related tenant id
     * @param contextId the async context id related to the upload make sure you pick attachment upload context kind
     * @param urlTimeToLiveInSec optional, how long in second the upload url should be valid for
     */
    async generateAttachmentUploadPresignedUrl(tenantId, contextId, urlTimeToLiveInSec) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(contextId, "context id");
        return this.fileStorageFunctions.generateAttachmentUploadPresignedUrl(tenantId, contextId, urlTimeToLiveInSec);
    }
    /**
     * Provide a download presigned url to be shared with the client to download an existing attachment
     * It will link to file in the tenant's "attachments" subfolder
     * See https://confluence.sage.com/display/XTREEM/Tenant+S3+file+system+organization
     * @param tenantId Related tenant id
     * @param attachmentId the attachement id, make sure status was correctly uploadded otherwise the file may not exist
     * @param urlTimeToLiveInSec optional, how long in second the download url should be valid for
     */
    async generateAttachmentDownloadPresignedUrl(tenantId, attachmentId, urlTimeToLiveInSec, filename) {
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(tenantId, "tenant id");
        FileStorageHandler.checkParametersForNullOrBlankOrInvalid(attachmentId, "attachmentId id");
        return this.fileStorageFunctions.generateAttachmentDownloadPresignedUrl(tenantId, attachmentId, urlTimeToLiveInSec, filename);
    }
}
exports.FileStorageHandler = FileStorageHandler;
