"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.X3ExpressionParser = void 0;
const xtrem_log_1 = require("@sage/xtrem-log");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const _ = __importStar(require("lodash"));
const ts = __importStar(require("typescript"));
const x3_node_generator_1 = require("../x3-node-generator");
const x3_expression_parser_error_1 = require("./x3-expression-parser-error");
const { parse } = require('espree');
const logger = xtrem_log_1.Logger.getLogger(__filename, 'x3-dictionary-helper');
class X3ExpressionParser {
    constructor(nodeName) {
        this.nodeName = nodeName;
    }
    static resolveLiteral(value) {
        let text;
        if (typeof value === 'string') {
            text = value;
        }
        else {
            text = `${value}`;
        }
        return text;
    }
    static getBinaryToken(operator) {
        switch (operator) {
            case '+':
                return ts.SyntaxKind.PlusToken;
            case '-':
                return ts.SyntaxKind.MinusToken;
            case '/':
                return ts.SyntaxKind.SlashToken;
            case '*':
                return ts.SyntaxKind.AsteriskToken;
            case '||':
                return ts.SyntaxKind.BarBarToken;
            case '&&':
                return ts.SyntaxKind.AmpersandAmpersandToken;
            case '<':
                return ts.SyntaxKind.LessThanToken;
            case '<=':
                return ts.SyntaxKind.LessThanEqualsToken;
            case '>':
                return ts.SyntaxKind.GreaterThanToken;
            case '>=':
                return ts.SyntaxKind.GreaterThanEqualsToken;
            case '==':
                return ts.SyntaxKind.EqualsEqualsToken;
            case '===':
                return ts.SyntaxKind.EqualsEqualsEqualsToken;
            case '!=':
                return ts.SyntaxKind.ExclamationEqualsToken;
            case '!==':
                return ts.SyntaxKind.ExclamationEqualsEqualsToken;
            default:
                throw new Error(`No SyntaxKind token defined for this operator '${operator}'`);
        }
    }
    convertBinaryExpression(expression) {
        let op = expression.operator;
        if (op === '==' || op === '!=') {
            if (expression.right.type !== 'Literal' || expression.right.value !== null) {
                op = op === '==' ? '===' : '!==';
            }
        }
        const left = this.convertExpression(expression.left, true);
        const right = this.convertExpression(expression.right, true);
        const operator = X3ExpressionParser.getBinaryToken(op);
        const newExpression = ts.factory.createBinaryExpression(left.newExpression, operator, right.newExpression);
        return {
            type: left.type,
            escapedText: `${left.escapedText} ${op} ${right.escapedText}`,
            newExpression,
            hasAwait: left.hasAwait || right.hasAwait,
            hasBooleanOperators: left.hasBooleanOperators || right.hasBooleanOperators,
            propertyPaths: _.union(left.propertyPaths, right.propertyPaths),
        };
    }
    getExpressionFromPath(propertyPath, isBinaryExpression) {
        let resultExpression = x3_node_generator_1.X3NodeGenerator.getExpressionFromPath(this.nodeName, propertyPath);
        if (isBinaryExpression) {
            resultExpression = ts.factory.createParenthesizedExpression(ts.factory.createAwaitExpression(resultExpression));
        }
        return resultExpression;
    }
    walkMemberExpression(expression, path = []) {
        // Parent needs the be a member expression or identifier
        switch (expression.object.type) {
            case 'MemberExpression':
                this.walkMemberExpression(expression.object, path);
                break;
            case 'Identifier':
                path.push(expression.object.name);
                break;
            default:
                throw new x3_expression_parser_error_1.ConversionError(expression.object, 'property is not a identifier or member expression');
        }
        if (expression.property.type !== 'Identifier') {
            // expression.expression.type should always be an Identifier
            throw new x3_expression_parser_error_1.ConversionError(expression.property, 'property is not an identifier');
        }
        path.push(expression.property.name);
        return path;
    }
    convertMemberExpression(expression, isBinaryExpression = false) {
        const path = this.walkMemberExpression(expression);
        const escapedText = path.join('.');
        // This will support member expressions like prop1.prop2.prop2
        // But prop1.prop2.prop2.length will fail as length is not property of the node
        // We could support using this.$.get('prop1.prop2.prop2.length')
        // But this is a bit dangerous and we can improve later
        const newExpression = this.getExpressionFromPath(path, isBinaryExpression);
        return {
            type: 'expression',
            escapedText,
            newExpression,
            hasAwait: path.length > 0,
            hasBooleanOperators: false,
            propertyPaths: [escapedText],
        };
    }
    static convertLiteral(value) {
        if (typeof value === 'bigint')
            throw new xtrem_shared_1.SystemError(`cannot convert bigint literal: ${value}`);
        let newExpression;
        let type;
        const escapedText = X3ExpressionParser.resolveLiteral(value);
        if (value === null) {
            type = 'null';
            newExpression = ts.factory.createNull();
        }
        else {
            switch (typeof value) {
                case 'string':
                    type = 'string';
                    newExpression = ts.factory.createStringLiteral(escapedText);
                    break;
                case 'number':
                    type = 'integer';
                    newExpression = ts.factory.createNumericLiteral(escapedText);
                    break;
                case 'boolean':
                    type = 'boolean';
                    newExpression = escapedText === 'true' ? ts.factory.createTrue() : ts.factory.createFalse();
                    break;
                default:
                    throw new Error(`Convert literal error. Not supported literal type ${typeof value}`);
            }
        }
        const result = {
            escapedText,
            newExpression,
            type,
            hasAwait: false,
            hasBooleanOperators: false,
            propertyPaths: [],
        };
        return result;
    }
    convertUnaryExpression(expression) {
        const arg = this.convertExpression(expression.argument, true);
        const op = expression.operator;
        let operator;
        let escapedText;
        let hasBooleanOperators = false;
        switch (op) {
            case '!':
                operator = ts.SyntaxKind.ExclamationToken;
                escapedText = `!${arg.escapedText}`;
                hasBooleanOperators = true;
                break;
            case '-':
                operator = ts.SyntaxKind.MinusToken;
                escapedText = `-${arg.escapedText}`;
                break;
            default:
                throw new x3_expression_parser_error_1.ConversionError(expression, `Unsupported unary operator: ${op}`);
        }
        if (!arg.newExpression)
            throw new Error(`No expression defined.`);
        const newExpression = ts.factory.createPrefixUnaryExpression(operator, arg.newExpression);
        return {
            type: 'expression',
            escapedText,
            newExpression,
            hasAwait: arg.hasAwait,
            hasBooleanOperators: hasBooleanOperators || arg.hasBooleanOperators,
            propertyPaths: arg.propertyPaths,
        };
    }
    convertLogicalExpression(expression) {
        const left = this.convertExpression(expression.left, true);
        const right = this.convertExpression(expression.right, true);
        let operator;
        const op = expression.operator;
        let hasBooleanOperators = false;
        // parenthesize OR because this result may get AND-ed with other results
        switch (op) {
            case '&&':
                operator = ts.SyntaxKind.AmpersandAmpersandToken;
                hasBooleanOperators = true;
                break;
            case '||':
                operator = ts.SyntaxKind.BarBarToken;
                hasBooleanOperators = true;
                break;
            case '??':
                operator = ts.SyntaxKind.QuestionQuestionToken;
                break;
            default:
                // situation should not arrive, only known cases are &&, || and ??
                /* istanbul ignore next */
                throw new x3_expression_parser_error_1.ConversionError(expression, `cannot convert operator: ${op}`);
        }
        const newExpression = ts.factory.createBinaryExpression(left.newExpression, operator, right.newExpression);
        return {
            type: 'expression',
            escapedText: `${left.escapedText} ${op} ${right.escapedText}`,
            newExpression,
            hasAwait: left.hasAwait || right.hasAwait,
            hasBooleanOperators: hasBooleanOperators || left.hasBooleanOperators || right.hasBooleanOperators,
            propertyPaths: _.union(left.propertyPaths, right.propertyPaths),
        };
    }
    convertIdentifier(expression, isBinaryExpression) {
        const propertyPath = [expression.name];
        const newExpression = this.getExpressionFromPath(propertyPath, isBinaryExpression);
        return {
            escapedText: expression.name,
            type: 'string',
            newExpression,
            hasAwait: isBinaryExpression,
            hasBooleanOperators: false,
            propertyPaths: propertyPath,
        };
    }
    convertExpression(expression, isBinaryExpression = false) {
        switch (expression.type) {
            case 'MemberExpression':
                return this.convertMemberExpression(expression, isBinaryExpression);
            case 'Literal':
                return X3ExpressionParser.convertLiteral(expression.value);
            case 'Identifier':
                return this.convertIdentifier(expression, isBinaryExpression);
            case 'BinaryExpression':
                return this.convertBinaryExpression(expression);
            case 'UnaryExpression':
                return this.convertUnaryExpression(expression);
            case 'LogicalExpression':
                return this.convertLogicalExpression(expression);
            default:
                logger.error(`Invalid expression ${expression.type}`);
                throw new x3_expression_parser_error_1.ConversionError(expression, `invalid expression type: ${expression.type}`);
        }
    }
    convertFunction(x3Expression) {
        let str = x3Expression.trim();
        str = `(${str})`;
        try {
            const parsed = parse(str, {
                ecmaVersion: 2022,
            });
            // below are cases that cannot happen, so the tests will never pass into them
            if (parsed.type !== 'Program')
                throw new x3_expression_parser_error_1.ConversionError(parsed, `expected Program, got ${parsed.type}`);
            const body = parsed.body[0]; // since parse.type would always be 'Program', so body would always exist
            if (!body)
                throw new x3_expression_parser_error_1.ConversionError(parsed, 'no program body');
            if (body.type !== 'ExpressionStatement')
                throw new x3_expression_parser_error_1.ConversionError(parsed, `expected ExpressionStatement, got ${body.type}`);
            const { expression } = body;
            const endConvert = this.convertExpression(expression);
            if (!endConvert.newExpression)
                throw new Error(`Missing converted expression ${endConvert.escapedText}`);
            return endConvert;
        }
        catch (e) {
            logger.error(e.message);
            logger.error(`Failed to convert function in ${str}`);
            throw e;
        }
    }
}
exports.X3ExpressionParser = X3ExpressionParser;
//# sourceMappingURL=x3-expression-parser.js.map