"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const xtrem_core_1 = require("@sage/xtrem-core");
const chai_1 = require("chai");
const fsp = require("path");
const Sinon = require("sinon");
// eslint-disable-next-line import/no-relative-packages
const index_1 = require("../index");
// eslint-disable-next-line import/no-relative-packages
const lib_1 = require("../lib");
const access_rights_manager_1 = require("./access-rights-manager");
const data_settings_manager_1 = require("./data-settings-manager");
const nodes_1 = require("./fixtures/nodes");
describe('Verify dependsOn test', () => {
    describe('sanitizeDependsOn function test', () => {
        it("should build path for simple dependency: [ 'costCategoryType' ]", () => {
            const sanitizeFunc = (0, index_1.sanitizeDependsOn)(['costCategoryType']);
            (0, chai_1.expect)(sanitizeFunc).to.eql(['costCategoryType']);
        });
        it("should build path for simple dependency: [ 'costCategoryType', 'foo' ]", () => {
            const sanitizeFunc = (0, index_1.sanitizeDependsOn)(['costCategoryType', 'foo']);
            (0, chai_1.expect)(sanitizeFunc).to.eql(['costCategoryType', 'foo']);
        });
        it("should build path for complex dependency: [{ document: [ 'itemSite' ] }]", () => {
            const sanitizeFunc = (0, index_1.sanitizeDependsOn)([{ document: ['itemSite'] }]);
            (0, chai_1.expect)(sanitizeFunc).to.eql(['document.itemSite']);
        });
        it("should build path for complex dependency: [{ workOrderComponent: [{ workOrder: ['site', 'company'] }]", () => {
            const sanitizeFunc = (0, index_1.sanitizeDependsOn)([{ workOrderComponent: [{ workOrder: ['site', 'company'] }] }]);
            (0, chai_1.expect)(sanitizeFunc).to.eql(['workOrderComponent.workOrder.site', 'workOrderComponent.workOrder.company']);
        });
        it("should build path for complex dependency: [{ list: ['counter'] }, 'counter', 'list']", () => {
            const sanitizeFunc = (0, index_1.sanitizeDependsOn)([{ list: ['counter'] }, 'counter', 'list']);
            (0, chai_1.expect)(sanitizeFunc).to.eql(['list.counter', 'counter', 'list']);
        });
        it("should build path for complex dependency: [ 'quantity', 'price', { purchaseInvoice: ['currency'] },' currency', 'purchaseInvoice']", () => {
            const sanitizeFunc = (0, index_1.sanitizeDependsOn)([
                'quantity',
                'price',
                { purchaseInvoice: ['currency'] },
                'currency',
                'purchaseInvoice',
            ]);
            (0, chai_1.expect)(sanitizeFunc).to.eql([
                'quantity',
                'price',
                'purchaseInvoice.currency',
                'currency',
                'purchaseInvoice',
            ]);
        });
    });
    describe('sanitizeCollectedDependencyPaths function test', () => {
        it("should build path for simple dependency: [ 'this.valuedCost', 'this.quantityInStockUnit' ] ", () => {
            const sanitizeFunc = (0, index_1.sanitizeCollectedDependencyPaths)(['this.valuedCost', 'this.quantityInStockUnit']);
            (0, chai_1.expect)(sanitizeFunc).to.eql(['valuedCost', 'quantityInStockUnit']);
        });
        it("should build path for complex dependency:  ['this.workOrderLine','this.workOrderLine.totalActualQuantity','this.releasedQuantity','this.workOrderLine.document','this.workOrderLine.document.actualMaterialCost','this.workOrderLine.document.actualProcessCost'] ", () => {
            const sanitizeFunc = (0, index_1.sanitizeCollectedDependencyPaths)([
                'this.workOrderLine',
                'this.workOrderLine.totalActualQuantity',
                'this.releasedQuantity',
                'this.workOrderLine.document',
                'this.workOrderLine.document.actualMaterialCost',
                'this.workOrderLine.document.actualProcessCost',
            ]);
            (0, chai_1.expect)(sanitizeFunc).to.eql([
                'workOrderLine',
                'workOrderLine.totalActualQuantity',
                'releasedQuantity',
                'workOrderLine.document',
                'workOrderLine.document.actualMaterialCost',
                'workOrderLine.document.actualProcessCost',
            ]);
        });
    });
    describe('Node tests', () => {
        const logger = xtrem_core_1.Logger.getLogger(fsp.join(__dirname, '../lib/verify-depends-on.ts'), 'verifyDependsOn');
        let loggerSpy;
        xtrem_core_1.Context.accessRightsManager = access_rights_manager_1.copAccessRightsManager;
        xtrem_core_1.Context.dataSettingsManager = data_settings_manager_1.copDataSettingsManager;
        before(() => xtrem_core_1.Application.createDbSchema('xtrem_cop_test'));
        beforeEach(() => {
            loggerSpy = Sinon.spy(logger, 'warn');
        });
        afterEach(() => {
            loggerSpy.restore();
        });
        function createApplication(nodes) {
            return xtrem_core_1.Test.createTestApplication({
                buildDir: fsp.join(__dirname, 'package.json'),
                api: { nodes: { ...nodes, TestUser: xtrem_core_1.TestUser, TestSysVendor: xtrem_core_1.TestSysVendor } },
                schemaName: 'xtrem_cop_test',
            });
        }
        it('should log a warning when a property has a dependency but does not have a dependsOn property attribute specified', async () => {
            xtrem_core_1.Test.application = await createApplication({ TestCopWithoutDependsOnNode: nodes_1.TestCopWithoutDependsOnNode });
            (0, lib_1.verifyDependsOn)(xtrem_core_1.Test.application);
            (0, chai_1.expect)(loggerSpy.calledWith(
            // eslint-disable-next-line @typescript-eslint/quotes
            `<TestCopWithoutDependsOnNode, 'propWithMissingDependsOn'> defaultValue: the dependsOn attribute should be equal to ["value"]`)).to.equal(true);
            loggerSpy.resetHistory();
        });
        it('should not log a warning when a property has a dependsOn property attribute specified', async () => {
            xtrem_core_1.Test.application = await createApplication({ TestCopWithDependsOnNode: nodes_1.TestCopWithDependsOnNode, TestVitalReferenceChild: nodes_1.TestVitalReferenceChild });
            (0, lib_1.verifyDependsOn)(xtrem_core_1.Test.application);
            (0, chai_1.expect)(loggerSpy.getCalls()).to.be.instanceof(Array);
            (0, chai_1.expect)(loggerSpy.getCalls()).to.have.lengthOf(0);
            loggerSpy.resetHistory();
        });
        it('test dependsOn with non vital references', async () => {
            xtrem_core_1.Test.application = await createApplication({ TestCopWithNonVitalReference: nodes_1.TestCopWithNonVitalReference, TestReference: nodes_1.TestReference });
            (0, lib_1.verifyDependsOn)(xtrem_core_1.Test.application);
            (0, chai_1.expect)(loggerSpy.getCalls()).to.be.instanceof(Array);
            (0, chai_1.expect)(loggerSpy.getCalls()).to.have.lengthOf(0);
            loggerSpy.resetHistory();
        });
        after(() => xtrem_core_1.Application.dropDbSchema('xtrem_cop_test'));
    });
    describe('Tests warnings raised by xtrem-core/node-factory', () => {
        const logger = xtrem_core_1.Logger.getLogger(fsp.join(__dirname, '../../../back-end/xtrem-core/index.ts'), 'node-factory');
        let loggerSpy;
        xtrem_core_1.Context.accessRightsManager = access_rights_manager_1.copAccessRightsManager;
        xtrem_core_1.Context.dataSettingsManager = data_settings_manager_1.copDataSettingsManager;
        before(() => xtrem_core_1.Application.createDbSchema('xtrem_cop_test'));
        beforeEach(() => {
            loggerSpy = Sinon.spy(logger, 'warn');
        });
        afterEach(() => {
            loggerSpy.restore();
        });
        function createApplication(nodes) {
            return xtrem_core_1.Test.createTestApplication({
                buildDir: fsp.join(__dirname, 'package.json'),
                api: { nodes: { ...nodes, TestUser: xtrem_core_1.TestUser, TestSysVendor: xtrem_core_1.TestSysVendor } },
                schemaName: 'xtrem_cop_test',
            });
        }
        it('Node-factory should log a warning when a property has a dependsOn property on a non vital reference', async () => {
            xtrem_core_1.Test.application = await createApplication({ TestCopWithNonVitalReference: nodes_1.TestCopWithNonVitalReference, TestReference: nodes_1.TestReference });
            (0, chai_1.expect)(loggerSpy.calledWith(
            // eslint-disable-next-line @typescript-eslint/quotes
            `TestCopWithNonVitalReference.details.dependsOn: TestCopWithNonVitalReference.reference is not part of the vital graph. The rule '{"reference":["details"]}' will be ignored.`)).to.equal(true);
            loggerSpy.resetHistory();
        });
        after(() => xtrem_core_1.Application.dropDbSchema('xtrem_cop_test'));
    });
    describe('Propose fixes for DependsOn rules', () => {
        it('simple rule', () => {
            chai_1.assert.deepEqual((0, lib_1.fixDependsOn)([], ['item']), ['item']);
            chai_1.assert.deepEqual((0, lib_1.fixDependsOn)([], ['item.stockUnit']), [{ item: ['stockUnit'] }]);
            chai_1.assert.deepEqual((0, lib_1.fixDependsOn)(['item'], ['item.stockUnit']), [{ item: ['stockUnit'] }]);
            chai_1.assert.deepEqual((0, lib_1.fixDependsOn)([{ item: ['stockUnit'] }], ['item.stockUnit', 'item.stockPrice']), [
                { item: ['stockUnit', 'stockPrice'] },
            ]);
            chai_1.assert.deepEqual((0, lib_1.fixDependsOn)(['document'], ['document.site', 'document.site.isFinance']), [
                { document: [{ site: ['isFinance'] }] },
            ]);
        });
    });
});
//# sourceMappingURL=verify-depends-on-test.js.map