"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Explorer = exports.ExplorationError = void 0;
const xtrem_shared_1 = require("@sage/xtrem-shared");
const parse = require('espree').parse;
class ExplorationError extends Error {
    constructor(node, message) {
        super(message);
        this.node = node;
    }
}
exports.ExplorationError = ExplorationError;
class Explorer {
    static exploreThisExpression() {
        return 'this';
    }
    // eslint-disable-next-line class-methods-use-this
    exploreMemberExpression(expression) {
        const parent = this.exploreExpression(expression.object);
        if (expression.property.type === 'Identifier') {
            const property = expression.property;
            return [`${parent}.${property.name}`];
        }
        return this.exploreExpression(expression.property);
    }
    exploreChainExpression(expression) {
        if (expression.expression.type === 'MemberExpression') {
            return this.exploreMemberExpression(expression.expression);
        }
        if (expression.expression?.type === 'CallExpression') {
            return this.exploreCallExpression(expression.expression);
        }
        throw new ExplorationError(expression.expression, 'kind of chain expression not supported');
    }
    exploreConditionalExpression(expression) {
        return [
            ...this.exploreExpression(expression.test),
            ...this.exploreExpression(expression.consequent),
            ...this.exploreExpression(expression.alternate),
        ];
    }
    exploreBinaryExpression(expression) {
        return [...this.exploreExpression(expression.left), ...this.exploreExpression(expression.right)];
    }
    exploreLogicalExpression(expression) {
        const explored = [...this.exploreExpression(expression.left), ...this.exploreExpression(expression.right)];
        if (['&&', '||', '??'].indexOf(expression.operator) < 0)
            throw new ExplorationError(expression, `cannot explore operator: ${expression.operator}`);
        return explored;
    }
    exploreUnaryExpression(expression) {
        const explored = this.exploreExpression(expression.argument);
        const op = expression.operator;
        if (['!', '+'].indexOf(op) < 0) {
            throw new ExplorationError(expression, `Unsupported unary operator: ${op}`);
        }
        return explored;
    }
    exploreTemplateLiteral(expression) {
        let explored = [];
        expression.expressions.forEach((_, i) => {
            explored = [...explored, ...this.exploreExpression(expression.quasis[i])];
            explored = [...explored, ...this.exploreExpression(expression.expressions[i])];
        });
        explored = [...explored, ...this.exploreExpression(expression.quasis[expression.quasis.length - 1])];
        return explored;
    }
    exploreBlockStatement(expression) {
        let explored = [];
        // eslint-disable-next-line no-return-assign
        expression.body.forEach(statement => (explored = [...explored, ...this.exploreExpression(statement)]));
        return explored;
    }
    exploreReturnStatement(expression) {
        return this.exploreExpression(expression.argument);
    }
    exploreSwitchStatement(expression) {
        let explored = this.exploreExpression(expression.discriminant);
        expression.cases.forEach((switchCase) => {
            switchCase.consequent.forEach(
            // eslint-disable-next-line no-return-assign
            statement => (explored = [...explored, ...this.exploreExpression(statement)]));
        });
        return explored;
    }
    exploreIfStatement(expression) {
        let explored = [...this.exploreExpression(expression.test), ...this.exploreExpression(expression.consequent)];
        if (expression.alternate)
            explored = [...explored, ...this.exploreExpression(expression.alternate)];
        return explored;
    }
    exploreCallExpression(expression) {
        let explored = this.exploreExpression(expression.callee);
        if (explored.length === 1) {
            const tokens = this.exploreExpression(expression.callee)[0].split('.');
            if (tokens.length > 1)
                explored = [tokens.slice(0, tokens.length - 1).join('.')];
        }
        // eslint-disable-next-line no-return-assign
        expression.arguments.forEach(argument => (explored = [...explored, ...this.exploreExpression(argument)]));
        return explored;
    }
    exploreVariableDeclaration(expression) {
        let explored = [];
        expression.declarations.forEach(declarator => {
            if (declarator.init)
                explored = [...explored, ...this.exploreExpression(declarator.init)];
        });
        return explored;
    }
    exploreProperty(expression) {
        if (expression.key.type !== 'Identifier') {
            throw new Error(`Invalid property type ${expression.key.type}`);
        }
        return this.exploreExpression(expression.value);
    }
    exploreObject(expression) {
        let explored = [];
        expression.properties.forEach(prop => {
            if (prop.type !== 'Property') {
                throw new Error(`Invalid object member ${prop.type}`);
            }
            explored = [...explored, ...this.exploreProperty(prop)];
        });
        return explored;
    }
    exploreArrayExpression(expression) {
        let explored = [];
        expression.elements.forEach(element => {
            if (!element)
                return;
            if (element.type === 'SpreadElement') {
                const spreadElement = element;
                explored = [...explored, ...this.exploreExpression(spreadElement.argument)];
            }
            else {
                explored = [...explored, ...this.exploreExpression(element)];
            }
        });
        return explored;
    }
    explorerNewExpression(expression) {
        let explored = [];
        // eslint-disable-next-line no-return-assign
        expression.arguments.forEach(argument => (explored = [...explored, ...this.exploreExpression(argument)]));
        return explored;
    }
    explorerSequenceExpression(expression) {
        let explored = [];
        // eslint-disable-next-line no-return-assign
        expression.expressions.forEach(element => (explored = [...explored, ...this.exploreExpression(element)]));
        return explored;
    }
    explorerThrowStatement(expression) {
        return this.exploreExpression(expression.argument);
    }
    exploreExpressionStatement(expression) {
        return this.exploreExpression(expression.expression);
    }
    exploreAwaitExpression(expression) {
        return this.exploreExpression(expression.argument);
    }
    exploreExpression(expression) {
        switch (expression.type) {
            case 'ExpressionStatement':
                return this.exploreExpressionStatement(expression);
            case 'BlockStatement':
                return this.exploreBlockStatement(expression);
            case 'MemberExpression':
                return this.exploreMemberExpression(expression);
            case 'TemplateLiteral':
                return this.exploreTemplateLiteral(expression);
            case 'ThisExpression':
                return [Explorer.exploreThisExpression()];
            case 'BinaryExpression':
                return this.exploreBinaryExpression(expression);
            case 'LogicalExpression':
                return this.exploreLogicalExpression(expression);
            case 'UnaryExpression':
                return this.exploreUnaryExpression(expression);
            case 'ObjectExpression':
                return this.exploreObject(expression);
            case 'ChainExpression':
                return this.exploreChainExpression(expression);
            case 'ConditionalExpression':
                return this.exploreConditionalExpression(expression);
            case 'ReturnStatement':
                return this.exploreReturnStatement(expression);
            case 'SwitchStatement':
                return this.exploreSwitchStatement(expression);
            case 'IfStatement':
                return this.exploreIfStatement(expression);
            case 'CallExpression':
                return this.exploreCallExpression(expression);
            case 'ArrowFunctionExpression':
                return this.exploreArrowFunctionExpression(expression);
            case 'VariableDeclaration':
                return this.exploreVariableDeclaration(expression);
            case 'ArrayExpression':
                return this.exploreArrayExpression(expression);
            case 'NewExpression':
                return this.explorerNewExpression(expression);
            case 'SequenceExpression':
                return this.explorerSequenceExpression(expression);
            case 'AwaitExpression':
                return this.exploreAwaitExpression(expression);
            case 'ThrowStatement':
                return this.explorerThrowStatement(expression);
            case 'TemplateElement':
            case 'Literal':
            case 'Identifier':
                return [];
            default:
                throw new ExplorationError(expression, `invalid expression type: ${expression.type}`);
                break;
        }
        return [];
    }
    exploreFunctionExpression(expression) {
        const statements = expression.body.body;
        let explored = [];
        // eslint-disable-next-line no-return-assign
        statements.forEach(statement => (explored = [...explored, ...this.exploreExpression(statement)]));
        return explored;
    }
    exploreArrowFunctionExpression(expression) {
        if (expression.body.type === 'BlockStatement')
            return this.exploreBlockStatement(expression.body);
        return this.exploreExpression(expression.body);
    }
    exploreFunction(fn) {
        let str = (0, xtrem_shared_1.removeCodeCoverageInstrumentation)(fn.toString());
        if (str.startsWith('async ')) {
            str = str.replace(/^async (\w+)/, 'async function');
            /* istanbul ignore next */
        }
        else if (!str.startsWith('function ') && !str.startsWith('()')) {
            str = `function ${str}`;
        }
        str = `(${str})`;
        const parsed = parse(`(${str})`, {
            ecmaVersion: 2022,
        });
        if (parsed.type !== 'Program')
            throw new ExplorationError(parsed, `expected Program, got ${parsed.type}`);
        const body = parsed.body[0];
        if (!body)
            throw new ExplorationError(parsed, 'no program body');
        if (body.type !== 'ExpressionStatement')
            throw new ExplorationError(parsed, `expected ExpressionStatement, got ${body.type}`);
        const allowedExpressions = ['FunctionExpression', 'ArrowFunctionExpression'];
        const expression = body.expression;
        if (!allowedExpressions.includes(expression.type)) {
            throw new ExplorationError(parsed, `expected ${allowedExpressions.join(' or ')}, got ${expression.type}`);
        }
        let explored = expression.type === 'ArrowFunctionExpression'
            ? this.exploreArrowFunctionExpression(expression)
            : this.exploreFunctionExpression(expression);
        const filtered = new Set();
        explored
            .filter(e => e.length && e.startsWith('this.') && e.indexOf('.$.') < 0 && e.indexOf('._') < 0)
            .forEach(dep => filtered.add(dep));
        explored = [...filtered.values()];
        return explored;
    }
}
exports.Explorer = Explorer;
//# sourceMappingURL=explorer.js.map