"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateContext = exports.serviceOptions = exports.nodes = exports.getSizingFromActivityCode = exports.enums = exports.datatypes = exports.allowedAccessCodes = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const xtrem_core_1 = require("@sage/xtrem-core");
const xtrem_service_1 = require("@sage/xtrem-service");
const xtrem_x3_syracuse_1 = require("@sage/xtrem-x3-syracuse");
const fs = require("fs");
const fsp = require("path");
const datatypes = require("./data-types");
exports.datatypes = datatypes;
const enums = require("./enums");
exports.enums = enums;
const nodes = require("./nodes");
exports.nodes = nodes;
const serviceOptions = require("./service-options");
exports.serviceOptions = serviceOptions;
const services = require("./services");
const services_1 = require("./services");
Object.defineProperty(exports, "allowedAccessCodes", { enumerable: true, get: function () { return services_1.allowedAccessCodes; } });
Object.defineProperty(exports, "getSizingFromActivityCode", { enumerable: true, get: function () { return services_1.getSizingFromActivityCode; } });
/**
 * Get the secret from the config
 * @returns
 */
function getSecret() {
    const { security } = (xtrem_core_1.ConfigManager.current || {});
    const { syracuse } = security || {};
    if (!syracuse.secret)
        throw new Error('xtrem.security.secret config value must be set');
    return syracuse.secret;
}
/**
 *
 * Get the config from the request headers
 * @param request
 * @returns
 */
function getConfigFromRequest(request) {
    const base64 = request.headers['x-etna-config'];
    const config = xtrem_core_1.ConfigManager.current;
    const x3Config = config.x3;
    if (!base64) {
        if (x3Config && x3Config.development && x3Config.development.folderName) {
            return { folderName: x3Config.development.folderName, session: { userName: config.user } };
        }
        throw new xtrem_core_1.SystemError('Missing folder name');
    }
    if (Array.isArray(base64)) {
        throw new xtrem_core_1.SystemError('multiple x-etna-config headers not allowed');
    }
    const secret = getSecret();
    return (0, xtrem_x3_syracuse_1.decrypt)(base64, secret);
}
/**
 * Get the workerId from the request headers
 * For X3 we use the folder name passed in the header or the folder name in the config when we are in development mode
 * @param request
 * @returns
 */
function getRequestSource(request) {
    const parsed = getConfigFromRequest(request);
    if (parsed.folderName) {
        return parsed.folderName;
    }
    throw new xtrem_core_1.SystemError('x-etna-config missing folder name');
}
/**
 *
 * Get the routing key from the request headers
 * @param request
 * @returns
 */
function getRequestRoutingKey(request) {
    const parsed = getConfigFromRequest(request);
    if (parsed.session) {
        return parsed.session.userName ?? parsed.session.login ?? parsed.login;
    }
    return parsed.folderName;
}
/**
 * Look at the add-ons folder and return a of end points to seed
 * We can use the folders in the add-ons folder as a best guess of the X3 endpoints
 * @returns
 */
function getSourcesToSeed() {
    const addOnsConfigPath = xtrem_core_1.ConfigManager.current.addOns?.folder;
    // default add-on path relative to the application root folder
    const defaultPath = fsp.join(process.cwd(), 'add-ons');
    // The root path to look for add-ons
    const addOnsRootPath = addOnsConfigPath ?? defaultPath;
    if (fs.existsSync(addOnsRootPath)) {
        const addOns = fs.readdirSync(addOnsRootPath);
        return addOns.filter(f => fs.statSync(fsp.join(addOnsRootPath, f)).isDirectory() && typesLib.strictNe(f, 'node_modules'));
    }
    return [];
}
function updateContext() {
    xtrem_core_1.Context.accessRightsManager = new services.XtremX3AccessRightsManager();
    xtrem_core_1.Context.localizationManager = new services.XtremX3LocalizationManager();
    xtrem_core_1.CoreHooks.createServiceOptionManager = application => new services.XtremX3ServiceOptionManager(application);
    xtrem_core_1.CoreHooks.sysManager = {
        getUserNode: () => nodes.SysUser,
    };
    xtrem_core_1.CoreHooks.createPackageManager = application => new services_1.X3PackageManager(application);
    xtrem_service_1.GraphqlEndpointHooks.getRequestSource = getRequestSource;
    xtrem_service_1.GraphqlEndpointHooks.getRequestRoutingKey = getRequestRoutingKey;
    xtrem_service_1.GraphqlEndpointHooks.getSourcesToSeed = getSourcesToSeed;
}
exports.updateContext = updateContext;
updateContext();
//# sourceMappingURL=index.js.map