"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isStockJournalInRecord = exports.getStockResults = exports.disableButton = exports.addValueToSelectedIdentifier = exports.isProductGlobalReceivedIssuedInStock = exports.getIdentifierFieldsCount = exports.getIdentifierValues = exports.initFieldsToBeVisible = exports.checkIdentifierField = exports.onChangeFilterLicensePlateNumber = exports.onRemoveLpnFilter = exports.onChangeFilterStock = exports.readStockIdFromSerialNumber = exports.calculateEndingSerialNumber = exports.optionsIdentifier = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const xtrem_client_1 = require("@sage/xtrem-client");
const ui = require("@sage/xtrem-ui");
exports.optionsIdentifier = [
    'productLotSublot',
    'productLot',
    'lotSublot',
    'lot',
    'productSerialNumber',
    'serialNumber',
    'productLocation',
    'location',
    'productIdentifier1',
    'identifier1',
    'productLicensePlateNumber',
    'licensePlateNumber',
];
const _partialListOptionsKey = [
    ['product'],
    ['location'],
    ['lot'],
    ['serialNumber', 'serial number'],
    ['identifier1'],
    ['identifier2'],
    ['licensePlateNumber', 'license plate number'],
];
const _fullListOptionsKey = [..._partialListOptionsKey, ['sublot']];
;
function calculateEndingSerialNumber(startingSerialNumber, quantity) {
    return startingSerialNumber.replace(/\d+$/, match => {
        const endingNumber = (typesLib.sub(typesLib.add(Number(match), quantity), 1)).toString();
        const lengthDiff = typesLib.max(typesLib.sub(endingNumber.length, match.length), 0);
        return endingNumber.padStart(typesLib.add(match.length, lengthDiff), '0');
    });
}
exports.calculateEndingSerialNumber = calculateEndingSerialNumber;
async function readStockIdFromSerialNumber(pageInstance) {
    const filterSerialNumber = { _and: [{}] };
    const page = pageInstance;
    filterSerialNumber._and.push({ stockSite: { code: page._stockSite } });
    if (page['product']?.value?.code) {
        filterSerialNumber._and.push({ product: { code: page['product'].value.code } });
    }
    if (page['serialNumber']?.value?.code) {
        filterSerialNumber._and.push({ code: page['serialNumber'].value.code });
    }
    return (0, xtrem_client_1.extractEdges)(await pageInstance.$.graph
        .node('@sage/x3-stock-data/SerialNumber')
        .query(ui.queryUtils.edgesSelector({ stockId: true }, {
        filter: filterSerialNumber,
        first: 1000,
    }))
        .execute())
        .filter(row => typesLib.strictNe(row.stockId, '0'))
        .map(row => row.stockId);
}
exports.readStockIdFromSerialNumber = readStockIdFromSerialNumber;
async function onChangeFilterStock(pageInstance, filter) {
    const page = pageInstance;
    const stockIds = await readStockIdFromSerialNumber(page);
    let stockFilter = filter;
    if (Number(stockIds?.length)) {
        if (page['product']?.value?.code) {
            if (typesLib.strictEq(page['product']?.value?.serialNumberManagementMode, 'globalReceivedIssued') &&
                page['serialNumber']?.value?.code) {
                stockFilter = {
                    ...stockFilter,
                    serialNumber: undefined,
                    stockId: { _in: stockIds },
                };
            }
        }
        else if (page['serialNumber']?.value?.code) {
            stockFilter = {
                ...stockFilter,
                serialNumber: undefined,
                stockId: { _in: stockIds },
            };
        }
    }
    return stockFilter;
}
exports.onChangeFilterStock = onChangeFilterStock;
async function onRemoveLpnFilter(pageInstance, filter) {
    const page = pageInstance;
    let stockFilter = filter;
    const transaction = page._getSavedInputs().selectedTransaction;
    if (transaction.isLocationChange) {
        stockFilter = {
            ...stockFilter,
            licensePlateNumber: '',
        };
    }
    return stockFilter;
}
exports.onRemoveLpnFilter = onRemoveLpnFilter;
async function onChangeFilterLicensePlateNumber(pageInstance, filter) {
    const page = pageInstance;
    const transaction = page._getSavedInputs().selectedTransaction;
    let stockFilter = filter;
    if (transaction.isLocationChange) {
        stockFilter = {
            ...stockFilter,
            licensePlateNumber: '',
        };
    }
    return stockFilter;
}
exports.onChangeFilterLicensePlateNumber = onChangeFilterLicensePlateNumber;
function checkIdentifierField(field, identifierFields) {
    const value = identifierFields
        ?.split(/(?=[A-Z])/)
        .join(' ')
        .toLowerCase() ?? undefined;
    return !!value && typesLib.strictNe(value.indexOf(field.toLowerCase()), -1);
}
exports.checkIdentifierField = checkIdentifierField;
async function initFieldsToBeVisible(pageInstance, identifierFields) {
    const page = pageInstance;
    const values = new Map();
    page._selectedIdentifierValues.split(',').forEach((element) => {
        const array = element.split(':');
        values.set(array[0], array[1]);
    });
    _fullListOptionsKey.forEach(([_field, _value]) => {
        if (checkIdentifierField(_value ?? _field, identifierFields)) {
            page[_field].isHidden = false;
            page[_field].isMandatory = true;
            if (values.has(_field)) {
                page[_field].value = {
                    code: values.get(_field),
                };
            }
        }
    });
}
exports.initFieldsToBeVisible = initFieldsToBeVisible;
function getIdentifierValues(pageInstance, identifierFields) {
    const page = pageInstance;
    const arrayValues = [];
    [
        ['product'],
        ['location'],
        ['serialNumber', 'serial number'],
        ['licensePlateNumber', 'license plate number'],
    ].forEach(([_field, _value]) => {
        if (checkIdentifierField(_value ?? _field, identifierFields) && page[_field]?.value?.code) {
            arrayValues.push(String(page[_field].value.code));
        }
    });
    [['sublot'], ['identifier1'], ['identifier2']].forEach(([_field, _value]) => {
        if (checkIdentifierField(_field, identifierFields) && page[_field]?.value) {
            arrayValues.push(String(page[_field].value));
        }
    });
    if (checkIdentifierField('lot', identifierFields) && page['lot']?.value?.lot) {
        arrayValues.push(String(page['lot']?.value.lot));
    }
    return arrayValues;
}
exports.getIdentifierValues = getIdentifierValues;
function getIdentifierFieldsCount(identifierFields) {
    let fieldsCount = 0;
    _partialListOptionsKey.forEach(([_field, _value]) => {
        if (checkIdentifierField(_value ?? _field, identifierFields)) {
            var _v;
            _v = fieldsCount, fieldsCount = typesLib.add(fieldsCount, 1), _v;
        }
    });
    return fieldsCount;
}
exports.getIdentifierFieldsCount = getIdentifierFieldsCount;
function isProductGlobalReceivedIssuedInStock(pageInstance) {
    const page = pageInstance;
    const stock = page['stock'];
    return stock.value.some(_ => typesLib.strictEq(_?.product?.product?.serialNumberManagementMode, 'globalReceivedIssued'));
}
exports.isProductGlobalReceivedIssuedInStock = isProductGlobalReceivedIssuedInStock;
function addValueToSelectedIdentifier(pageInstance, field, value) {
    const page = pageInstance;
    const values = page._selectedIdentifierValues?.split(',');
    values?.forEach((element, index) => {
        if (element.includes(field)) {
            values.splice(index, 1);
        }
    });
    page._selectedIdentifierValues = values?.join(',');
    page._selectedIdentifierValues = [page._selectedIdentifierValues, `${field}:${value}`].filter(Boolean).join(',');
}
exports.addValueToSelectedIdentifier = addValueToSelectedIdentifier;
function disableButton(pageInstance) {
    const page = pageInstance;
    page.nextButton.isDisabled = true;
    page.searchButton.isDisabled = true;
    page.selectAllSwitch.isDisabled = true;
}
exports.disableButton = disableButton;
async function getStockResults(pageInstance, filter, maxResult = 500) {
    const page = pageInstance;
    return (0, xtrem_client_1.extractEdges)(await page.$.graph
        .node('@sage/x3-stock-data/Stock')
        .query(ui.queryUtils.edgesSelector({
        owner: true,
        qualityAnalysisRequestId: true,
        allocatedQuantity: true,
        quantityInStockUnit: true,
        lotReference: {
            majorVersion: {
                _id: true,
                code: true,
            },
            lotCustomField2: true,
            lotCustomField1: true,
            useByDate: true,
            expirationDate: true,
        },
        stockId: true,
        stockCustomField2: true,
        stockCustomField1: true,
        identifier2: true,
        identifier1: true,
        packingUnitToStockUnitConversionFactor: true,
        packingUnit: {
            code: true,
            numberOfDecimals: true,
        },
        status: { code: true },
        serialNumber: true,
        sublot: true,
        lot: true,
        location: {
            _id: true,
            code: true,
            category: true,
        },
        licensePlateNumber: { code: true },
        product: {
            product: {
                _id: true,
                code: true,
                stockUnit: {
                    code: true,
                    numberOfDecimals: true,
                },
                serialNumberManagementMode: true,
                lotManagementMode: true,
                description1: true,
                localizedDescription1: true,
                expirationManagementMode: true,
                productCategory: { code: true },
                productSites: {
                    query: {
                        edges: {
                            node: {
                                stockSite: {
                                    code: true
                                },
                            },
                        },
                    },
                },
            },
        },
        quantityInPackingUnit: true,
        _id: true,
        availableQuantity: true,
        isBeingCounted: true,
    }, {
        filter,
        first: typesLib.max(maxResult, 500),
    }))
        .execute());
}
exports.getStockResults = getStockResults;
function isStockJournalInRecord(record, line) {
    return (typesLib.strictEq(line.packingUnit, record?.packingUnit?.code) && typesLib.strictEq(line.packingUnitToStockUnitConversionFactor, record?.packingUnitToStockUnitConversionFactor) && typesLib.strictEq(line.location, record?.location?.code) && typesLib.strictEq(line.licensePlateNumber, record?.licensePlateNumber?.code) && typesLib.strictEq(line.lot, record?.lot) && typesLib.strictEq(line.status, record?.status?.code) && typesLib.strictEq(line.sublot, record?.sublot) &&
        (typesLib.strictEq(line.serialNumber, record?.serialNumber) || typesLib.strictEq(record?.product?.product?.serialNumberManagementMode, 'globalReceivedIssued')) && typesLib.strictEq(line.identifier1, record?.identifier1) && typesLib.strictEq(line.identifier2, record?.identifier2) && typesLib.strictEq(line.stockCustomField1, record?.stockCustomField1) && typesLib.strictEq(line.stockCustomField2, record?.stockCustomField2));
}
exports.isStockJournalInRecord = isStockJournalInRecord;
//# sourceMappingURL=stock-change-by-identifier-details-control.js.map