"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.useByDateDefaultValue = exports.expirationDateDefaultValue = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const xtrem_date_time_1 = require("@sage/xtrem-date-time");
/**
 * Return the default value corresponding to the product rule for the expiry date
 * @param product
 * @param effectiveDate
 * @return
 */
function expirationDateDefaultValue(productExpirationManagement, productExpirationLeadTime, productExpirationLeadTimeUnit, effectiveDate) {
    let _tmpDate = effectiveDate ? xtrem_date_time_1.date.parse(effectiveDate) : null;
    let expiryDate;
    if (typesLib.strictEq(_tmpDate, null))
        return null;
    if (typesLib.strictNe(productExpirationManagement, null)) {
        // Check if the product is not managed in expiration mode
        if (typesLib.strictEq(productExpirationManagement, 'notManaged') || typesLib.strictEq(productExpirationManagement, 'manualEntry')) {
            // returns 2999-12-31
            //expiryDate = date.make(2999, 12, 31);
            return null;
        }
        else {
            expiryDate = null;
            if (typesLib.strictEq(productExpirationLeadTimeUnit, 'calendarDays')) {
                // the expiry date is effectiveDate + expirationLeadTime duration value in days
                expiryDate = _tmpDate.addDays(productExpirationLeadTime);
            }
            // Check if expiration time unit is months
            if (typesLib.strictEq(productExpirationLeadTimeUnit, 'month')) {
                // the expiry date is effectiveDate + expirationLeadTime in months
                expiryDate = _tmpDate.addMonths(productExpirationLeadTime);
            }
            if (typesLib.strictEq(productExpirationManagement, 'roundingMonthEnd') || typesLib.strictEq(productExpirationManagement, 'roundingBeginningMonth1')) {
                // Check if the product expiration mode indicate a rounding rule
                if (typesLib.strictEq(productExpirationManagement, 'roundingMonthEnd')) {
                    // if the rounding rule is month end
                    expiryDate = _tmpDate.endOfMonth();
                }
                else if (typesLib.strictEq(productExpirationManagement, 'roundingBeginningMonth1')) {
                    // if the rounding rule is next month begin
                    expiryDate = _tmpDate.begOfMonth().addMonths(1);
                }
            }
        }
    }
    else {
        return null;
    }
    // if none of the above conditions are met:
    return expiryDate ? expiryDate.format('YYYY-MM-DD') : null;
}
exports.expirationDateDefaultValue = expirationDateDefaultValue;
function useByDateDefaultValue(expirationDate, referenceDate, productExpirationManagement, useByDateCoefficient) {
    let _tmpExpirationDate = expirationDate ? xtrem_date_time_1.date.parse(expirationDate) : null;
    let _tmpReferenceDate = referenceDate ? xtrem_date_time_1.date.parse(referenceDate) : null;
    let useByDate;
    if (typesLib.strictEq(_tmpExpirationDate, null) && typesLib.strictEq(_tmpReferenceDate, null))
        return null;
    if (typesLib.strictNe(productExpirationManagement, null)) {
        // Check if the product is not managed in expiration mode
        if (typesLib.strictEq(productExpirationManagement, 'notManaged') || typesLib.strictEq(productExpirationManagement, 'manualEntry')) {
            return null;
        }
        if (typesLib.gt(useByDateCoefficient, 1) || typesLib.strictEq(useByDateCoefficient, 0) || typesLib.lt(useByDateCoefficient, 0)) {
            useByDateCoefficient = 1;
        }
        if (_tmpExpirationDate) {
            if (_tmpReferenceDate && typesLib.gt(_tmpExpirationDate, _tmpReferenceDate)) {
                const delay = typesLib.round(typesLib.mul(useByDateCoefficient, _tmpExpirationDate.daysDiff(_tmpReferenceDate)));
                useByDate = _tmpReferenceDate.addDays(delay);
            }
            else {
                useByDate = _tmpExpirationDate;
            }
        }
        else {
            return null;
        }
    }
    else {
        return null;
    }
    return useByDate.format('YYYY-MM-DD');
}
exports.useByDateDefaultValue = useByDateDefaultValue;
//# sourceMappingURL=defaultValue.js.map