"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.controlLotReceipt = exports.renumberStockCountList = exports.validateWithDetails = exports.validatePage = exports.validate = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const xtrem_client_1 = require("@sage/xtrem-client");
const ui = require("@sage/xtrem-ui");
function validate(pageInstance, isPageValid, fieldsData) {
    const fieldErrors = new Array();
    let areAllFieldsValid = true;
    fieldsData.forEach(field => {
        if (!_validateField(field)) {
            areAllFieldsValid = false;
            fieldErrors.push(field.fieldName ?? '');
        }
    });
    if (typesLib.gte(fieldErrors.length, 1))
        _notifyFieldNotValid(pageInstance, fieldErrors);
    return isPageValid && areAllFieldsValid;
}
exports.validate = validate;
function _validateField(data) {
    return !data.validate ? true : false;
}
function _notifyFieldNotValid(pageInstance, fields) {
    pageInstance.$.removeToasts();
    pageInstance.$.showToast(ui.localize('@sage/x3-stock/pages__utils__notification__invalid_inputs_error', `Check your entry for {{#each fieldNames}}\n - {{this}}{{/each}}`, { fieldNames: fields.map(field => pageInstance[field].title) }), { type: 'error', timeout: 5000 });
}
async function validatePage(pageInstance) {
    const errors = await pageInstance.$.page.validate();
    if (typesLib.strictEq(errors.length, 0)) {
        return true;
    }
    pageInstance.$.removeToasts();
    pageInstance.$.showToast(`${ui.localize('@sage/x3-stock/notification-validation-error', 'Error')}: ${errors[0]}`, {
        type: 'error',
        timeout: 30000,
    });
    return false;
}
exports.validatePage = validatePage;
async function validateWithDetails(pageInstance) {
    // to handle such edgy cases as user clearing an input from a field and then directly clicking on a button without blurring that field
    await pageInstance.$.commitValueAndPropertyChanges();
    const errors = await pageInstance.$.page.validateWithDetails();
    if (typesLib.strictEq(errors.length, 0)) {
        return true;
    }
    pageInstance.$.removeToasts();
    pageInstance.$.showToast(ui.localize('@sage/x3-stock/pages__utils__notification__invalid_inputs_error', `Check your entry for {{#each fieldNames}}\n - {{this}}{{/each}}`, {
        // TODO: Issue: Perhaps a better way in Typescript to dynamically retrieve a page's components rather than to cast it to 'any' type
        fieldNames: errors.map((error) => `${pageInstance[error.elementId]?.title ?? error.elementId}`),
    }), { type: 'error', timeout: 5000 });
    return false;
}
exports.validateWithDetails = validateWithDetails;
// Decision made to not present an error to the user if one occurs during renumbering
async function renumberStockCountList(stockCountSessionNumber, stockCountListNumber, page) {
    try {
        const _renumberListArgs = {
            stockCountSessionNumber: stockCountSessionNumber,
            stockCountListNumber: stockCountListNumber,
        };
        await page.$.graph
            .node('@sage/x3-stock/StockCountListDetail')
            .mutations.renumberCountList({
            stockCountSessionNumber: true,
            stockCountListNumber: true,
        }, {
            parameters: _renumberListArgs,
        })
            .execute();
    }
    catch (e) {
        ui.console.error(`renumberCountList :\n${JSON.stringify(e)}`);
    }
}
exports.renumberStockCountList = renumberStockCountList;
async function controlLotReceipt(pageInstance, lot, product, entryType, site) {
    if (!lot || !product || !entryType)
        throw new Error('Invalid arguments');
    let stockJournalFilter = {
        stockSite: site,
        documentType: entryType,
        product: { code: product },
        isUpdated: true,
        lot: lot,
    };
    const response = (0, xtrem_client_1.extractEdges)(await pageInstance.$.graph
        .node('@sage/x3-stock/StockJournal')
        .query(ui.queryUtils.edgesSelector({
        lot: true,
        sublot: true,
    }, {
        filter: stockJournalFilter,
    }))
        .execute());
    if (typesLib.gt(response.length, 0)) {
        pageInstance.$.showToast(ui.localize('@sage/x3-stock/notification-error-receipt-lot', 'The lot number {{ lot }} already exists for this product.', { lot: lot }), { type: 'error' });
        return false;
    }
    return true;
}
exports.controlLotReceipt = controlLotReceipt;
//# sourceMappingURL=control.js.map