"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const fs = require("fs");
const path = require("path");
const lib_1 = require("../lib");
const Module = require('module');
const zlib = require('zlib');
async function testCompile(args, outputExt, testList = false) {
    const filename = path.join(__dirname, '..', 'build', 'test', 'fixtures', 'test-script.js');
    const output = path.join(__dirname, '..', 'build', 'test', 'fixtures', `test-script.${outputExt}`);
    if (fs.existsSync(output)) {
        fs.unlinkSync(output);
    }
    let getPreservedRanges = (originalSource) => {
        const startPosition = originalSource.search(/.*getDirectoriesFromPath = \(p\)/);
        const endPosition = originalSource.indexOf(';', startPosition);
        return [[startPosition, endPosition]];
    };
    if (!testList) {
        getPreservedRanges = undefined;
    }
    const compiled = await (0, lib_1.compileFile)({ ...args, filename, output, getPreservedRanges });
    chai_1.assert.equal(compiled, output);
    // eslint-disable-next-line import/no-dynamic-require,global-require
    const required = require(compiled);
    (0, chai_1.assert)(Object.keys(required).includes('getDirectoriesFromPath'));
    if (testList) {
        let preservedRangeList = [];
        let preservedString = '';
        const scriptCode = Module.wrap(fs.readFileSync(filename, 'utf-8').replace(/^#!.*/, ''));
        preservedRangeList = getPreservedRanges(scriptCode);
        preservedString = scriptCode.substring(preservedRangeList[0][0], preservedRangeList[0][1] + 1);
        const outputCode = zlib.inflateSync(fs.readFileSync(output), (err) => {
            if (err) {
                // eslint-disable-next-line no-console
                console.error('An error occurred:', err);
                process.exitCode = 1;
            }
        });
        (0, chai_1.assert)(outputCode.indexOf(preservedString) >= 0, `${preservedString} not found in ${output}`);
    }
}
describe('Bytenode', () => {
    it('Compile - normal jsb', async () => {
        await testCompile({ compileAsModule: true }, 'jsb');
    });
    it('Compile - with compression jsbz', async () => {
        await testCompile({
            compileAsModule: true,
            compress: true,
        }, 'jsbz');
    });
    it('Compile - with compression jsbz preservedRangeList', async () => {
        await testCompile({
            compileAsModule: true,
            compress: true,
        }, 'jsbz', true);
    });
});
//# sourceMappingURL=bytenode-test.js.map