"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildMissingNullUnionError = void 0;
/**
 * @fileoverview Check property decorators
 * @author Sage
 */
const utils_1 = require("@typescript-eslint/utils");
function buildMissingNullUnionError(propertyName) {
    return `The property ${propertyName} is declared as nullable: its return type should be xxx | null.`;
}
exports.buildMissingNullUnionError = buildMissingNullUnionError;
const rule = {
    defaultOptions: [],
    meta: {
        type: 'problem',
        docs: {
            url: '',
            description: 'Check nullable property',
            recommended: 'recommended',
        },
        fixable: 'code',
        schema: [
        // fill in your schema
        ],
        messages: {
            error: '{{error}}',
        },
    },
    create(context) {
        // Retrieve all the property decorators
        const decoratorsSelector = 'PropertyDefinition > Decorator > CallExpression > ObjectExpression';
        return {
            // Rule: a nullable property must have a return type that looks like xxx | null
            [`${decoratorsSelector} > Property[key.name="isNullable"][value.raw="true"]`](property) {
                // Walking up the selector to get the PropertyDefinition
                const classProperty = property.parent?.parent?.parent?.parent;
                if (classProperty.key.type !== 'Identifier')
                    return;
                let typeAnnotation = classProperty.typeAnnotation?.typeAnnotation;
                // unwrap Promise<...> and Reference<...>
                if (typeAnnotation.type === 'TSTypeReference' &&
                    typeAnnotation.typeName.type === 'Identifier' &&
                    /^(Promise|Reference)$/.test(typeAnnotation.typeName.name)) {
                    typeAnnotation = typeAnnotation.typeParameters?.params[0];
                }
                if (typeAnnotation.type === utils_1.AST_NODE_TYPES.TSUnionType &&
                    typeAnnotation.types[1]?.type === utils_1.AST_NODE_TYPES.TSNullKeyword) {
                    // That's ok
                    return;
                }
                context.report({
                    node: classProperty, // Put the error on the property name
                    messageId: 'error',
                    data: {
                        error: buildMissingNullUnionError(classProperty.key.name),
                    },
                    fix: fixer => 
                    // fnToTest.parent.key is the name of the function
                    fixer.insertTextAfter(typeAnnotation, ' | null'),
                });
            },
        };
    },
};
exports.default = rule;
//# sourceMappingURL=check-nullable-property.js.map