"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.runScenario = void 0;
const xtrem_core_1 = require("@sage/xtrem-core");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const chai_1 = require("chai");
const fs = require("fs");
const glob = require("glob");
const graphql_1 = require("graphql");
const js_yaml_1 = require("js-yaml");
const lodash_1 = require("lodash");
const fsp = require("path");
const mocha_1 = require("./mocha");
// This is the single entry-point used by the Mocha runner to execute all Graphql tests
// for a given application and a given pattern. These variables are given at runtime and are dynamically set
// through environment variables.
// Load config from current dir so that we get rootdir/x3/xtrem-config.yml instead of rootdir/xtrem-config.yml
// when executing x3 unit tests
xtrem_core_1.ConfigManager.load(process.cwd(), 'test');
// disable logs if config file says so.
if (xtrem_core_1.ConfigManager.current?.logs?.disabledForTests) {
    xtrem_core_1.Logger.disable();
}
const logger = xtrem_core_1.Logger.getLogger(__filename, 'graphql-tests');
// Execute setupTestSuite before any other test
before(mocha_1.setupTestSuite);
const helpMessage = 'See https://confluence.sage.com/display/XTREEM/3+GraphQL+unit+tests for a detailed documentation.';
/**
 * Sets the layers property for the passed in TestOptions from parameters passed
 * @param opts
 * @param parameters
 */
function setLayersFromParameters(options, parameters) {
    const opts = options || {};
    if (parameters.layers) {
        opts.config = opts.config || {};
        opts.config.layers = parameters.layers;
        opts.config.sql = opts.config.sql || {};
        opts.config.sql.layers = parameters.layers;
    }
    return opts;
}
function fixTestActiveServiceOptions(options) {
    if (options.testActiveServiceOptions) {
        // Service options are strings in the parameters.json file. Convert them to ServiceOption instances
        options.testActiveServiceOptions = options.testActiveServiceOptions.map(name => xtrem_core_1.Test.application.findServiceOption(name));
    }
}
/**
 * Validate that required query files have the correct structure
 * @param directory
 * @param requestGraphqlPath
 * @param responseHbsPath
 * @param requestHbsPath
 * @param responseJsonPath
 * @returns {boolean} isExpectingParameters
 */
const validateQueryFileStructure = (isResponseHbs, isRequestHbs, isResponseJson, isRequestGraphql, isExpectingParameters, directory) => {
    chai_1.assert.isFalse(!isRequestGraphql && !isRequestHbs, `Request file is missing, you need to add either 'request.graphql' or 'request.graphql.hbs' to '${directory}'. ${helpMessage}`);
    chai_1.assert.isFalse(!isResponseJson && !isResponseHbs, `Response file is missing, you need to add either 'response.json' or 'response.json.hbs' to '${directory}'. ${helpMessage}`);
    chai_1.assert.isFalse(isRequestGraphql && isRequestHbs, `Use either 'request.graphql' or 'request.graphql.hbs' but not both. Please delete one of the two from '${directory}'. ${helpMessage}`);
    chai_1.assert.isFalse(isResponseJson && isResponseHbs, `Use either 'response.json' or 'response.json.hbs' but not both. Please delete one of the two from '${directory}'. ${helpMessage}`);
    chai_1.assert.isFalse(!isExpectingParameters && (!isRequestGraphql || !isResponseJson), `Parameters file is missing: you need to add a file called 'parameters.json' or 'parameters.yml' under '${directory}'. ${helpMessage}`);
};
/**
 * Load parameters.json file and validate
 * @param parametersPath
 * @param directory
 * @returns
 */
const loadParameterFile = (parametersPath, directory) => {
    const parametersString = fs.readFileSync(parametersPath, 'utf8');
    const parameters = parametersPath.endsWith('.json') ? (0, xtrem_shared_1.safeJsonParse)(parametersString) : (0, js_yaml_1.load)(parametersString);
    chai_1.assert.exists(parameters, `'parameters.json' under '${directory}' is not a valid JSON file. ${helpMessage}`);
    return parameters;
};
/**
 * Runs a test scenario, i.e. any key defined in 'parameters.json'.
 *
 * @export
 * @param {TestScenarioInput} {
 *     parameters,
 *     scenario,
 *     applicationDir,
 *     requestHbs,
 *     schema,
 *     responseHbs,
 *     directory,
 * }
 */
async function runScenario({ parameters, scenario, requestHbs, schema, responseHbs, directory, }) {
    const { input, variables, envConfigs } = parameters[scenario];
    let options = envConfigs ?? {};
    // if mocks are provided in parameters.json then set the directory and scenario in the options object
    if (options?.mocks) {
        options.directory = directory;
        options.scenario = scenario;
    }
    chai_1.assert.isFalse(input === undefined && variables === undefined, `You need to specify either 'input' or 'variables' for test scenario '${scenario}'. ${helpMessage}`);
    chai_1.assert.isUndefined(input && variables, `Use either 'input' or 'variables' for test scenario '${scenario}' but not both. ${helpMessage}`);
    options = setLayersFromParameters(options, parameters[scenario]);
    fixTestActiveServiceOptions(options);
    options.source = 'graphql';
    let actual = {};
    if (input) {
        const { variables: requestTemplateVariables, result: request } = (0, xtrem_shared_1.handleBars)(requestHbs, input);
        logger.debug(() => request);
        /**
         * inputParameters is a key added on handlebar we don't need the properties inputParameters in the parameter.json file
         */
        if (requestTemplateVariables.includes('inputParameters')) {
            requestTemplateVariables.splice(requestTemplateVariables.findIndex(variable => variable === 'inputParameters'));
        }
        /**
         * inputData is a key added on handlebar we don't need the properties inputData in the parameter.json file
         */
        if (requestTemplateVariables.includes('inputData')) {
            requestTemplateVariables.splice(requestTemplateVariables.findIndex(variable => variable === 'inputData'));
        }
        // if input is provided then check if all the required variables are provided vs the request template
        const missingRequestVariables = (0, lodash_1.difference)(requestTemplateVariables, Object.keys((0, xtrem_shared_1.dotNotate)(input)));
        chai_1.assert.isFalse(missingRequestVariables.length > 0, `You need to add the following variables as an object to '${scenario}' under the 'input' key: ${JSON.stringify(missingRequestVariables)}. ${helpMessage}`);
        actual = await xtrem_core_1.Test.withContext(async (context) => {
            return (0, xtrem_core_1.mapGraphQlResult)((await (0, graphql_1.graphql)({ schema, source: request, contextValue: context })));
        }, options);
    }
    else {
        actual = await xtrem_core_1.Test.withContext(async (context) => {
            return (0, xtrem_core_1.mapGraphQlResult)((await (0, graphql_1.graphql)({
                schema,
                source: requestHbs,
                contextValue: context,
                variableValues: variables,
            })));
        }, options);
    }
    logger.debug(() => `Actual result of the query : \n ${JSON.stringify(actual, null, 4)}`);
    // fetch output from parameters.json
    const output = parameters[scenario].output;
    chai_1.assert.isFalse(output === undefined, `You need to specify an 'output' property for test scenario '${scenario}'. ${helpMessage}`);
    const isJsonOutput = responseHbs.indexOf('{{{output}}}') !== -1;
    // if the attribute of an output is an array, the handle bars resolution does not work well
    // so we check if the the output attribute is an array and then stringfy it like {{{output}}}
    const handleBarData = Object.keys(output).reduce((r, k) => {
        const val = output[k];
        r[k] = Array.isArray(val) ? JSON.stringify(val) : val;
        return r;
    }, {});
    const { variables: responseTemplateVariables, result: responseJson } = isJsonOutput
        ? (0, xtrem_shared_1.handleBars)(responseHbs, {
            output: JSON.stringify(output),
        })
        : (0, xtrem_shared_1.handleBars)(responseHbs, handleBarData);
    if (!isJsonOutput) {
        const missingResponseVariables = (0, lodash_1.difference)(responseTemplateVariables, Object.keys((0, xtrem_shared_1.dotNotate)(output)));
        chai_1.assert.isFalse(missingResponseVariables.length > 0, `You need to add the following variables as an object to '${scenario}' under the 'output' key: ${JSON.stringify(missingResponseVariables)}. ${helpMessage}`);
    }
    const response = (0, xtrem_shared_1.safeJsonParse)(responseJson);
    chai_1.assert.exists(response, `response file under '${directory}' could not be parsed. ${helpMessage}`);
    // compare the results of query (actual) vs the response (expected)
    chai_1.assert.deepEqual(actual, response, `Scenario '${scenario}' has failed. All errors are displayed above.`);
}
exports.runScenario = runScenario;
const resolveSingleScenario = async (isResponseHbs, isRequestHbs, isResponseJson, isRequestGraphql, { schema, directory }, requestGraphqlPath, responseJsonPath, parameters, scenario) => {
    validateQueryFileStructure(isResponseHbs, isRequestHbs, isResponseJson, isRequestGraphql, !!parameters, directory);
    if (isResponseHbs && isRequestHbs) {
        const requestHbs = fs.readFileSync(requestGraphqlPath, 'utf8');
        chai_1.assert.isNotEmpty(requestHbs, `'request.graphql.hbs' under '${directory}' is empty. ${helpMessage}`);
        const responseHbs = fs.readFileSync(responseJsonPath, 'utf8');
        chai_1.assert.isNotEmpty(responseHbs, `'response.json.hbs' file is empty. ${helpMessage}`);
        await runScenario({
            parameters: parameters,
            scenario: scenario || directory,
            requestHbs,
            schema,
            responseHbs,
            directory,
        });
        return;
    }
    let options = { source: 'graphql' };
    if (parameters) {
        const { input, variables, output } = parameters;
        if (!isRequestHbs) {
            chai_1.assert.isFalse(input !== undefined, `'input' property is only for handlebars tests (with request.graphql.hbs and response.json.hbs). ${helpMessage}`);
            chai_1.assert.isFalse(output !== undefined, `'output' property is only for handlebars tests (with request.graphql.hbs and response.json.hbs). ${helpMessage}`);
            chai_1.assert.isFalse(variables !== undefined, `'variables' property is only for handlebars tests (with request.graphql.hbs and response.json.hbs). ${helpMessage}`);
        }
        options = parameters.envConfigs ?? {};
        if (options?.mocks)
            options.directory = directory;
        options = setLayersFromParameters(options, parameters);
        fixTestActiveServiceOptions(options);
    }
    const requestGraphql = fs.readFileSync(requestGraphqlPath, 'utf8');
    chai_1.assert.isNotEmpty(requestGraphql, `'request.graphql' under '${directory}' is empty. ${helpMessage}`);
    const responseJson = fs.readFileSync(responseJsonPath, 'utf8');
    chai_1.assert.isNotEmpty(responseJson, `'response.json' under '${directory}' is empty. ${helpMessage}`);
    const response = (0, xtrem_shared_1.safeJsonParse)(responseJson);
    chai_1.assert.exists(response, `'response.json' under '${directory}' is not a valid JSON file. ${helpMessage}`);
    logger.debug(() => requestGraphql);
    const actual = await xtrem_core_1.Test.withContext(async (context) => {
        return (0, xtrem_core_1.mapGraphQlResult)((await (0, graphql_1.graphql)({ schema, source: requestGraphql, contextValue: context })));
    }, options);
    logger.debug(() => `Actual result of the query : \n ${JSON.stringify(actual, null, 4)}`);
    // as this ia single request in the it the resetting of the sql pool is handled in the before hook
    // in mocha.ts
    chai_1.assert.deepEqual(actual, response, `Test scenario '${directory}' has failed.`);
};
const runGraphqlTest = (directory) => {
    const requestGraphqlPath = fsp.join(directory, 'request.graphql');
    const responseJsonPath = fsp.join(directory, 'response.json');
    const requestHbsPath = fsp.join(directory, 'request.graphql.hbs');
    const responseHbsPath = fsp.join(directory, 'response.json.hbs');
    const parametersJsonPath = fsp.join(directory, 'parameters.json');
    const parametersYmlPath = fsp.join(directory, 'parameters.yml');
    const isExpectingJsonParameters = fs.existsSync(parametersJsonPath);
    const isExpectingYmlParameters = fs.existsSync(parametersYmlPath);
    const isExpectingParameters = isExpectingJsonParameters || isExpectingYmlParameters;
    const parametersPath = isExpectingJsonParameters ? parametersJsonPath : parametersYmlPath;
    const isResponseJson = fs.existsSync(responseJsonPath);
    const isRequestGraphql = fs.existsSync(requestGraphqlPath);
    const isResponseHbs = fs.existsSync(responseHbsPath);
    const isRequestHbs = fs.existsSync(requestHbsPath);
    if (!isExpectingParameters) {
        // don't expect parameter just normal request and response filed
        // case of without handlebars and without parameters.json
        it('default test scenario', async () => {
            chai_1.assert.exists(xtrem_core_1.Test.application, "Couldn't find a valid application object.");
            if (!schema)
                schema = await xtrem_core_1.Test.application.getGraphQLSchema();
            await resolveSingleScenario(isResponseHbs, isRequestHbs, isResponseJson, isRequestGraphql, { schema, directory }, requestGraphqlPath, responseJsonPath);
        });
    }
    else {
        // parameters provided, so we expect to run 1 to many scenarios
        let parameters = {};
        if (isResponseHbs && isRequestHbs) {
            parameters = loadParameterFile(parametersPath, directory);
        }
        else {
            parameters = loadParameterFile(parametersPath, directory);
        }
        const getItOverride = (mode) => {
            switch (mode) {
                case 'only':
                    return it.only;
                case 'skip':
                    return it.skip;
                default:
                    return it;
            }
        };
        if (isResponseHbs && isRequestHbs) {
            // handlebars case
            Object.keys(parameters).forEach(scenario => {
                const overriddenIt = getItOverride(parameters[scenario].executionMode);
                overriddenIt(scenario, async () => {
                    if (!schema)
                        schema = await xtrem_core_1.Test.application.getGraphQLSchema();
                    await resolveSingleScenario(isResponseHbs, isRequestHbs, isResponseJson, isRequestGraphql, { schema, directory }, requestHbsPath, responseHbsPath, parameters, scenario);
                });
            });
        }
        else {
            // normal case with env variables and layers
            const overriddenIt = getItOverride(parameters.executionMode);
            overriddenIt('default test scenario', async () => {
                if (!schema)
                    schema = await xtrem_core_1.Test.application.getGraphQLSchema();
                await resolveSingleScenario(isResponseHbs, isRequestHbs, isResponseJson, isRequestGraphql, { schema, directory }, requestGraphqlPath, responseJsonPath, parameters);
            });
        }
    }
};
let schema;
/**
 * Initializes and runs the matching graphQL tests
 * @param {string} applicationDir
 * @param [string] pattern
 */
const describeGraphqlTests = (applicationDir, pattern = '') => {
    const graphqlDir = fsp.join(applicationDir, 'test/graphql');
    if (fs.existsSync(graphqlDir)) {
        describe('Graphql tests', () => {
            const directories = glob
                .sync(`./test/graphql/${pattern}*/`, {
                cwd: applicationDir,
                absolute: true,
                realpath: true,
            })
                .sort(xtrem_shared_1.defaultStringCompare);
            directories.forEach(directory => {
                // extract the scenario name from the directory
                const message = directory.substring(applicationDir.length + 1);
                describe(message, () => {
                    runGraphqlTest(directory);
                });
            });
        });
    }
    else {
        // eslint-disable-next-line no-console
        console.warn(`${graphqlDir} is missing!`);
    }
};
describeGraphqlTests(process.env.APPLICATION_DIR ?? process.cwd(), process.env.GRAPHQL_TEST_PATTERN);
//# sourceMappingURL=graphql-tests-entry-point.js.map