"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthyEventLoop = void 0;
/**
 * This class can be used in async loops to keep the event loop healthy
 *
 * @example
 *  const healthyEventLoop = new HealthyEventLoop(10000)
 *
 *  then, in loops
 *      if (healthyEventLoop.shouldYield) {
 *          await healthyEventLoop.yield();
 *      }
 */
class HealthyEventLoop {
    #yieldCount = 0;
    #tick = 0;
    #lastYieldTime = Date.now();
    constructor(yieldModulo = 10000) {
        this.yieldModulo = yieldModulo;
    }
    getYieldCount() {
        return this.#yieldCount;
    }
    /**
     * Should the yield() function be called ?
     */
    get shouldYield() {
        this.#tick += 1;
        return this.#tick % this.yieldModulo === 0;
    }
    /**
     * Yield with setImmediate to keep event loop healthy during any array iterations
     */
    async yield() {
        const now = Date.now();
        // Do not yield too often when the body of the loop is very fast
        if (now - this.#lastYieldTime < 100)
            return;
        // Yield with setImmediate, to give a turn to all higher priority events
        // See https://nodejs.org/en/docs/guides/event-loop-timers-and-nexttick
        await new Promise(setImmediate);
        this.#yieldCount += 1;
        this.#lastYieldTime = now;
    }
}
exports.HealthyEventLoop = HealthyEventLoop;
//# sourceMappingURL=healthy-event-loop.js.map