"use strict";
/// Low level functions used by compile-server.ts
///
/// This code was borrowed almost verbatim from xtrem-cli/lib/command/compile/server-compilation.ts
/// Code duplication is not an issue because this is throw away code.
/// It will go away once all the packages have been converted to async/await.
Object.defineProperty(exports, "__esModule", { value: true });
exports.printCompilerError = exports.getCompilerOptions = exports.getFiles = exports.getTsProgram = void 0;
const fs = require("fs");
const path = require("path");
const ts = require("typescript");
const tsConfigFilename = 'tsconfig.json';
function getAccessibleFileSystemEntries(p) {
    try {
        const entries = fs.readdirSync(p || '.').sort();
        const files = [];
        const directories = [];
        entries.forEach(entry => {
            // This is necessary because on some file system node fails to exclude
            // "." and "..". See https://github.com/nodejs/node/issues/4002
            if (entry === '.' || entry === '..') {
                return;
            }
            const name = ts.combinePaths(p, entry);
            let stat;
            try {
                stat = fs.statSync(name);
            }
            catch (e) {
                return;
            }
            if (stat.isFile()) {
                files.push(entry);
            }
            else if (stat.isDirectory()) {
                directories.push(entry);
            }
        });
        return { files, directories };
    }
    catch (e) {
        return { files: [], directories: [] };
    }
}
const getTsProgram = (dir, options) => {
    const files = (0, exports.getFiles)(dir, options);
    return ts.createProgram(files, (0, exports.getCompilerOptions)(dir, options?.configFileName || tsConfigFilename, options?.tsconfigOverride));
};
exports.getTsProgram = getTsProgram;
const getFiles = (dir, options) => {
    const tsconfigPath = path.resolve(dir, options?.configFileName || tsConfigFilename);
    const { include: includes, exclude: excludes, extends: extensions, } = JSON.parse(fs.readFileSync(tsconfigPath, 'utf-8'));
    /*
    See https://github.com/microsoft/TypeScript/blob/23f1d5ccb7202a1eac0a853abfdaf060b145a4b6/src/compiler/utilities.ts#L8085
    The "matchFiles" function has the following signature:
    (path: string, extensions: ReadonlyArray<string> | undefined, excludes: ReadonlyArray<string> | undefined, includes: ReadonlyArray<string> | undefined, useCaseSensitiveFileNames: boolean, currentDirectory: string, depth: number | undefined, getFileSystemEntries: (path: string) => FileSystemEntries, realpath: (path: string) => string): string[]
    and is useful in order to get the list of files to be compiled from the "include" and "exclude" properties.
    We need a complete list of files in order to support the "composite" feature and compile project references.
    */
    return ts.matchFiles(dir, extensions, options?.excludeOverride || excludes, options?.includeOverride || includes.filter((i) => !i.endsWith('.json')), true, process.cwd(), undefined, getAccessibleFileSystemEntries, ts.sys.realpath);
};
exports.getFiles = getFiles;
const getCompilerOptionsFromTsConfig = (tsconfigPath, basePath) => {
    const configFile = ts.readConfigFile(tsconfigPath, ts.sys.readFile);
    const tsConfig = ts.parseJsonConfigFileContent(configFile.config, ts.sys, basePath || path.dirname(tsconfigPath));
    tsConfig.options.configFilePath = tsconfigPath;
    return tsConfig;
};
const getCompilerOptions = (dir, configFileNme = tsConfigFilename, tsconfigOverride) => {
    const tsconfigPath = path.resolve(dir, configFileNme);
    const buildDir = path.resolve(dir, 'build');
    let compilerOptions = {};
    try {
        // The following function not only reads the actual "tsconfig.json" files
        // but also takes care of possible extensions.
        compilerOptions = getCompilerOptionsFromTsConfig(tsconfigPath).options;
        // eslint-disable-next-line no-empty
    }
    catch (err) { }
    const mandatoryOptions = {
        module: ts.ModuleKind.CommonJS,
        moduleResolution: ts.ModuleResolutionKind.NodeJs,
        target: ts.ScriptTarget.ES2022,
        outDir: buildDir,
        rootDir: path.resolve(dir),
        baseUrl: path.resolve(dir),
        experimentalDecorators: true,
        declaration: true,
        declarationMap: true,
        composite: true,
        sourceMap: true,
        stripInternal: true,
        watch: false,
        noUnusedLocals: false,
        noImplicitAny: true,
        noImplicitThis: true,
        strictNullChecks: true,
        useDefineForClassFields: false,
        skipLibCheck: true,
        ...tsconfigOverride,
    };
    return {
        ...compilerOptions,
        ...mandatoryOptions,
    };
};
exports.getCompilerOptions = getCompilerOptions;
const printCompilerError = (diagnostic) => {
    if (!diagnostic.file || !diagnostic.start) {
        console.error('BAD TYPESCRIPT COMPILER DIAGNOSTIC', diagnostic);
        return;
    }
    const { line, character } = diagnostic.file.getLineAndCharacterOfPosition(diagnostic.start);
    const message = ts.flattenDiagnosticMessageText(diagnostic.messageText, '\n').trim();
    console.error(`${diagnostic.file?.fileName} (${line + 1},${character + 1}): ${message}`);
};
exports.printCompilerError = printCompilerError;
//# sourceMappingURL=server-compilation.js.map