"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAsyncGenerator = exports.AsyncGenericReader = void 0;
const _1 = require(".");
const async_array_1 = require("./async-array");
/**
 * Generic Implementation of AsyncReader interface
 */
class AsyncGenericReader {
    constructor(options) {
        this.options = options;
    }
    #readCount = 0;
    get readCount() {
        return this.#readCount;
    }
    async read() {
        if (_1.asyncHealtyEventLoop.shouldYield) {
            await _1.asyncHealtyEventLoop.yield();
        }
        const item = await this.options.read();
        if (item !== undefined)
            this.#readCount += 1;
        return item;
    }
    // eslint-disable-next-line require-await
    async stop() {
        return this.options.stop();
    }
    async forEach(body) {
        try {
            let i = 0;
            // eslint-disable-next-line no-constant-condition
            while (true) {
                const item = await this.read();
                if (item === undefined)
                    break;
                await body(item, i);
                i += 1;
            }
        }
        finally {
            await this.stop();
        }
    }
    map(body) {
        let i = 0;
        return new AsyncGenericReader({
            read: async () => {
                const item = await this.read();
                const mapped = item !== undefined ? body(item, i) : undefined;
                i += 1;
                return mapped;
            },
            stop: () => this.stop(),
        });
    }
    filter(body) {
        return new AsyncGenericReader({
            read: async () => {
                // eslint-disable-next-line no-constant-condition
                while (true) {
                    const item = await this.read();
                    if (item === undefined)
                        return undefined;
                    if (body(item))
                        return item;
                }
            },
            stop: () => this.stop(),
        });
    }
    async reduce(body, initial) {
        try {
            let result = initial;
            // eslint-disable-next-line no-constant-condition
            while (true) {
                const item = await this.read();
                if (item === undefined)
                    return result;
                result = await body(result, item);
            }
        }
        finally {
            await this.stop();
        }
    }
    async find(body) {
        try {
            let i = 0;
            // eslint-disable-next-line no-constant-condition
            while (true) {
                const item = await this.read();
                if (item === undefined)
                    return undefined;
                if (await body(item, i))
                    return item;
                i += 1;
            }
        }
        finally {
            await this.stop();
        }
    }
    async findIndex(body) {
        try {
            let i = 0;
            // eslint-disable-next-line no-constant-condition
            while (true) {
                const item = await this.read();
                if (item === undefined)
                    return -1;
                if (await body(item, i))
                    return i;
                i += 1;
            }
        }
        finally {
            await this.stop();
        }
    }
    async some(body) {
        return (await this.find(body)) !== undefined;
    }
    async every(body) {
        return ((await this.find(async (item) => {
            return !(await body(item));
        })) === undefined);
    }
    async readAll() {
        try {
            const result = [];
            // eslint-disable-next-line no-constant-condition
            while (true) {
                const item = await this.read();
                if (item === undefined)
                    break;
                result.push(item);
            }
            return result;
        }
        finally {
            await this.stop();
        }
    }
    sort(compare) {
        return new async_array_1.AsyncArrayReader(() => this.readAll()).sort(compare);
    }
    toAsyncGenerator() {
        return createAsyncGenerator(this);
    }
}
exports.AsyncGenericReader = AsyncGenericReader;
function createAsyncGenerator(reader) {
    async function* generator() {
        // TODO: add timer to exit the loop if feed is not read to the end
        // See https://seg.phault.net/blog/2018/03/async-iterators-cancellation/
        try {
            let item;
            // eslint-disable-next-line no-cond-assign
            while ((item = await reader.read()) != null) {
                yield item;
            }
        }
        finally {
            await reader.stop();
        }
    }
    return generator();
}
exports.createAsyncGenerator = createAsyncGenerator;
//# sourceMappingURL=async-reader.js.map