"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setUnitTestMode = void 0;
const dynamoDbWrapper_1 = require("./wrapper/dynamoDbWrapper");
let dbMock;
function setUnitTestMode(mock) {
    dbMock = mock;
}
exports.setUnitTestMode = setUnitTestMode;
class InvalidatedToken {
    constructor() {
        this._invalidatedTokenList = new Set();
        this._interval = undefined;
        this._cluster = "";
        this._tableName = "";
    }
    static get Instance() {
        return this._instance || (this._instance = new this());
    }
    startListening(config) {
        if (!this._interval) {
            this._db = new dynamoDbWrapper_1.DynamoDbWrapper(config.awsConfig, dbMock);
            this._cluster = config.cluster;
            this._tableName = process.env.XTREM_INVALIDATED_TOKEN_TABLE || config.tableName;
            if (!this._tableName || !this._cluster) {
                throw new Error("Cluster or Table name is missing from configuration");
            }
            this._logger = config.logger;
            if (this._logger)
                this._logger.info(`Start loop to fetch invalidated token for cluster ${this._cluster}`);
            this._interval = setInterval(() => { this.fetchInvalidatedToken(); }, config.frequencyInSec * 1000);
        }
    }
    stopListening() {
        if (this._interval) {
            if (this._logger)
                this._logger.info(`Stop loop to fetch invalidated token for cluster ${this._cluster}`);
            clearInterval(this._interval);
            this._interval = undefined;
        }
    }
    async fetchInvalidatedToken() {
        const dbParams = {
            TableName: this._tableName,
            KeyConditionExpression: "#PK = :PK",
            ExpressionAttributeNames: {
                "#PK": "PK"
            },
            ExpressionAttributeValues: {
                ":PK": { S: this._cluster }
            }
        };
        if (dbMock) {
            // unit test mode : fetch only 2 by query to ensure loop is working correctly
            dbParams.Limit = 2;
        }
        if (this._logger)
            this._logger.verbose(() => `Fetching invalidated token for cluster ${this._cluster}`);
        try {
            const tokenList = new Set();
            do {
                const result = await this._db?.query(dbParams);
                if (result) {
                    if (result.Items) {
                        for (const item of result.Items) {
                            tokenList.add(item.SK.S);
                        }
                    }
                    dbParams.ExclusiveStartKey = result.LastEvaluatedKey;
                }
            } while (dbParams.ExclusiveStartKey);
            // override list only once fetch is over, so we don't allow an invalid token if isJTIValid is called during fetch
            this._invalidatedTokenList.clear();
            this._invalidatedTokenList = tokenList;
            if (this._logger)
                this._logger.verbose(() => `End of invalidated token Fetch for cluster ${this._cluster}`);
        }
        catch (e) {
            if (this._logger)
                this._logger.error(`Error while fetching invalidated token for cluster ${this._cluster} : ${e.message}`);
        }
    }
    isJTIValid(jti) {
        return !this._invalidatedTokenList.has(jti);
    }
}
exports.default = InvalidatedToken;
