import { camelCase, cloneDeep, groupBy, isArray, isEmpty, isObject, set } from 'lodash';
import { MultiReferenceControlObject, NodeBrowserTreeControlObject, VitalPodControlObject, } from '../component/control-objects';
import * as xtremRedux from '../redux';
import { CollectionValue } from '../service/collection-data-service';
import { nonSavableFieldType } from '../service/graphql-api';
import { localizeEnumMember } from '../service/i18n-service';
import { serializePageData } from '../service/value-serializer-service';
import { normalizeUnderscoreBind } from './abstract-fields-utils';
import { convertDeepBindToPathNotNull } from './nested-field-utils';
const isReferenceFieldValue = (fieldValue) => fieldValue && !Array.isArray(fieldValue) && typeof fieldValue === 'object' && !(fieldValue instanceof Set);
/**
 * Converts object to have unique ids using fieldname__key
 */
export function mergedValueToSplitValue(mergedValue) {
    return splitReferenceFieldsProperties(mergedValue);
}
const splitReferenceFieldsProperties = (mergedValue) => {
    if (!isReferenceFieldValue(mergedValue)) {
        return mergedValue;
    }
    const parsedValues = Object.keys(mergedValue)
        // We filter the reference fields with the following criteria
        .filter(key => !key.startsWith('__') && isReferenceFieldValue(mergedValue[key]))
        .reduce((_parsedValues, key) => {
        const fieldName = key;
        const fieldValue = mergedValue[key];
        const splitProperties = Object.keys(fieldValue)
            .filter(_key => _key !== '_id')
            .reduce((_parsedValue, _key) => ({
            ..._parsedValue,
            [`${fieldName}__${_key}`]: { [_key]: fieldValue[_key], _id: fieldValue._id },
        }), {});
        delete mergedValue[key]; // Remove the original reference field; e.g. user { name: , age: }
        return { ..._parsedValues, ...splitProperties };
    }, {});
    return { ...parsedValues, ...mergedValue };
};
/**
 * Converts the object to a normal normal value without __ (see mergedValueToSplitValue)
 */
export function splitValueToMergedValue(splitValue) {
    return mergeReferenceFieldsProperties(splitValue);
}
function mergeReferenceFieldsProperties(splitValue) {
    if (!isReferenceFieldValue(splitValue)) {
        return splitValue;
    }
    const groupedKeys = groupBy(Object.keys(splitValue), normalizeUnderscoreBind);
    const objectWithRemappedKeys = Object.keys(groupedKeys).reduce((mergedValue, groupKey) => {
        groupedKeys[groupKey].forEach(key => {
            if (mergedValue[groupKey]) {
                mergedValue[groupKey] = {
                    ...underscoreObjectCleaner(splitValue[key]),
                    ...mergedValue[groupKey],
                };
            }
            else {
                mergedValue[groupKey] = underscoreObjectCleaner(splitValue[key]);
            }
        });
        return mergedValue;
    }, {});
    return Object.keys(objectWithRemappedKeys).reduce((previousValue, key) => {
        if (key) {
            const value = isObject(objectWithRemappedKeys[key])
                ? mergeReferenceFieldsProperties(objectWithRemappedKeys[key])
                : objectWithRemappedKeys[key];
            // Empty objects should be replaced with nulls
            previousValue[key] = isObject(value) && !isArray(value) && isEmpty(value) ? null : value;
        }
        return previousValue;
    }, {});
}
function underscoreObjectCleaner(value) {
    if (isReferenceFieldValue(value)) {
        return Object.keys(value).reduce((cleanValue, key) => ({ ...cleanValue, [normalizeUnderscoreBind(key)]: value[key] }), {});
    }
    return value;
}
export const hexToRgb = (hex) => {
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
    return result
        ? {
            r: parseInt(result[1], 16),
            g: parseInt(result[2], 16),
            b: parseInt(result[3], 16),
        }
        : null;
};
export const serializeDataForTransientPage = (screenId, useBindKeys = true, isRequestingDefaults = false) => {
    const state = xtremRedux.getStore().getState();
    const screenDefinition = state.screenDefinitions[screenId];
    let screenValues = {};
    if (screenDefinition?.values && screenDefinition?.metadata?.uiComponentProperties) {
        const values = screenDefinition.values;
        const uiProperties = screenDefinition.metadata.uiComponentProperties;
        const addValue = (reduced, key, value) => {
            const keyToSet = useBindKeys ? convertDeepBindToPathNotNull(uiProperties[key].bind || key) : key;
            const target = cloneDeep(reduced);
            set(target, keyToSet, value);
            return target;
        };
        screenValues = Object.keys(values)
            .filter(k => Object.keys(uiProperties).includes(k))
            .filter(key => uiProperties[key] && values[key] !== undefined)
            .reduce((reduced, key) => {
            const value = values[key];
            const uiProps = uiProperties[key];
            if (uiProps.pluginPackage) {
                const plugin = state.plugins[uiProps.pluginPackage];
                return addValue(reduced, key, plugin.transformToGraphValue ? plugin.transformToGraphValue(value) : value);
            }
            if (value instanceof Array) {
                if (screenDefinition.metadata.controlObjects[key] instanceof MultiReferenceControlObject) {
                    const referenceIds = value
                        .map(v => v._id)
                        .filter(Boolean)
                        .map(v => String(v));
                    return addValue(reduced, key, referenceIds);
                }
                return addValue(reduced, key, value);
            }
            if (value instanceof CollectionValue) {
                if (Object.prototype.hasOwnProperty.call(uiProps, 'levels')) {
                    // nested grid
                    const changeSet = value.getChangedRecordsAsTree();
                    if (changeSet.length > 0) {
                        return addValue(reduced, key, changeSet);
                    }
                }
                else {
                    const changeSet = value.getNormalizedChangedRecords(isRequestingDefaults);
                    const cleanChangeSet = changeSet.map(element => mergeReferenceFieldsProperties(cleanMetadataFromRecord(element)));
                    if (changeSet.length > 0) {
                        return addValue(reduced, key, cleanChangeSet);
                    }
                }
                // If no items are in the changeset, we leave it as and don't push it to the server
                return reduced;
            }
            if (value instanceof Object) {
                const controlObject = screenDefinition.metadata.controlObjects[key];
                if (controlObject instanceof VitalPodControlObject) {
                    const updatedValue = { ...value };
                    return addValue(reduced, key, Object.keys(updatedValue).reduce((prevValue, prop) => {
                        if (isObject(updatedValue[prop]) && updatedValue[prop]._id) {
                            prevValue[prop] = `${updatedValue[prop]._id}`;
                        }
                        else {
                            prevValue[prop] = updatedValue[prop];
                        }
                        return prevValue;
                    }, {}));
                }
                if (controlObject instanceof NodeBrowserTreeControlObject) {
                    return addValue(reduced, key, value);
                }
                if ('data' in value && Array.isArray(value.data)) {
                    return addValue(reduced, key, value.data);
                }
                if ('node' in uiProps) {
                    // TODO: This logic above is not very bright, let's add the field type to the uiProperties
                    // Check for reference fields
                    return addValue(reduced, key, value && value._id ? `_id:${value._id}` : null);
                }
                return addValue(reduced, key, value);
            }
            return addValue(reduced, key, value);
        }, {});
    }
    return screenValues;
};
export const parseScreenValues = (screenId, useBindKeys = true, isRequestingDefaults = false, state = xtremRedux.getStore().getState()) => {
    const screenDefinition = state.screenDefinitions[screenId];
    let screenValues = {};
    if (screenDefinition?.values && screenDefinition?.metadata?.uiComponentProperties) {
        const uiComponentProperties = screenDefinition.metadata.uiComponentProperties;
        const screenProperties = uiComponentProperties[screenId];
        screenValues =
            screenProperties.node && state.nodeTypes[schemaTypeNameFromNodeName(screenProperties.node)]
                ? serializePageData({
                    nodeTypes: state.nodeTypes,
                    targetNode: schemaTypeNameFromNodeName(screenProperties.node),
                    values: screenDefinition.values,
                    uiComponentProperties,
                    isRequestingDefaults,
                })
                : serializeDataForTransientPage(screenId, useBindKeys, isRequestingDefaults);
    }
    return screenValues || {};
};
export const getScreenValues = (screenId, useBindKeys = true) => {
    const state = xtremRedux.getStore().getState();
    const screenDefinition = state.screenDefinitions[screenId];
    let screenValues = {};
    if (screenDefinition?.values && screenDefinition?.metadata?.uiComponentProperties) {
        const values = screenDefinition.values;
        const uiProperties = screenDefinition.metadata.uiComponentProperties;
        screenValues = Object.keys(values)
            .filter(key => Object.keys(uiProperties).includes(key) &&
            uiProperties[key] &&
            !uiProperties[key].isTransient &&
            uiProperties[key]._controlObjectType &&
            !nonSavableFieldType.includes(uiProperties[key]._controlObjectType) &&
            values[key] !== undefined)
            .reduce((previousValue, key) => {
            const keyToSet = useBindKeys && uiProperties[key].bind ? convertDeepBindToPathNotNull(uiProperties[key].bind) : key;
            set(previousValue, keyToSet, values[key]);
            return previousValue;
        }, {});
    }
    return screenValues;
};
export const capitalize = (s) => s[0].toUpperCase() + s.substring(1);
export const pascalCase = (s) => capitalize(camelCase(s));
export function schemaTypeNameFromNodeName(name) {
    if (!name) {
        return undefined;
    }
    const parts = String(name).split('/').map(pascalCase);
    return parts[parts.length - 1];
}
export const getArtifactDescription = (path) => {
    const pathElements = path.split('/');
    const name = pathElements[2] || 'index'; // E.g. CheckboxField (if no artifact name is provided, all the module artifacts are loaded through index.js)
    return {
        name,
        package: pathElements[1], // E.g. xtrem-show-case
        vendor: pathElements[0], // E.g. @sage
    };
};
export const cleanMetadataFromRecord = (record, removeAction = true, removeSortValue = false) => {
    if (record === undefined || record === null || typeof record !== 'object') {
        return record;
    }
    const updatedRecord = cloneDeep(record);
    Object.keys(updatedRecord).forEach(key => {
        if (key === '_sortValue' && removeSortValue) {
            delete updatedRecord[key];
            return;
        }
        if (key === '__dirtyColumns') {
            delete updatedRecord[key];
            return;
        }
        if (key === '__validationState') {
            delete updatedRecord[key];
            return;
        }
        if (key === '__uncommitted') {
            delete updatedRecord[key];
            return;
        }
        if (key.startsWith('__')) {
            if (key === '__action' && !removeAction) {
                return;
            }
            delete updatedRecord[key];
        }
    });
    delete updatedRecord.$loki;
    delete updatedRecord[''];
    return updatedRecord;
};
export const cleanMetadataAndNonPersistentIdFromRecord = (record) => {
    const updatedRecord = cleanMetadataFromRecord(record);
    if (updatedRecord._id && Number(updatedRecord._id) < 0) {
        delete updatedRecord._id;
    }
    return updatedRecord;
};
export const addOptionsAndLocalizationToProps = (state, baseProps) => {
    if (baseProps.fieldProperties.optionType) {
        const optionTypes = state.enumTypes[schemaTypeNameFromNodeName(baseProps.fieldProperties.optionType)] || [];
        const enumOptions = [...optionTypes];
        const localizedOptions = enumOptions.reduce((value, key) => {
            return { ...value, [key]: localizeEnumMember(baseProps.fieldProperties.optionType, key) };
        }, {});
        return { ...baseProps, enumOptions, localizedOptions };
    }
    return baseProps;
};
export const getNestedFieldArrayFromCardDefinition = (cardDefinition) => Object.values(cardDefinition).filter(v => !!v);
export const getNestedFieldArrayFromCardExtensionDefinition = (cardDefinition) => Object.values(cardDefinition).filter(v => !!v);
export const arrayMoveMutate = (array, from, to) => {
    const startIndex = from < 0 ? array.length + from : from;
    if (startIndex >= 0 && startIndex < array.length) {
        const endIndex = to < 0 ? array.length + to : to;
        const [item] = array.splice(from, 1);
        array.splice(endIndex, 0, item);
    }
};
export function forceReadOnlyModeOnNestedField(nestedField) {
    if (!nestedField) {
        return undefined;
    }
    return {
        ...nestedField,
        defaultUiProperties: {
            ...nestedField.defaultUiProperties,
            isReadOnly: true,
        },
        properties: {
            ...nestedField.properties,
            isReadOnly: true,
        },
    };
}
/**
 * Not a pure function, mutates the value. Deep clean is advised before calling.
 *
 * Removes nested structures that only contain keys with null values.
 * For example: { _id: 3, code: "Test", site: { country: null, site: { name: null } } } becomes just { _id: 3, code: "Test" }
 * See XT-30693 for more details.
 * @param data
 * @returns
 */
export const removeNullOnlyLeaves = (data) => {
    if (data === null) {
        return true;
    }
    if (Array.isArray(data)) {
        return false;
    }
    if (typeof data !== 'object') {
        return false;
    }
    const keys = Object.keys(data);
    const keysToRemove = keys.filter(key => {
        return removeNullOnlyLeaves(data[key]);
    });
    // If all children keys of the property are null, then we delete key.
    if (keysToRemove.length === keys.length) {
        keysToRemove.forEach(k => {
            delete data[k];
        });
        return true;
    }
    keys.forEach(k => {
        // If the object is empty after the null cleanup, we delete the object
        if (typeof data[k] === 'object' &&
            data[k] !== null &&
            !Array.isArray(data[k]) &&
            Object.keys(data[k]).length === 0) {
            delete data[k];
        }
    });
    return false;
};
//# sourceMappingURL=transformers.js.map