import { isArray, isFunction } from 'lodash';
import { RANGE, RANGE_DIVIDER, SET } from '../../component/types';
import { localize } from '../../service/i18n-service';
import { triggerFieldEvent } from '../events';
import { resolveByValue } from '../resolve-value-utils';
import { AUTO_COLUMN_ID, removeNumericPostfix } from '../table-component-utils';
import { cleanMetadataFromRecord, splitValueToMergedValue } from '../transformers';
import { COLUMN_ID_LINE_NUMBER, COLUMN_ID_ROW_ACTIONS, COLUMN_ID_ROW_SELECTION } from './ag-grid-column-config';
import { shouldDisplayColumnInPanel } from './ag-grid-service';
export const ROW_HEIGHT = 40;
export const getFilterValueFromAgGridFilterStatement = (filterBase, filterKey) => {
    const filterType = filterBase.filterType;
    const type = filterBase.type;
    const lookup1 = filterType === 'date'
        ? filterBase.dateFrom
        : filterBase.filter;
    const lookup2 = filterType === 'date'
        ? filterBase.dateTo
        : filterType === 'number'
            ? filterBase.filterTo
            : filterType === 'text'
                ? filterBase.filterTo
                : '';
    const filterValue = type === RANGE ? `${lookup1}${RANGE_DIVIDER}${lookup2}` : lookup1;
    if (filterType === SET) {
        return {
            filterValue: filterBase.values,
            filterType: {
                text: `${filterKey} in list`,
                value: SET,
            },
            filterById: filterBase.filterById,
        };
    }
    return {
        filterValue,
        filterType: { text: `${filterKey} ${type} ${filterValue}`, value: type },
    };
};
export function callGridMethod(gridApi, method, ...args) {
    if (!gridApi || gridApi?.isDestroyed()) {
        return undefined;
    }
    return gridApi[method]?.(...args);
}
export function getSafeGridApiContext(cb, gridApi, method, ...args) {
    if (!gridApi || gridApi?.isDestroyed()) {
        return undefined;
    }
    const result = gridApi[method]?.(...args);
    return cb(result);
}
export const mapAgGridFilterToXtremFilters = (filterModel) => (filterKey) => {
    const filterBase = filterModel[filterKey];
    const operator = filterBase.operator;
    const formattedFilterKey = removeNumericPostfix(filterKey);
    if (operator && isArray(filterBase.conditions)) {
        // Combined Filter
        return {
            // When the table value is reset the column names are getting an _{NUMBER} postfix, we remove that
            id: formattedFilterKey,
            value: filterBase.conditions.map((c) => getFilterValueFromAgGridFilterStatement(c, filterKey)),
        };
    }
    // Single Filter
    return {
        // When the table value is reset the column names are getting an _{NUMBER} postfix, we remove that
        id: formattedFilterKey,
        value: [getFilterValueFromAgGridFilterStatement(filterBase, filterKey)],
    };
};
export const tryToCommitPhantomRow = async ({ api, screenId, value, elementId, }) => {
    const phantomRow = api.getPinnedTopRow(0);
    if (!phantomRow) {
        return null;
    }
    const dirtyColumns = new Set(api.getColumnDefs()?.map(c => c.field));
    value.setRecordValue({
        recordData: { _id: phantomRow.data._id },
        level: phantomRow.data.__level,
        toBeMarkedAsDirty: Array.from(dirtyColumns),
        isOrganicChange: true,
    });
    const recordData = value.getRawRecord({
        id: phantomRow.data._id,
        level: phantomRow.data.__level,
        cleanMetadata: false,
    });
    const validationResult = await value.runValidationOnRecord({ recordData });
    if (Object.keys(validationResult || {}).length === 0) {
        api.flashCells({ rowNodes: [phantomRow] });
        const { __level: level, __parentId: parentId } = recordData;
        value.commitPhantomRow({
            id: recordData._id,
            level,
            parentId,
        });
        setTimeout(() => {
            value.createNewPhantomRow({ level, parentId });
            triggerFieldEvent(screenId, elementId, 'onRowAdded', recordData._id, splitValueToMergedValue(cleanMetadataFromRecord(recordData)));
        }, 500);
        return null;
    }
    return validationResult;
};
export const tabToNextCell = (screenId, elementId, value, onTelemetryEvent) => (event) => {
    const { nextCellPosition, previousCellPosition, api } = event;
    if (event.backwards === false &&
        previousCellPosition.column.getColId() === COLUMN_ID_ROW_ACTIONS &&
        previousCellPosition.rowIndex === 0 &&
        previousCellPosition.rowPinned === 'top') {
        const phantomRow = event.api.getPinnedTopRow(0);
        if (phantomRow?.data?.__dirty) {
            onTelemetryEvent?.(`tablePhantomRowCommittedByBlur-${elementId}`, {
                screenId,
                elementId,
            });
            tryToCommitPhantomRow({ api, screenId, elementId, value: value() }).then(r => {
                if (r) {
                    const firstColKey = Object.keys(r)[0];
                    api.startEditingCell({
                        ...event.previousCellPosition,
                        colKey: firstColKey,
                        rowPinned: 'top',
                    });
                }
            });
            return event.previousCellPosition;
        }
    }
    if (event.backwards === false &&
        nextCellPosition?.column?.getColId() === COLUMN_ID_ROW_ACTIONS &&
        nextCellPosition.rowIndex === 0 &&
        nextCellPosition.rowPinned === 'top') {
        // If the phantom row is not dirty and we don't have the cancel icon, we should tabulate to the first editable cell of the first row
        const phantomRow = api.getPinnedTopRow(0);
        if (!phantomRow?.data?.__dirty) {
            const editableColumnId = getFirstEditableColumn(api, {}, true);
            const editableColumn = api.getColumn(editableColumnId);
            if (editableColumn && editableColumnId !== COLUMN_ID_ROW_SELECTION) {
                return {
                    ...nextCellPosition,
                    rowPinned: null,
                    column: editableColumn,
                };
            }
            setTimeout(() => {
                api.stopEditing();
                api.setFocusedCell(nextCellPosition.rowIndex, COLUMN_ID_ROW_SELECTION);
            }, 0);
            return nextCellPosition;
        }
        if (event.editing) {
            api.stopEditing();
            return null;
        }
    }
    if (nextCellPosition?.column?.getColId() === COLUMN_ID_ROW_ACTIONS &&
        event.editing === true &&
        event.backwards === false &&
        nextCellPosition.rowPinned !== 'top' &&
        (nextCellPosition.column.getColDef()?.cellRendererParams?.inlineActionsLength > 0 ||
            nextCellPosition.column.getColDef()?.cellRendererParams?.dropdownActionsLength > 0)) {
        /*
         *when in 'editing' mode (`event.editing === true`), ag-grid skips readonly cells from tab navigation, we have to override imperatively this behavior as we don't want the row actions cell to be ghosted.
         */
        const focused = event.context?.previousCellPosition || event.api.getFocusedCell();
        api.setGridOption('context', { previousCellPosition });
        if (focused &&
            focused.column.getColId() === previousCellPosition.column.getColId() &&
            focused.rowIndex === previousCellPosition.rowIndex) {
            setTimeout(() => {
                api.stopEditing();
                api.setFocusedCell(nextCellPosition.rowIndex, COLUMN_ID_ROW_ACTIONS, nextCellPosition.rowPinned || undefined);
                api.setGridOption('context', {
                    ...event.context,
                    previousCellPosition: undefined,
                });
            }, 0);
            return nextCellPosition;
        }
        return previousCellPosition;
    }
    return nextCellPosition;
};
export const getFirstEditableColumn = (api, data = {}, includeSelect = false) => {
    return (api.getColumnDefs() ?? [])
        .filter((colDef) => !colDef.hide &&
        ((includeSelect && colDef.colId === COLUMN_ID_ROW_SELECTION) ||
            (colDef.colId !== COLUMN_ID_ROW_SELECTION &&
                !!colDef.cellEditor &&
                (colDef.editable === true ||
                    (isFunction(colDef.editable) &&
                        colDef.editable({
                            data,
                            api,
                            colDef,
                        }))))))
        .map(c => c.colId)[0];
};
export const getFilterModel = (api, groupByColumnField) => {
    const filterModel = api?.getFilterModel();
    return filterModel == null
        ? undefined
        : Object.keys(filterModel).reduce((acc, curr) => {
            const key = curr === AUTO_COLUMN_ID ? groupByColumnField : curr;
            if (key != null) {
                acc[key] = filterModel[curr];
            }
            return acc;
        }, {});
};
export const getAllUnselectedIds = (gridApi) => {
    const unselected = [];
    gridApi.forEachNode(node => {
        if (!node.isSelected()) {
            if (!node.data.__isGroup) {
                unselected.push(node.data._id);
            }
        }
    });
    return unselected;
};
export const getColumnStatesForColumnPanel = (screenId, columnDefinitions, columnStates) => {
    if (!columnDefinitions) {
        return [];
    }
    const columns = [];
    if (columnDefinitions[0]?.columnId === COLUMN_ID_LINE_NUMBER) {
        const columnState = (columnStates || []).find(state => COLUMN_ID_LINE_NUMBER && state.colId && state.colId === COLUMN_ID_LINE_NUMBER);
        columns.push({
            colId: COLUMN_ID_LINE_NUMBER,
            isHidden: columnState?.hide === true,
            isMandatory: false,
            isSpecialColumn: true,
            title: localize('@sage/xtrem-ui/table-line-number', 'Line number'),
        });
    }
    const columnsToDisplayInPanel = [
        ...columnDefinitions.filter(c => !!c.colId && shouldDisplayColumnInPanel(c.columnDefinition)),
    ];
    columns.push(...columnsToDisplayInPanel
        .sort((a, b) => {
        if (columnStates) {
            const aColumnStateIndex = columnStates.findIndex(acs => acs.colId === a.colId);
            const bColumnStateIndex = columnStates.findIndex(bcs => bcs.colId === b.colId);
            return aColumnStateIndex - bColumnStateIndex;
        }
        const aColumnDefinitionIndex = columnDefinitions.findIndex(acs => acs.colId === a.colId);
        const bColumnDefinitionIndex = columnDefinitions.findIndex(bcs => bcs.colId === b.colId);
        return aColumnDefinitionIndex - bColumnDefinitionIndex;
    })
        .map(c => {
        const columnState = (columnStates || []).find(state => c.colId && state.colId && state.colId === c.colId);
        const properties = c.columnDefinition?.properties;
        const title = resolveByValue({
            propertyValue: properties.title,
            rowValue: null,
            fieldValue: null,
            skipHexFormat: true,
            screenId,
        });
        return {
            colId: String(c.colId),
            title,
            isMandatory: !!properties.isMandatory || c.cellRenderer === 'agGroupCellRenderer',
            isHidden: columnState ? columnState.hide === true : !!properties.isHiddenOnMainField,
        };
    }));
    return columns;
};
//# sourceMappingURL=ag-grid-table-utils.js.map