import { formatDateToCurrentLocale, isValidIsoDate } from '@sage/xtrem-date-time';
import { CUSTOM_DATA_PROPERTY } from '@sage/xtrem-shared';
import Loader from 'carbon-react/esm/components/loader';
import { get, isArray, isNil, isObject, set } from 'lodash';
import { navigationPanelId } from '../../component/container/navigation-panel/navigation-panel-types';
import { MENU_SEPARATOR } from '../../component/control-objects';
import { CheckboxCellRenderer } from '../../component/field/checkbox/checkbox-cell-renderer';
import { CountCellRenderer } from '../../component/field/count/count-cell-renderer';
import DateCellEditor from '../../component/field/date/date-cell-editor';
import { DropdownListCellEditor } from '../../component/field/dropdown-list/dropdown-list-cell-editor';
import { DropdownListCellRenderer } from '../../component/field/dropdown-list/dropdown-list-cell-renderer';
import FilterSelectCellEditor from '../../component/field/filter-select/filter-select-cell-editor';
import { FilterSelectRenderer } from '../../component/field/filter-select/filter-select-cell-renderer';
import { IconCellRenderer } from '../../component/field/icon/icon-cell-renderer';
import { ImageCellRenderer } from '../../component/field/image/image-cell-renderer';
import { LabelCellRenderer } from '../../component/field/label/label-cell-renderer';
import { LinkCellRenderer } from '../../component/field/link/link-cell-renderer';
import NumericEditor from '../../component/field/numeric/numeric-cell-editor';
import { ProgressCellRenderer } from '../../component/field/progress/progress-cell-renderer';
import ReferenceCellEditor from '../../component/field/reference/reference-cell-editor';
import ReferenceRenderer from '../../component/field/reference/reference-cell-renderer';
import { getReferenceValueField } from '../../component/field/reference/reference-utils';
import { RelativeDateCellRenderer } from '../../component/field/relative-date/relative-date-cell-renderer';
import { SelectCellEditor } from '../../component/field/select/select-cell-editor';
import { SelectRenderer } from '../../component/field/select/select-cell-renderer';
import { SwitchRenderer } from '../../component/field/switch/switch-cell-renderer';
import { FieldKey } from '../../component/types';
import { DesktopNestedGridEmptyComponent, DesktopTableEmptyComponent, } from '../../component/ui/no-rows-found/no-rows-found-component';
import { CellWrapper } from '../../component/ui/table-shared/cell/cell-wrapper';
import { CheckboxCell } from '../../component/ui/table-shared/cell/checkbox-cell';
import { DefaultCellEditor } from '../../component/ui/table-shared/cell/default-cell-editor';
import { DefaultCellRenderer } from '../../component/ui/table-shared/cell/default-cell-renderer';
import { TableDropdownActionsCell } from '../../component/ui/table-shared/cell/table-dropdown-action-cell';
import { ValidationCellRenderer } from '../../component/ui/table-shared/cell/table-validation-cell-renderer';
import { BooleanFilter } from '../../component/ui/table-shared/filters/boolean/boolean-filter';
import ClearFloatingFilter from '../../component/ui/table-shared/filters/clear/clear-floating-filter';
import DateFilter from '../../component/ui/table-shared/filters/date/date-filter';
import DateFloatingFilter from '../../component/ui/table-shared/filters/date/date-floating-filter';
import NumericFilter from '../../component/ui/table-shared/filters/numeric/numeric-filter';
import NumericFloatingFilter from '../../component/ui/table-shared/filters/numeric/numeric-floating-filter';
import ReferenceFilter from '../../component/ui/table-shared/filters/reference/reference-filter';
import ReferenceFloatingFilter from '../../component/ui/table-shared/filters/reference/reference-floating-filter';
import { TableRowSelectionHeader } from '../../component/ui/table-shared/table-row-selection-header';
import { calculateDeepPaths } from '../../service/customization-service';
import { localize, localizeEnumMember } from '../../service/i18n-service';
import { ContextType, GraphQLKind, GraphQLTypes } from '../../types';
import { getElementAccessStatusWithoutId } from '../access-utils';
import { xtremConsole } from '../console';
import { formatNumericValue } from '../formatters';
import { convertDeepBindToPathNotNull } from '../nested-field-utils';
import { findDeepPropertyType } from '../node-utils';
import { resolveByValue } from '../resolve-value-utils';
import { getPageDefinitionFromState, getPagePropertiesFromPageDefinition } from '../state-utils';
import { cleanMetadataFromRecord, schemaTypeNameFromNodeName, splitValueToMergedValue } from '../transformers';
import { suppressKeyboardEventForActionCellEditor, suppressKeyboardEventForReferenceCellEditor, } from './ag-grid-cell-editor-utils';
function getFilterableScalarPropertyType(type) {
    return {
        kind: GraphQLKind.Scalar,
        type,
        name: '',
        canFilter: true,
    };
}
const textFilterOptions = () => [
    'contains',
    'notContains',
    'startsWith',
    'endsWith',
    'equals',
    'notEqual',
    {
        displayKey: 'greaterThanOrEqual',
        displayName: localize('@sage/xtrem-ui/greaterThanOrEqual', 'Greater than or equal to'),
        predicate: ([filterValue], cellValue) => String(cellValue).localeCompare(String(filterValue)) > 0 ||
            String(filterValue).toLowerCase() === String(cellValue).toLowerCase(),
    },
    {
        displayKey: 'lessThanOrEqual',
        displayName: localize('@sage/xtrem-ui/table-lessThanOrEqual', 'Less than or equal'),
        predicate: ([filterValue], cellValue) => String(cellValue).localeCompare(String(filterValue)) < 0 ||
            String(filterValue).toLowerCase() === String(cellValue).toLowerCase(),
    },
    {
        displayKey: 'inRange',
        displayName: localize('@sage/xtrem-ui/table-inRange', 'In range'),
        predicate: ([filterValue1, filterValue2], cellValue) => String(cellValue).localeCompare(String(filterValue1)) < 0 &&
            String(cellValue).localeCompare(String(filterValue2)) > 0,
        numberOfInputs: 2,
    },
]; // ['equals', 'notEqual', 'contains', 'notContains', 'startsWith', 'endsWith']
/**
 * ['equals', 'notEqual', 'contains', 'notContains', 'startsWith', 'endsWith', 'lessThan', 'lessThanOrEqual', 'greaterThan', 'greaterThanOrEqual', 'inRange', 'empty']
 */
const numberFilterOptions = ['equals', 'notEqual', 'lessThanOrEqual', 'greaterThanOrEqual', 'inRange'];
const dateFilterOptions = ['equals', 'notEqual', 'lessThanOrEqual', 'greaterThanOrEqual', 'inRange'];
export const defaultValueGetter = (params, screenId, columnProperties) => (mapper = (v) => v) => {
    if (!params.colDef.field) {
        return null;
    }
    const map = columnProperties.map;
    if (!params.data) {
        return null;
    }
    const bind = convertDeepBindToPathNotNull(columnProperties.bind);
    // Do not display _id column for groups. Convention for groups is id = "__group-{groupValue}"
    const cellValue = params.data?.__isGroup && bind === '_id' ? '' : get(params.data, bind, null);
    if (map) {
        return resolveByValue({
            skipHexFormat: true,
            propertyValue: map,
            screenId,
            fieldValue: cellValue,
            rowValue: splitValueToMergedValue(cleanMetadataFromRecord(params.data)),
        });
    }
    return mapper(cellValue);
};
export const defaultValueSetter = (params) => {
    const bind = params.colDef.field || params.column.getColId();
    const currentValue = get(params.data, bind, null);
    const newValue = !isNil(params.newValue) ? params.newValue : null;
    if (currentValue === newValue) {
        return false;
    }
    set(params.data, bind, newValue);
    return true;
};
/**
 * Check the node type and colum properties to find the appropriate filter type
 * @param columnProperties
 * @param graphqlType
 * @param hasFloatingFilters
 * @returns
 */
const getFilterTypeFromGraphqlType = (props) => {
    const properties = {};
    // Check only whether the "_id" property of the reference node type is filterable for reference fields
    const graphqlType = props.columnProperties._controlObjectType === FieldKey.Reference
        ? findDeepPropertyType(props.propertyGraphType?.type, '_id', props.nodeTypes, true)
        : props.propertyGraphType;
    const parentNodeType = props.propertyGraphType?.parentNode
        ? props.nodeTypes[props.propertyGraphType.parentNode]
        : null;
    const isCustomField = props.propertyGraphType?.type === GraphQLTypes.Json && props.propertyGraphType.name === CUSTOM_DATA_PROPERTY;
    const { selectorSegments } = calculateDeepPaths(props.columnId);
    const subfieldFieldType = get(parentNodeType?.properties, selectorSegments.join('.'));
    if (!props.propertyGraphType?.type ||
        props.columnProperties.isTransient ||
        props.columnProperties.canFilter === false ||
        graphqlType?.canSort === false) {
        properties.sortable = false;
    }
    // If the column is transient, filtering is not allowed.
    if (!props.propertyGraphType?.type ||
        props.columnProperties.isTransient ||
        props.columnProperties.canFilter === false ||
        graphqlType?.canFilter === false) {
        properties.floatingFilter = false;
        properties.filter = false;
        return properties;
    }
    properties.floatingFilter = props.hasFloatingFilters;
    properties.filter = true;
    if (props.propertyGraphType?.type === GraphQLTypes.Enum ||
        (isCustomField && subfieldFieldType && props.columnProperties.options)) {
        const enumType = props.propertyGraphType?.enumType || props.columnProperties.optionType;
        const formattedEnumType = schemaTypeNameFromNodeName(enumType);
        applyEnumFilterOptions(properties, props.screenId, props.enumTypes[formattedEnumType], enumType, props.hasFloatingFilters, props.columnProperties.options, props.columnProperties.map);
    }
    else if (props.propertyGraphType?.type === GraphQLTypes.String ||
        props.propertyGraphType?.type === GraphQLTypes.ID ||
        (props.propertyGraphType.type === GraphQLTypes.IntOrString && props.propertyGraphType.name === '_id')) {
        applyTextFilterOptions(properties, props.hasFloatingFilters);
    }
    else if (props.propertyGraphType?.type === GraphQLTypes.Date ||
        props.propertyGraphType?.type === GraphQLTypes.DateTime) {
        applyDateFilterOptions(properties, props.hasFloatingFilters);
    }
    else if (props.propertyGraphType?.type === GraphQLTypes.Int ||
        props.propertyGraphType?.type === GraphQLTypes.IntOrString ||
        props.propertyGraphType?.type === GraphQLTypes.Float ||
        props.propertyGraphType?.type === GraphQLTypes.Decimal) {
        applyNumericFilterOptions(properties, props.hasFloatingFilters);
    }
    else if (props.propertyGraphType?.type && props.propertyGraphType?.kind === GraphQLKind.Object) {
        applyReferenceFilterOptions(properties, props);
    }
    else if (props.propertyGraphType?.type === GraphQLTypes.Boolean) {
        applyBooleanFilterOptions(properties, props);
    }
    else if (props.propertyGraphType?.type === GraphQLTypes.Json && props.columnId.split('.').length > 1) {
        // If the field is a deeply nested JSON column we apply the filters based on the column definition type because there is not a real server side data type
        switch (props.columnProperties._controlObjectType) {
            case FieldKey.Numeric:
                applyNumericFilterOptions(properties, props.hasFloatingFilters);
                break;
            case FieldKey.Text:
                applyTextFilterOptions(properties, props.hasFloatingFilters);
                break;
            case FieldKey.Date:
                applyDateFilterOptions(properties, props.hasFloatingFilters);
                break;
            // TODO: enum and enum array fields
            default:
                properties.floatingFilter = false;
                properties.filter = false;
        }
    }
    else {
        properties.floatingFilter = false;
        properties.filter = false;
    }
    return properties;
};
export const getReferenceFieldColumnConfiguration = (props) => {
    const { columnProperties, screenId } = props;
    const cellProps = getCellProps(props);
    const valueGetter = (params) => defaultValueGetter(params, screenId, columnProperties)(v => {
        if (v &&
            columnProperties.valueField &&
            getReferenceValueField(columnProperties) !== undefined &&
            get(v, getReferenceValueField(columnProperties))) {
            return get(v, getReferenceValueField(columnProperties));
        }
        return '';
    });
    return {
        enableRowGroup: props.tableElementId === navigationPanelId,
        allowedAggFuncs: ['count'],
        cellEditor: FieldKey.Reference,
        filter: columnProperties.canFilter !== false,
        cellEditorParams: cellProps,
        cellRenderer: 'ReferenceRenderer',
        cellRendererParams: cellProps,
        minWidth: 100 + (columnProperties.columns ? 24 : 0) + (columnProperties.imageField ? 24 : 0),
        suppressKeyboardEvent: suppressKeyboardEventForReferenceCellEditor,
        valueGetter,
        valueParser: (params) => {
            const newData = { ...params.data };
            set(newData, params.colDef.field, params.newValue);
            return valueGetter({ colDef: params.colDef, data: newData });
        },
        valueFormatter: (params) => {
            return params.data?.__isGroup &&
                params.data?.__groupKey === `${convertDeepBindToPathNotNull(columnProperties.bind)}._id`
                ? `${params.value || localize('@sage/xtrem-ui/no-value', 'No value')} (${params.data.__groupCount})`
                : params?.value;
        },
        cellEditorPopup: true,
        ...getFilterTypeFromGraphqlType(props),
    };
};
export const getImageFieldColumnConfiguration = (props) => ({
    editable: false,
    sortable: false,
    filter: false,
    cellRendererParams: getCellProps(props),
    cellRenderer: FieldKey.Image,
});
export const getLabelFieldColumnConfiguration = (props) => {
    const { columnProperties, enumTypes } = props;
    const enumOptions = columnProperties.optionType
        ? enumTypes[schemaTypeNameFromNodeName(columnProperties.optionType)]
        : undefined;
    const localizedOptions = enumOptions
        ? enumOptions.reduce((value, key) => {
            value[key] = localizeEnumMember(columnProperties.optionType, key);
            return value;
        }, {})
        : undefined;
    const cellProps = { ...getCellProps(props), localizedOptions, enumOptions };
    const properties = {
        editable: false,
        cellRenderer: FieldKey.Label,
        cellRendererParams: cellProps,
    };
    if (props.columnProperties.optionType &&
        props.columnProperties.canFilter !== false &&
        !props.columnProperties.isTransient) {
        properties.filter = 'agSetColumnFilter';
        properties.filterParams = {
            values: enumOptions,
            defaultToNothingSelected: true,
            defaultJoinOperator: 'OR',
            valueFormatter: (v) => {
                return localizedOptions?.[v.value] || v.value;
            },
        };
        properties.suppressHeaderMenuButton = false;
    }
    return {
        ...properties,
        ...getFilterTypeFromGraphqlType({
            ...props,
            // If the table is transient we just handle the label type as a string
            ...(props.tableProperties?.isTransient && !props.propertyGraphType
                ? getFilterableScalarPropertyType(GraphQLTypes.String)
                : {}),
        }),
        enableRowGroup: props.tableElementId === navigationPanelId,
        valueGetter: (params) => defaultValueGetter(params, props.screenId, columnProperties)(v => {
            return v ?? '';
        }),
    };
};
export const getIconFieldColumnConfiguration = (props) => ({
    editable: false,
    filter: false,
    sortable: false,
    cellRenderer: FieldKey.Icon,
    cellRendererParams: getCellProps(props),
    ...getFilterTypeFromGraphqlType(props),
});
export const getCountFieldColumnConfiguration = (props) => ({
    editable: false,
    filter: false,
    sortable: false,
    cellRenderer: FieldKey.Count,
    cellRendererParams: getCellProps(props),
});
export const getCheckboxFieldColumnConfiguration = (props) => {
    // If the table is transient we just handle the checkbox type as a boolean
    if (props.tableProperties?.isTransient && !props.propertyGraphType?.type) {
        props.propertyGraphType = getFilterableScalarPropertyType(GraphQLTypes.Boolean);
    }
    const cellParams = getCellProps(props);
    return {
        editable: true,
        cellRendererParams: cellParams,
        cellEditorParams: { ...cellParams, isEditing: true },
        cellRenderer: FieldKey.Checkbox,
        cellEditor: FieldKey.Checkbox,
        minWidth: 44,
        ...getFilterTypeFromGraphqlType(props),
    };
};
export const getDateFieldColumnConfiguration = (props) => {
    const cellProps = getCellProps(props);
    // If the table is transient we just handle the date type as a date
    if (props.tableProperties?.isTransient && !props.propertyGraphType?.type) {
        props.propertyGraphType = getFilterableScalarPropertyType(GraphQLTypes.Date);
    }
    return {
        cellRenderer: DefaultCellRenderer,
        cellRendererParams: cellProps,
        cellEditor: FieldKey.Date,
        valueFormatter: (params) => {
            if (params.value !== undefined && params.value !== null && isValidIsoDate(params.value)) {
                return formatDateToCurrentLocale(params.value, props.locale, 'FullDate');
            }
            return '';
        },
        cellEditorParams: cellProps,
        enableRowGroup: props.tableElementId === navigationPanelId,
        valueGetter: (params) => defaultValueGetter(params, props.screenId, props.columnProperties)(v => {
            return v ?? '';
        }),
        ...getFilterTypeFromGraphqlType(props),
    };
};
const getNumericValueFormatter = (screenId, columnProperties) => (params) => {
    let value = formatNumericValue({
        screenId,
        value: params.value,
        scale: columnProperties.scale,
        rowValue: params.data,
    });
    value = resolveTextPrefix(params.value, value, params.data, screenId, columnProperties.prefix);
    return resolveTextPostfix(params.value, value, params.data, screenId, columnProperties.postfix);
};
const resolveTextPrefix = (initialValue, currentValue, data, screenId, prefix) => {
    const textPrefix = resolveByValue({
        fieldValue: initialValue,
        propertyValue: prefix,
        rowValue: splitValueToMergedValue(cleanMetadataFromRecord(data)),
        screenId,
        skipHexFormat: true,
    });
    return textPrefix ? `${textPrefix} ${currentValue ?? ''}` : currentValue;
};
const resolveTextPostfix = (initialValue, currentValue, data, screenId, postfix) => {
    const textPostfix = resolveByValue({
        fieldValue: initialValue,
        propertyValue: postfix,
        rowValue: splitValueToMergedValue(cleanMetadataFromRecord(data)),
        skipHexFormat: true,
        screenId,
    });
    return textPostfix ? `${currentValue ?? ''} ${textPostfix}` : currentValue;
};
export const getValueFormatter = (screenId, columnProperties) => (params) => {
    const value = resolveTextPrefix(params.value, params.value, params.data, screenId, columnProperties.prefix);
    return resolveTextPostfix(params.value, value, params.data, screenId, columnProperties.postfix);
};
const applyTextFilterOptions = (properties, hasFloatingFilters = false) => {
    properties.suppressHeaderMenuButton = false;
    properties.filter = 'agTextColumnFilter';
    properties.menuTabs = hasFloatingFilters ? ['generalMenuTab'] : ['filterMenuTab', 'generalMenuTab'];
    properties.filterParams = {
        ...properties.filterParams,
        filterOptions: textFilterOptions(),
        defaultJoinOperator: 'OR',
    };
};
const applyEnumFilterOptions = (properties, screenId, enumDefinition, optionType, hasFloatingFilters = false, options, map) => {
    properties.menuTabs = hasFloatingFilters ? ['generalMenuTab'] : ['filterMenuTab', 'generalMenuTab'];
    // If no enum definition was found for the enum type, we cannot filter on it.
    if ((!enumDefinition || !optionType) && (!options || !map)) {
        return;
    }
    const localizedOptions = enumDefinition?.reduce((value, key) => {
        value[key] = optionType ? localizeEnumMember(optionType, key) : key;
        return value;
    }, {});
    properties.filter = 'agSetColumnFilter';
    properties.filterParams = {
        values: enumDefinition || options,
        defaultToNothingSelected: true,
        defaultJoinOperator: 'OR',
        valueFormatter: (v) => {
            if (localizedOptions) {
                return localizedOptions?.[v.value];
            }
            if (map) {
                return resolveByValue({
                    propertyValue: map,
                    screenId,
                    skipHexFormat: true,
                    rowValue: null,
                    fieldValue: v.value,
                });
            }
            return v.value;
        },
    };
    properties.suppressHeaderMenuButton = false;
};
export const getTextFieldColumnConfiguration = (props) => {
    const { columnProperties, screenId } = props;
    const properties = {
        cellRenderer: DefaultCellRenderer,
        cellRendererParams: getCellProps(props),
        cellEditor: 'agTextCellEditor',
        valueGetter: (params) => defaultValueGetter(params, screenId, columnProperties)((value) => {
            // Filter out functional developer introduced faulty values.
            if (isObject(value) || isArray(value)) {
                xtremConsole.warn(`Invalid type set to table text field in ${params.colDef.field}`, value);
                return null;
            }
            return value;
        }),
    };
    if (columnProperties.prefix || columnProperties.postfix) {
        properties.valueFormatter = getValueFormatter(screenId, columnProperties);
    }
    return {
        ...properties,
        ...getFilterTypeFromGraphqlType({
            ...props,
            // If the table is transient we just handle the label type as a string
            ...(props.tableProperties?.isTransient && !props.propertyGraphType?.type
                ? { propertyGraphType: getFilterableScalarPropertyType(GraphQLTypes.String) }
                : {}),
        }),
    };
};
export const getLinkFieldColumnConfiguration = (props) => ({
    editable: false,
    cellRendererParams: getCellProps(props),
    cellRenderer: FieldKey.Link,
    onCellClicked: undefined,
    ...getFilterTypeFromGraphqlType({
        ...props,
        // If the table is transient we just handle the label type as a string
        ...(props.tableProperties?.isTransient && !props.propertyGraphType?.type
            ? { propertyGraphType: getFilterableScalarPropertyType(GraphQLTypes.String) }
            : {}),
    }),
});
export const getRelativeDateFieldColumnConfiguration = (props) => ({
    editable: false,
    cellRendererParams: {
        ...getCellProps(props),
        relativeDateScope: props.propertyGraphType?.type === 'Date' ? 'date' : 'datetime',
    },
    cellRenderer: FieldKey.RelativeDate,
    onCellClicked: undefined,
    ...getFilterTypeFromGraphqlType(props),
});
export const getProgressFieldColumnConfiguration = (props) => {
    return {
        editable: false,
        cellRenderer: FieldKey.Progress,
        cellRendererParams: getCellProps(props),
        ...getFilterTypeFromGraphqlType({
            ...props,
            // If the table is transient we just handle the progress type as a decimal
            ...(props.tableProperties?.isTransient && !props.propertyGraphType?.type
                ? { propertyGraphType: getFilterableScalarPropertyType(GraphQLTypes.Decimal) }
                : {}),
        }),
    };
};
const applyNumericFilterOptions = (properties, hasFloatingFilters = false) => {
    properties.suppressHeaderMenuButton = false;
    properties.filter = 'agNumberColumnFilter';
    properties.menuTabs = hasFloatingFilters ? ['generalMenuTab'] : ['filterMenuTab', 'generalMenuTab'];
    properties.floatingFilterComponent = NumericFloatingFilter;
    properties.filter = NumericFilter;
    properties.filterParams = {
        ...properties.filterParams,
        defaultJoinOperator: 'OR',
        filterOptions: numberFilterOptions,
    };
};
const applyBooleanFilterOptions = (properties, props) => {
    properties.suppressHeaderMenuButton = false;
    properties.filter = BooleanFilter;
    properties.filterParams = {
        ...properties.filterParams,
        defaultJoinOperator: 'OR',
        ...{
            values: [true, false],
            controlObjectType: props.columnProperties._controlObjectType,
        },
    };
    properties.suppressFiltersToolPanel = true;
};
const applyDateFilterOptions = (properties, hasFloatingFilters = false) => {
    properties.suppressHeaderMenuButton = false;
    properties.filter = DateFilter;
    // We need to set a special floating filter to render the current filter in the header
    if (hasFloatingFilters) {
        properties.floatingFilterComponent = DateFloatingFilter;
    }
    properties.menuTabs = hasFloatingFilters ? ['generalMenuTab'] : ['filterMenuTab', 'generalMenuTab'];
    properties.filterParams = {
        ...properties.filterParams,
        defaultJoinOperator: 'OR',
        filterOptions: dateFilterOptions,
    };
};
const applyReferenceFilterOptions = (properties, props) => {
    properties.suppressHeaderMenuButton = false;
    // Custom filter component is used here to provide a scrollable, paginated selection experience
    properties.filter = ReferenceFilter;
    // We need to set a special floating filter to render the current filter in the header
    if (props.hasFloatingFilters) {
        properties.floatingFilterComponent = ReferenceFloatingFilter;
    }
    properties.menuTabs = props.hasFloatingFilters ? ['generalMenuTab'] : ['filterMenuTab', 'generalMenuTab'];
    properties.filterParams = {
        ...properties.filterParams,
        defaultJoinOperator: 'OR',
        filterOptions: props.columnProperties,
    };
};
export const getNumericFieldColumnConfiguration = (props) => {
    const cellProps = getCellProps(props);
    // If the table is transient we just handle the number type as a decimal
    if (props.tableProperties?.isTransient && !props.propertyGraphType?.type) {
        props.propertyGraphType = getFilterableScalarPropertyType(GraphQLTypes.Decimal);
    }
    return {
        type: 'rightAligned',
        cellRenderer: DefaultCellRenderer,
        cellRendererParams: cellProps,
        cellEditor: FieldKey.Numeric,
        cellEditorParams: cellProps,
        valueFormatter: getNumericValueFormatter(props.screenId, props.columnProperties),
        ...getFilterTypeFromGraphqlType({
            ...props,
            // If the table is transient we just handle the numeric field type as a decimal
            ...(props.tableProperties?.isTransient && !props.propertyGraphType?.type
                ? { propertyGraphType: getFilterableScalarPropertyType(GraphQLTypes.Decimal) }
                : {}),
        }),
    };
};
export const getAggregateFieldColumnConfiguration = (props) => {
    const cellProps = getCellProps(props);
    const properties = {
        type: 'rightAligned',
        cellRenderer: DefaultCellRenderer,
        cellRendererParams: cellProps,
        valueFormatter: getNumericValueFormatter(props.screenId, props.columnProperties),
        filter: false,
        sortable: false,
        editable: false,
    };
    return properties;
};
export const getFilterSelectFieldColumnConfiguration = (props) => {
    const cellProps = getCellProps(props);
    // If the table is transient we just handle the filter select type as a string
    if (props.tableProperties?.isTransient && !props.propertyGraphType?.type) {
        props.propertyGraphType = getFilterableScalarPropertyType(GraphQLTypes.String);
    }
    return {
        cellEditorParams: cellProps,
        cellEditor: FieldKey.FilterSelect,
        cellRenderer: FilterSelectRenderer,
        cellRendererParams: cellProps,
        suppressKeyboardEvent: suppressKeyboardEventForReferenceCellEditor,
        valueGetter: (params) => defaultValueGetter(params, props.screenId, props.columnProperties)((value) => {
            // Filter out functional developer introduced faulty values.
            if (isObject(value) || isArray(value)) {
                xtremConsole.warn(`Invalid type set to table text field in ${params.colDef.field}`, value);
                return null;
            }
            return value;
        }),
        cellEditorPopup: true,
        ...getFilterTypeFromGraphqlType(props),
    };
};
export const getDropdownListFieldColumnConfiguration = (props) => {
    const { columnProperties, enumTypes } = props;
    const enumOptions = columnProperties.optionType
        ? enumTypes[schemaTypeNameFromNodeName(columnProperties.optionType)]
        : undefined;
    const localizedOptions = enumOptions
        ? enumOptions.reduce((value, key) => {
            value[key] = localizeEnumMember(columnProperties.optionType, key);
            return value;
        }, {})
        : undefined;
    const cellProps = { ...getCellProps(props), localizedOptions, enumOptions };
    // If the table is transient we just handle the drop down list type as a string
    if (props.tableProperties?.isTransient && !props.propertyGraphType?.type) {
        props.propertyGraphType = getFilterableScalarPropertyType(GraphQLTypes.String);
    }
    return {
        cellEditor: FieldKey.DropdownList,
        cellEditorParams: cellProps,
        cellRenderer: DropdownListCellRenderer,
        cellRendererParams: cellProps,
        suppressKeyboardEvent: suppressKeyboardEventForReferenceCellEditor,
        cellEditorPopup: true,
        ...getFilterTypeFromGraphqlType(props),
    };
};
export const getSelectFieldColumnConfiguration = (props) => {
    const { columnProperties, enumTypes } = props;
    const enumType = columnProperties.optionType || props.propertyGraphType?.enumType;
    const enumOptions = enumType ? enumTypes[schemaTypeNameFromNodeName(enumType)] : undefined;
    const localizedOptions = enumOptions && enumType
        ? enumOptions.reduce((value, key) => {
            value[key] = localizeEnumMember(enumType, key);
            return value;
        }, {})
        : undefined;
    const cellProps = { ...getCellProps(props), localizedOptions, enumOptions };
    // If the table is transient we just handle the select type as a string
    if (props.tableProperties?.isTransient && !props.propertyGraphType?.type) {
        props.propertyGraphType = getFilterableScalarPropertyType(GraphQLTypes.String);
    }
    return {
        cellEditor: FieldKey.Select,
        cellEditorParams: cellProps,
        cellRenderer: SelectRenderer,
        cellRendererParams: cellProps,
        suppressKeyboardEvent: suppressKeyboardEventForReferenceCellEditor,
        cellEditorPopup: true,
        ...getFilterTypeFromGraphqlType(props),
    };
};
export const getSwitchFieldColumnConfiguration = (props) => {
    // If the table is transient we just handle the switch type as a boolean
    if (props.tableProperties?.isTransient && !props.propertyGraphType?.type) {
        props.propertyGraphType = getFilterableScalarPropertyType(GraphQLTypes.Boolean);
    }
    const cellParams = (params) => {
        const onChange = (value) => {
            if (typeof params?.node?.setDataValue === 'function') {
                params.node.setDataValue(props.columnId, value);
            }
        };
        return { ...getCellProps(props), onChange };
    };
    return {
        editable: true,
        cellRenderer: SwitchRenderer,
        cellEditor: SwitchRenderer,
        minWidth: 80,
        cellRendererParams: cellParams,
        cellEditorParams: (params) => {
            return { ...cellParams(params), isEditing: true };
        },
        ...getFilterTypeFromGraphqlType(props),
    };
};
export const COLUMN_ID_ROW_SELECTION = '__select';
export const getRowSelectColumnConfiguration = ({ screenId, isParentFieldDisabled, tableElementId, node, fieldProperties, contextType, activeOptionsMenuItem, }) => ({
    pinned: 'left',
    columnId: COLUMN_ID_ROW_SELECTION,
    headerCheckboxSelectionFilteredOnly: true,
    // Add select all header only to navigation panel
    ...(contextType === ContextType.navigationPanel && { headerComponent: TableRowSelectionHeader }),
    headerComponentParams: { screenId, elementId: tableElementId, node, fieldProperties, activeOptionsMenuItem },
    checkboxSelection: props => !props.node.footer,
    field: COLUMN_ID_ROW_SELECTION,
    suppressColumnsToolPanel: true,
    headerName: '',
    editable: false,
    sortable: false,
    filter: false,
    suppressHeaderMenuButton: true,
    cellClass: () => {
        const classNames = ['e-table-field-select-row'];
        if (isParentFieldDisabled) {
            classNames.push('e-table-field-select-row-disabled');
        }
        return classNames;
    },
    headerClass: 'e-table-field-select-all',
    minWidth: 44,
    maxWidth: 44,
    width: 44,
    screenId,
    suppressSizeToFit: true,
    suppressAutoSize: true,
    suppressKeyboardEvent: (params) => {
        return params.event.key === 'Enter';
    },
    isEditable: () => false,
});
export const COLUMN_ID_LINE_NUMBER = '__line_number';
export const getLineNumberColumnConfiguration = (screenId) => ({
    pinned: 'left',
    columnId: COLUMN_ID_LINE_NUMBER,
    checkboxSelection: false,
    field: COLUMN_ID_LINE_NUMBER,
    suppressColumnsToolPanel: true,
    suppressMovable: true,
    headerName: '',
    editable: false,
    sortable: false,
    filter: false,
    suppressHeaderMenuButton: true,
    cellClass: 'e-table-field-line-number',
    minWidth: 44,
    maxWidth: 44,
    width: 44,
    screenId,
    suppressSizeToFit: true,
    suppressAutoSize: false,
    isEditable: () => false,
    valueGetter: (params) => {
        if (params.node?.rowPinned) {
            return '';
        }
        return String((params.node?.rowIndex || 0) + 1);
    },
});
export const COLUMN_ID_VALIDATIONS = '__validation';
export const getValidationColumnConfiguration = (screenId, getColumns) => ({
    cellRenderer: ValidationCellRenderer.name,
    columnId: COLUMN_ID_VALIDATIONS,
    pinned: 'left',
    editable: false,
    field: COLUMN_ID_VALIDATIONS,
    suppressColumnsToolPanel: true,
    filter: false,
    headerName: '',
    initialHide: true,
    hide: true,
    maxWidth: 25,
    minWidth: 25,
    screenId,
    sortable: false,
    suppressAutoSize: true,
    suppressHeaderMenuButton: true,
    suppressSizeToFit: true,
    headerClass: 'e-table-field-validation-summary-header',
    cellClass: 'e-table-field-validation-summary',
    width: 25,
    isEditable: () => false,
    getColumns,
});
export const COLUMN_ID_AUTO_COLUMN = 'ag-Grid-AutoColumn';
export const COLUMN_ID_ROW_ACTIONS = '__actions';
export const COLUMN_CELLCLASS_ROW_ACTIONS = 'e-table-field-dropdown-actions-cell';
export const FOCUSED_BUT_BORDERLESS = 'e-ag-cell-focus--no-border';
export const getDropdownActionsColumnConfiguration = (screenId, tableElementId, level, wrapper, isParentFieldDisabled, inlineActions, dropdownActions, hasFloatingFilters) => {
    const cellParams = {
        elementId: tableElementId,
        screenId,
        level,
        wrapper,
        isParentFieldDisabled,
        inlineActions,
        dropdownActions,
    };
    return {
        pinned: 'right',
        columnId: COLUMN_ID_ROW_ACTIONS,
        field: COLUMN_ID_ROW_ACTIONS,
        suppressColumnsToolPanel: true,
        cellRenderer: TableDropdownActionsCell.name,
        cellEditor: TableDropdownActionsCell.name,
        editable: true,
        sortable: false,
        suppressHeaderMenuButton: true,
        cellClass: COLUMN_CELLCLASS_ROW_ACTIONS,
        headerClass: 'e-table-field-dropdown-actions-header',
        cellEditorPopup: true,
        minWidth: Math.max(44, 44 * inlineActions.length + 44 * Math.sign(dropdownActions.length)),
        maxWidth: Math.max(44, 44 * inlineActions.length + 44 * Math.sign(dropdownActions.length)),
        width: Math.max(44, 44 * inlineActions.length + 44 * Math.sign(dropdownActions.length)),
        screenId,
        suppressSizeToFit: true,
        suppressAutoSize: true,
        suppressKeyboardEvent: suppressKeyboardEventForActionCellEditor,
        isEditable: () => false,
        cellRendererParams: cellParams,
        cellEditorParams: {
            ...cellParams,
            isEditing: true,
        },
        headerName: '',
        filter: hasFloatingFilters,
        suppressHeaderFilterButton: true,
        floatingFilter: hasFloatingFilters,
        floatingFilterComponent: hasFloatingFilters ? ClearFloatingFilter : null,
        floatingFilterComponentParams: { suppressFilterButton: true, elementId: tableElementId },
    };
};
export const frameworkComponents = {
    [FieldKey.Checkbox]: CheckboxCellRenderer,
    [FieldKey.Count]: CountCellRenderer,
    [FieldKey.Date]: DateCellEditor,
    [FieldKey.DropdownList]: DropdownListCellEditor,
    [FieldKey.FilterSelect]: FilterSelectCellEditor,
    [FieldKey.Icon]: IconCellRenderer,
    [FieldKey.Image]: ImageCellRenderer,
    [FieldKey.Label]: LabelCellRenderer,
    [FieldKey.Link]: LinkCellRenderer,
    [FieldKey.Numeric]: NumericEditor,
    [FieldKey.Progress]: ProgressCellRenderer,
    [FieldKey.Reference]: ReferenceCellEditor,
    [FieldKey.RelativeDate]: RelativeDateCellRenderer,
    [FieldKey.Select]: SelectCellEditor,
    [ValidationCellRenderer.name]: ValidationCellRenderer,
    [TableDropdownActionsCell.name]: TableDropdownActionsCell,
    DefaultCellEditor,
    DefaultCellRenderer,
    DropdownListCellRenderer,
    FilterSelectRenderer,
    Loader,
    ReferenceRenderer,
    SelectRenderer,
    SwitchRenderer,
    CheckboxCell,
    DesktopTableEmptyComponent,
    DesktopNestedGridEmptyComponent,
};
export function getTooltipMessage(props) {
    const columnBind = props.colDef.field;
    return Object.prototype.hasOwnProperty.call(props.data?.__validationState || {}, columnBind)
        ? props.data?.__validationState[columnBind].message
        : undefined;
}
export function getTextAlignment(props) {
    const type = props.colDef.type;
    if (type === 'rightAligned') {
        return 'right';
    }
    if (type === 'leftAligned') {
        return 'left';
    }
    if (type === 'centerAligned') {
        return 'center';
    }
    return 'left';
}
export function getCellProps({ columnProperties, tableProperties, tableElementId, columnId, screenId, isParentFieldDisabled, level, isReadOnly, }) {
    const levels = tableProperties.levels;
    const contextNode = levels ? levels[level].node : tableProperties.node;
    const isTableReadOnly = isReadOnly;
    return {
        columnId,
        contextNode,
        elementId: columnId,
        fieldProperties: { ...columnProperties, wrapper: CellWrapper },
        screenId,
        tableElementId,
        isParentFieldDisabled,
        isTableReadOnly,
        isTree: tableProperties._controlObjectType === FieldKey.Tree,
    };
}
export const addStaticColumns = ({ accessBindings, activeOptionsMenuItem, columns = [], contextType, fieldProperties, hasFloatingFilters, isParentFieldDisabled, level, lookupSelectionMode, screenId, tableElementId, }) => {
    columns.unshift(getValidationColumnConfiguration(screenId, () => fieldProperties.columns || []));
    if (fieldProperties.canSelect && lookupSelectionMode !== 'single') {
        const node = fieldProperties._controlObjectType === FieldKey.Table
            ? String(fieldProperties.node)
            : String(getPagePropertiesFromPageDefinition(getPageDefinitionFromState(screenId)).node);
        columns.unshift(getRowSelectColumnConfiguration({
            screenId,
            tableElementId,
            isParentFieldDisabled,
            node,
            fieldProperties,
            contextType,
            activeOptionsMenuItem,
        }));
    }
    if (fieldProperties.hasLineNumbers) {
        columns.unshift(getLineNumberColumnConfiguration(screenId));
    }
    const inlineActions = (fieldProperties.inlineActions ?? []).filter(a => {
        const accessRule = getElementAccessStatusWithoutId(accessBindings || {}, a.access);
        return accessRule === 'authorized';
    });
    const dropdownActions = (fieldProperties.dropdownActions ?? []).filter(a => {
        if (a === MENU_SEPARATOR) {
            return true;
        }
        const accessRule = getElementAccessStatusWithoutId(accessBindings || {}, a.access);
        return accessRule === 'authorized';
    });
    if (inlineActions.length > 0 ||
        dropdownActions.length > 0 ||
        fieldProperties.canAddNewLine === true ||
        hasFloatingFilters) {
        columns.push(getDropdownActionsColumnConfiguration(screenId, tableElementId, level, CellWrapper, isParentFieldDisabled, inlineActions, dropdownActions, hasFloatingFilters));
    }
};
export const INTERNAL_COLUMN_IDS = [
    COLUMN_ID_ROW_SELECTION,
    COLUMN_ID_LINE_NUMBER,
    COLUMN_ID_VALIDATIONS,
    COLUMN_ID_AUTO_COLUMN,
    COLUMN_ID_ROW_ACTIONS,
];
//# sourceMappingURL=ag-grid-column-config.js.map