import { get } from 'lodash';
import { getReferencePath, getReferenceValueField } from '../component/field/reference/reference-utils';
import { ReadonlyFieldControlObject } from '../component/readonly-field-control-object';
import { FieldKey } from '../component/types';
import { GraphQLTypes } from '../types';
import { xtremConsole } from './console';
import { convertDeepBindToPath, convertDeepBindToPathNotNull } from './nested-field-utils';
import { isDevMode } from './window';
export function getNestedFieldElementId(field) {
    if (!field) {
        return null;
    }
    if (field.type === FieldKey.Reference || field.type === FieldKey.MultiReference) {
        const fieldProperties = field.properties;
        const valueField = getReferenceValueField(fieldProperties)?.split('.').join('__');
        const helperTextField = fieldProperties.helperTextField;
        if (!fieldProperties.bind || (!valueField && !helperTextField)) {
            throw new Error(`Missing information for the following reference field: ${JSON.stringify(field)}`);
        }
        return `${convertDeepBindToPathNotNull(fieldProperties.bind)}__${convertDeepBindToPathNotNull(valueField || String(helperTextField))}`;
    }
    return convertDeepBindToPath(field.properties.bind) ?? null;
}
export const filterFields = (controlObjects) => Object.keys(controlObjects)
    .filter(key => controlObjects[key] instanceof ReadonlyFieldControlObject)
    .reduce((result, key) => ({ ...result, [key]: controlObjects[key] }), {});
export function getValueTypes(controlObjects, uiComponentProperties) {
    return Object.keys(controlObjects).reduce((result, key) => {
        if (controlObjects[key] instanceof ReadonlyFieldControlObject) {
            result[uiComponentProperties?.[key]?.bind ?? key] = controlObjects[key];
        }
        return result;
    }, {});
}
export const setDefaultProperties = (decoratorProperties, defaultProperties, controlObjectType) => {
    Object.keys(defaultProperties)
        .filter(key => decoratorProperties[key] === undefined)
        .forEach(key => {
        decoratorProperties[key] = defaultProperties[key];
    });
    decoratorProperties._controlObjectType = controlObjectType || decoratorProperties._controlObjectType;
    return decoratorProperties;
};
const hasParentInHierarchy = (screen, controlObject, elementId) => {
    const parentGetter = controlObject.parent;
    if (parentGetter) {
        const parent = parentGetter.apply(screen);
        if (parent?.id === elementId) {
            return true;
        }
        if (parent?.parent) {
            return hasParentInHierarchy(screen, parent, elementId);
        }
    }
    return false;
};
export const getContainerChildFields = (screen, controlObjects, elementId) => controlObjects.filter(controlObject => controlObject.id === elementId || hasParentInHierarchy(screen, controlObject, elementId));
export const getAllParents = (screen, fields) => {
    return fields.reduce((reduced, field) => {
        const parentGetter = field.parent;
        if (parentGetter) {
            const parent = parentGetter.apply(screen);
            return reduced.concat([parent, ...getAllParents(screen, [parent])]);
        }
        return reduced;
    }, []);
};
export const getPropertyScalarType = (nodeTypes, propertyType, fieldType, valueField) => {
    // If we can't find a scalar type for the field, we prevent it from being displayed in the filter manager
    const selectFieldScalarType = fieldType === FieldKey.Select ? GraphQLTypes.Enum : null;
    const matchingScalarType = Object.keys(GraphQLTypes).find(typeKey => GraphQLTypes[typeKey] === propertyType);
    const referenceFieldPropertyType = valueField && nodeTypes[propertyType]
        ? get(nodeTypes[propertyType].properties.type, getReferencePath(valueField))
        : null;
    return selectFieldScalarType || matchingScalarType || referenceFieldPropertyType;
};
export const handleChange = (bind, value, // TODO Type this properly
setFieldValue, validate, onChange) => {
    setFieldValue(bind, value)
        .then(() => {
        let validationPromise = Promise.resolve(undefined);
        if (validate) {
            validationPromise = validate(bind, value);
        }
        else if (isDevMode()) {
            xtremConsole.info(`Validation is being skipped on ${bind} since it does not provide a validate handler`);
        }
        if (onChange && validationPromise) {
            // eslint-disable-next-line no-console
            validationPromise.then(() => onChange(value)).catch(xtremConsole.error);
        }
        else if (isDevMode()) {
            xtremConsole.info(`Change event cannot be triggered on ${bind} since it does not provide a change handler`);
        }
    })
        .catch(err => {
        xtremConsole.error(`Failed to set the value of ${bind}`, err);
    });
};
export const normalizeUnderscoreBind = (bind) => bind.split('__')[0];
export const findColumnDefinitionByBind = (columns, bind) => columns.find(c => getNestedFieldElementId(c) === bind || getNestedFieldElementId(c).startsWith(`${bind}__`));
export const filterColumnDefinitionByBind = (columns, bind) => columns.filter(c => getNestedFieldElementId(c) === bind || getNestedFieldElementId(c).startsWith(`${bind}__`));
//# sourceMappingURL=abstract-fields-utils.js.map