import { DateValue, datePropertyValueToDateString } from '@sage/xtrem-date-time';
import { get } from 'lodash';
import { getReferenceValueField } from '../component/field/reference/reference-utils';
import { FieldKey } from '../component/types';
import { getStore } from '../redux';
import { findColumnDefinitionByBind, getContainerChildFields } from '../utils/abstract-fields-utils';
import { getElementAccessStatus } from '../utils/access-utils';
import { xtremConsole } from '../utils/console';
import { dateToString } from '../utils/formatters';
import { isHiddenOrDisabledInLayout } from '../utils/layout-utils';
import { resolveByValue } from '../utils/resolve-value-utils';
import { checkIfPageIsLoaded } from '../utils/state-utils';
import { splitValueToMergedValue } from '../utils/transformers';
import { isNumber } from '../utils/type-utils';
import { isDevMode } from '../utils/window';
import { CollectionValue } from './collection-data-service';
import { localize } from './i18n-service';
import { getScreenElement } from './screen-base-definition';
export const CHILDREN_VALIDATION_ERROR_MESSAGE = () => localize('@sage/xtrem-ui/validation-error-of-children', 'Validation error in children records');
export const CHILDREN_VALIDATION_RULE = () => 'children-records';
export const executeValidationRulesOnField = async ({ screenId, elementId, fieldProperties, value, rowValue: inputRowValue, columnId, recordId, }) => {
    const rowValue = inputRowValue ? splitValueToMergedValue(inputRowValue) : inputRowValue;
    // Fields Validation
    const noErrors = [
        {
            elementId,
            screenId,
            message: '',
            validationRule: '',
            columnId,
            recordId,
        },
    ];
    // If the field properties are not available, then no validation can be executed.
    if (!fieldProperties) {
        return noErrors;
    }
    const titleOrId = resolveByValue({
        screenId,
        propertyValue: fieldProperties.title,
        skipHexFormat: true,
        rowValue: undefined,
        fieldValue: undefined,
    }) ||
        columnId ||
        elementId;
    const isNotZeroValue = resolveByValue({
        screenId,
        propertyValue: fieldProperties.isNotZero,
        skipHexFormat: true,
        rowValue: undefined, // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
        fieldValue: rowValue, // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
    });
    const isMandatoryValue = resolveByValue({
        screenId,
        propertyValue: fieldProperties.isMandatory,
        skipHexFormat: true,
        rowValue: undefined, // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
        fieldValue: rowValue, // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
    });
    const isNullReference = !(value instanceof Array) &&
        fieldProperties._controlObjectType === FieldKey.Reference &&
        get(value, getReferenceValueField(fieldProperties), null) === null;
    // IsMandatory Validation
    if ((isMandatoryValue || isNotZeroValue) &&
        ((!value && value !== 0) || // Numeric field values
            isNullReference ||
            (value instanceof Array && value.length === 0)) // Array field values (multi reference, multi select)
    ) {
        return [
            {
                elementId,
                screenId,
                message: localize('@sage/xtrem-ui/field-mandatory', '{{0}} is mandatory', [titleOrId]),
                validationRule: 'isMandatory',
                columnId,
                recordId,
            },
        ];
    }
    // isNotZero validation
    if (isNotZeroValue && parseFloat(value) === 0) {
        return [
            {
                elementId,
                screenId,
                message: localize('@sage/xtrem-ui/field-non-zero', '{{0}} cannot be zero', [titleOrId]),
                validationRule: 'isNotZero',
                columnId,
                recordId,
            },
        ];
    }
    const minValidationValue = resolveByValue({
        screenId,
        propertyValue: fieldProperties.min,
        skipHexFormat: true,
        rowValue: undefined, // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
        fieldValue: rowValue, // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
    });
    // Numeric min-max validation
    if (isNumber(minValidationValue) && value < minValidationValue) {
        return [
            {
                elementId,
                screenId,
                message: localize('@sage/xtrem-ui/field-minimum-value', '{{0}} minimum value is {{1}}', [
                    titleOrId,
                    minValidationValue,
                ]),
                validationRule: 'min',
                columnId,
                recordId,
            },
        ];
    }
    const maxValidationValue = resolveByValue({
        screenId,
        propertyValue: fieldProperties.max,
        skipHexFormat: true,
        rowValue: undefined, // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
        fieldValue: rowValue, // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
    });
    if (isNumber(maxValidationValue) && value > maxValidationValue) {
        return [
            {
                elementId,
                screenId,
                message: localize('@sage/xtrem-ui/field-maximum-value', '{{0}} maximum value is {{1}}', [
                    titleOrId,
                    maxValidationValue,
                ]),
                validationRule: 'max',
                columnId,
                recordId,
            },
        ];
    }
    // Date min-max date validation
    const minDateValidationValue = resolveByValue({
        screenId,
        propertyValue: fieldProperties.minDate,
        skipHexFormat: true,
        rowValue: undefined, // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
        fieldValue: rowValue, // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
    });
    if (minDateValidationValue &&
        value &&
        DateValue.parse(datePropertyValueToDateString(value)).compare(DateValue.parse(datePropertyValueToDateString(minDateValidationValue))) < 0) {
        const minDate = minDateValidationValue;
        const stringMinDate = dateToString(minDate, localize('@sage/xtrem-ui/date-format', 'DD/MM/YYYY'));
        return [
            {
                elementId,
                screenId,
                message: localize('@sage/xtrem-ui/field-minimum-date-value', '{{0}} minimum date is {{1}}', [
                    titleOrId,
                    stringMinDate,
                ]),
                validationRule: 'minDate',
                columnId,
                recordId,
            },
        ];
    }
    const maxDateValidationValue = resolveByValue({
        screenId,
        propertyValue: fieldProperties.maxDate,
        skipHexFormat: true,
        rowValue: undefined, // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
        fieldValue: rowValue, // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
    });
    if (maxDateValidationValue &&
        value &&
        DateValue.parse(datePropertyValueToDateString(value)).compare(DateValue.parse(datePropertyValueToDateString(maxDateValidationValue))) > 0) {
        const maxDate = maxDateValidationValue;
        const stringMaxDate = dateToString(maxDate, localize('@sage/xtrem-ui/date-format', 'DD/MM/YYYY'));
        return [
            {
                elementId,
                screenId,
                message: localize('@sage/xtrem-ui/field-maximum-date-value', '{{0}} maximum date is {{1}}', [
                    titleOrId,
                    stringMaxDate,
                ]),
                validationRule: 'maxDate',
                columnId,
                recordId,
            },
        ];
    }
    // Text field max-min length validation
    const minLengthValue = resolveByValue({
        screenId,
        propertyValue: fieldProperties.minLength,
        skipHexFormat: true,
        rowValue: undefined, // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
        fieldValue: rowValue, // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
    });
    if (minLengthValue && String(value) && String(value).length < minLengthValue) {
        return [
            {
                elementId,
                screenId,
                message: localize('@sage/xtrem-ui/field-minimum-length-value', '{{0}} minimum length is {{1}}', [
                    titleOrId,
                    minLengthValue,
                ]),
                validationRule: 'minLength',
                columnId,
                recordId,
            },
        ];
    }
    const maxLengthValue = resolveByValue({
        screenId,
        propertyValue: fieldProperties.maxLength,
        skipHexFormat: true,
        rowValue: undefined, // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
        fieldValue: rowValue, // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
    });
    if (maxLengthValue && value && value.length > maxLengthValue) {
        return [
            {
                elementId,
                screenId,
                message: localize('@sage/xtrem-ui/field-maximum-length-value', '{{0}} maximum length is {{1}}', [
                    titleOrId,
                    maxLengthValue,
                ]),
                validationRule: 'maxLength',
                columnId,
                recordId,
            },
        ];
    }
    const minItems = resolveByValue({
        screenId,
        propertyValue: fieldProperties.minItems,
        skipHexFormat: true,
        rowValue: undefined, // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
        fieldValue: rowValue, // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
    });
    if (minItems != null && minItems > (value?.length ?? 0)) {
        return [
            {
                elementId,
                screenId,
                message: localize('@sage/xtrem-ui/field-min-items-value', '{{0}} the minimum number of selectable items is {{1}}', [titleOrId, minItems]),
                validationRule: 'minItems',
                columnId,
                recordId,
            },
        ];
    }
    const maxItems = resolveByValue({
        screenId,
        propertyValue: fieldProperties.maxItems,
        skipHexFormat: true,
        rowValue: undefined, // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
        fieldValue: rowValue, // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
    });
    if (maxItems != null && maxItems < (value?.length ?? 0)) {
        return [
            {
                elementId,
                screenId,
                message: localize('@sage/xtrem-ui/field-max-items-value', '{{0}} the maximum number of selectable items is {{1}}', [titleOrId, maxItems]),
                validationRule: 'maxItems',
                columnId,
                recordId,
            },
        ];
    }
    if (value instanceof CollectionValue) {
        if (value.hasDirtyPhantomRows()) {
            return [
                {
                    elementId,
                    screenId,
                    message: localize('@sage/xtrem-ui/dirty-phantom-row-validation-error-message', 'You need to validate all lines so that you can save the document.'),
                    validationRule: 'dirtyPhantomRow',
                },
            ];
        }
        if (fieldProperties.isTransient) {
            await value.validate();
        }
        const result = await value.validateField({ skipDispatch: true, withoutServerErrors: true });
        if (result.length > 0) {
            return result;
        }
    }
    const validation = fieldProperties.validation;
    if (!validation) {
        return noErrors;
    }
    // TODO: XT-638 ADD THE NESTED VALIDATION EXECUTION HERE
    // Check if there are fieldProperties.columns and then call this method recursively
    // Regexp Validation
    if (validation instanceof RegExp) {
        if (!validation.test(String(value))) {
            return [
                {
                    elementId,
                    screenId,
                    message: localize('@sage/xtrem-ui/field-not-valid', '{{0}} is not valid', [titleOrId]),
                    validationRule: 'validation',
                    columnId,
                    recordId,
                },
            ];
        }
        return noErrors;
    }
    // Custom function validation
    try {
        const message = await resolveByValue({
            screenId,
            propertyValue: fieldProperties.validation,
            rowValue,
            fieldValue: value instanceof CollectionValue ? value.getFormattedActiveRecords() : value,
            skipHexFormat: true,
        });
        if (message) {
            return [
                {
                    elementId,
                    screenId,
                    message,
                    validationRule: 'validation',
                    columnId,
                    recordId,
                },
            ];
        }
    }
    catch (error) {
        if (isDevMode()) {
            xtremConsole.error(`${screenId} - ${elementId} could not be validated to due an unexpected error:`);
            xtremConsole.error(error);
        }
        return [
            {
                elementId,
                screenId,
                message: error.message ? error.message : String(error),
                validationRule: 'validation',
                columnId,
                recordId,
            },
        ];
    }
    return [
        {
            elementId,
            screenId,
            message: '',
            validationRule: '',
            columnId,
            recordId,
        },
    ];
};
export const checkNestedRecordValidationStatus = async ({ rowValue, columnDefinitions, columnsToValidate, screenId, elementId, recordId, }) => {
    const validationResults = await Promise.all(Array.from(columnsToValidate).map(bind => executeValidationRulesOnField({
        screenId,
        elementId,
        columnId: bind,
        fieldProperties: findColumnDefinitionByBind(columnDefinitions, bind)
            ?.properties,
        recordId,
        rowValue,
        value: get(rowValue, bind.replace('__', '.')),
    })));
    return validationResults
        .flat()
        .filter(r => !!r.message)
        .reduce((prevValue, currentValue) => {
        return {
            ...prevValue,
            [currentValue.columnId]: currentValue,
        };
    }, {});
};
/**
 * The function checks all validation rules.
 *
 * @param screenElement
 * @param elementId
 * @param componentProperties
 * @param value
 */
export const checkValidationStatus = async (screenElement, elementId, componentProperties, value, accessBindings, nodeTypes, dataTypes, contextNode) => {
    const screenId = screenElement._pageMetadata.screenId;
    // Fields Validation
    const result = [
        {
            elementId,
            screenId,
            message: '',
            validationRule: '',
        },
    ];
    const fieldProperties = componentProperties[elementId];
    if (!fieldProperties) {
        return result;
    }
    // If the user doesn't have to access to edit the field, then we don't validate it.
    const accessStatus = getElementAccessStatus({
        accessBindings,
        bind: elementId,
        elementProperties: componentProperties,
        contextNode,
        nodeTypes,
        dataTypes,
    });
    if (accessStatus === 'unavailable' || accessStatus === 'unauthorized') {
        return result;
    }
    if (isHiddenOrDisabledInLayout({
        componentProperties,
        screenId,
        elementId,
        accessBindings,
        contextNode,
        dataTypes,
        nodeTypes,
    })) {
        return result;
    }
    return executeValidationRulesOnField({ value, fieldProperties, screenId, elementId });
};
export const getSectionValidationMessage = (screenId, sectionId, state) => {
    const screen = state.screenDefinitions[screenId];
    const screenElement = getScreenElement(screen);
    const controlObjects = state.screenDefinitions[screenId].metadata.controlObjects;
    const children = getContainerChildFields(screenElement, Object.values(controlObjects), sectionId);
    const totalErrors = children.flatMap(c => screen.errors[c.id]).filter(c => !!c).length;
    if (totalErrors === 0) {
        return null;
    }
    return totalErrors > 1
        ? localize('@sage/xtrem-ui/validation-errors-total', '{{0}} errors', [totalErrors])
        : localize('@sage/xtrem-ui/validation-error-total', '1 error');
};
export const getValidationState = (screenId, elementId, state = getStore().getState()) => {
    checkIfPageIsLoaded(screenId, state);
    const screen = state.screenDefinitions[screenId];
    return (getContainerChildFields(getScreenElement(screen), Object.values(screen.metadata.controlObjects), elementId)
        .map(co => screen.errors[co.id])
        .filter(e => !!e).length === 0);
};
//# sourceMappingURL=validation-service.js.map