import { ActionType, getStore } from '../redux';
import { setPageValidationErrors } from '../redux/actions';
import { triggerScreenEvent } from '../utils/events';
import { transformServerErrorToToast } from '../utils/server-error-transformer';
import { getPageDefinitionFromState, isScreenDefinitionDirty } from '../utils/state-utils';
import { parseScreenValues } from '../utils/transformers';
import { getDialogApi } from './dialog-api';
import { notifyConsumerAboutDirtyStatus, onPageDirtyChange } from './dirty-state-service';
import { GraphQLApi } from './graphql-api';
import { localize } from './i18n-service';
import loader from './loader';
import { removeToasts, showToast } from './toast-service';
import { Router } from './router';
import { SoundApi } from './sound-api';
import * as storageService from './storage-service';
import { commitTransaction, isTransactionInProgress, setFieldValue } from './transactions-service';
export class BaseDeveloperApi {
    constructor(screenId) {
        this.router = new Router(screenId);
        this.dialog = getDialogApi(screenId, getStore());
        this.loader = loader;
        this.sound = new SoundApi();
        this.showToast = (content, options) => showToast(content, { ...options, language: 'markdown' });
        this.removeToasts = removeToasts;
    }
    /**
     * Returns the unique ID (email address) of the user that is currently uses the application
     */
    get username() {
        const state = getStore().getState();
        return state.applicationContext?.login || null;
    }
    /**
     * The session username can be used if the application has a username that is different from the login user.
     */
    get userCode() {
        const state = getStore().getState();
        return state.applicationContext?.userCode || null;
    }
    get locale() {
        const state = getStore().getState();
        return state.applicationContext?.locale || null;
    }
    /** Cross-page storage object that preserves the content for the session */
    get storage() {
        return Object.seal(storageService);
    }
}
export class DeveloperApi extends BaseDeveloperApi {
    constructor(screenBase) {
        super(screenBase._pageMetadata.screenId);
        this.screenBase = screenBase;
        this.graph = new GraphQLApi(screenBase);
    }
    /**
     * Returns a new object containing the values of all the page's non-transient fields.
     * It is a heavy operation that serializes all values of the page. It should only be used
     * when you need the value of most of the fields in a server friendly way.
     */
    get values() {
        return parseScreenValues(this.screenBase._pageMetadata.screenId);
    }
    set values(boundValues) {
        Object.keys(boundValues).forEach(key => {
            setFieldValue(this.screenBase._pageMetadata.screenId, key, boundValues[key]);
        });
    }
    get businessActions() {
        const extensionActions = this.screenBase._pageMetadata.businessActionsExtensionsThunk.flatMap(e => e.apply(this.screenBase));
        return this.screenBase._pageMetadata.businessActionsThunk
            ? Object.seal([
                ...this.screenBase._pageMetadata.businessActionsThunk.apply(this.screenBase),
                ...extensionActions,
            ])
            : [];
    }
    setPageClean() {
        const store = getStore();
        const pageMetadata = this.screenBase._pageMetadata;
        const screenId = pageMetadata.screenId;
        store.dispatch({ type: ActionType.SetPageClean, value: screenId });
        // A fresh copy of the updated state is needed.
        notifyConsumerAboutDirtyStatus(store.getState());
        onPageDirtyChange(store.getState(), screenId, false);
        triggerScreenEvent(screenId, 'onDirtyStateUpdated', false);
    }
    /** Whether the page currently contains any dirty fields */
    get isDirty() {
        const state = getStore().getState();
        return isScreenDefinitionDirty(getPageDefinitionFromState(this.screenBase._pageMetadata.screenId, state));
    }
    isTransactionInProgress() {
        return isTransactionInProgress(this.screenBase._pageMetadata.screenId);
    }
    /** Update the screen with the changes that were applied by the functional code. */
    commitValueAndPropertyChanges() {
        return new Promise(resolve => {
            commitTransaction(this.screenBase._pageMetadata.screenId);
            // An immediate would probably do too, but just be sure that all rendering can
            setTimeout(resolve, 50);
        });
    }
    finish(result) {
        const state = getStore().getState();
        const screenId = this.screenBase._pageMetadata.screenId;
        const screenDefinition = state.screenDefinitions[screenId];
        screenDefinition.onFinish?.(result);
    }
    async processServerErrors(error) {
        const transformedError = await transformServerErrorToToast(error, this.screenBase);
        if (transformedError.validationErrors && Object.keys(transformedError.validationErrors).length > 0) {
            const store = getStore();
            const dispatch = store.dispatch;
            await dispatch(setPageValidationErrors(this.screenBase._pageMetadata.screenId, transformedError.validationErrors));
        }
        if (transformedError.globalError) {
            throw transformedError.globalError;
        }
    }
    isServiceOptionEnabled(serviceOptionName) {
        const state = getStore().getState();
        return state.serviceOptions[serviceOptionName] || false;
    }
}
// eslint-disable-next-line @typescript-eslint/no-unused-vars
export class ScreenBase {
    constructor() {
        this.$standardDeleteButton = localize('@sage/xtrem-ui/crud-delete-record-button', 'Delete');
        this.$standardCancelButton = localize('@sage/xtrem-ui/crud-cancel', 'Cancel');
        /**
         * Confirmation dialog title for the standard CRUD delete platform action.
         * Override this property to change the dialog title.
         * */
        this.$standardDeletePromptTitle = localize('@sage/xtrem-ui/crud-delete-record-warning-title', 'Delete record');
        /**
         * Confirmation dialog message content for the standard CRUD delete platform action.
         * Override this property to change the dialog title.
         * */
        this.$standardDeletePromptMessage = localize('@sage/xtrem-ui/crud-delete-record-warning-message', 'You are about to delete this record.');
        /**
         * Toast content for the standard CRUD save platform action for successfully updating an existing record.
         * Override this property to change the content of the toast.
         * */
        this.$standardSaveSuccessMessage = localize('@sage/xtrem-ui/crud-update-success', 'Record has been updated successfully.');
        /**
         * Toast content for the standard CRUD save platform action for successfully creating an existing record.
         * Override this property to change the content of the toast.
         * */
        this.$standardCreateSuccessMessage = localize('@sage/xtrem-ui/crud-create-success', 'Record has been created successfully.');
        /**
         * Toast content for the standard CRUD save platform action for save failure.
         * Override this property to change the content of the toast.
         * */
        this.$standardSaveFailedMessage = localize('@sage/xtrem-ui/crud-save-failed', 'Failed to save record.');
        /**
         * Toast content for the standard CRUD duplicate platform action for duplication failure.
         * Override this property to change the content of the toast.
         * */
        this.$standardDuplicateFailedMessage = localize('@sage/xtrem-ui/crud-duplicate-failed', 'Failed to duplicate record.');
        // eslint-disable-next-line no-proto
        this._pageMetadata = this.__proto__.constructor._pageMetadata;
    }
    /**
     * This function serializes the pages values into a format that is ready for server operations. This function is used
     * by the standard CRUD actions, by overriding this function, the application developers can control what values are
     * sent to the server.
     *
     * @returns serializable data which will be used for server interactions
     */
    getSerializedValues() {
        return parseScreenValues(this._pageMetadata.screenId, false);
    }
}
//# sourceMappingURL=screen-base.js.map