import { actions, getStore } from '../redux';
import { getPageDefinitionFromState, isScreenDefinitionDirty } from '../utils/state-utils';
import { errorDialog } from './dialog-service';
import { openDirtyPageConfirmationDialog } from './dirty-state-service';
import { notifyConsumerOnError } from './telemetry-service';
import { findNavPanelRecordLocation } from '../redux/actions/router-actions';
import { NEW_PAGE, REG_EXP_INTERNAL_URL_PATTERN, REG_EXP_URL_PATTERN } from '../utils/constants';
import { wipeCache } from './graphql-utils';
export class Router {
    constructor(screenId) {
        this.screenId = screenId;
    }
    isScreenDirty() {
        // Special screen name starts with $ symbol, such as $dashboards, in this case we don't need to check if they are dirty
        return (!this.screenId.startsWith('$') &&
            isScreenDefinitionDirty(getStore().getState().screenDefinitions[this.screenId]));
    }
    async wrapNavigationMethod(skipDirtyCheck, navigationImplementation) {
        const storeDispatch = getStore().dispatch;
        if (!skipDirtyCheck && this.isScreenDirty()) {
            return new Promise((resolve, reject) => openDirtyPageConfirmationDialog(this.screenId).then(() => storeDispatch(navigationImplementation())
                .then(resolve)
                .catch(error => {
                errorDialog(this.screenId, 'Navigation error', error?.message);
                notifyConsumerOnError(error);
                reject();
            })));
        }
        return storeDispatch(navigationImplementation());
    }
    async goTo(path, queryParameters = {}, skipDirtyCheck = false) {
        await this.wrapNavigationMethod(skipDirtyCheck, () => actions.navigate(path, queryParameters));
    }
    goToExternal(link) {
        if (!link.match(REG_EXP_URL_PATTERN) && !link.match(REG_EXP_INTERNAL_URL_PATTERN)) {
            throw new Error('Unexpected link format.');
        }
        else {
            window.open(link, '_blank', 'noopener=true');
        }
    }
    async refresh(skipDirtyCheck = false) {
        return this.wrapNavigationMethod(skipDirtyCheck, () => {
            const pageDefinition = getPageDefinitionFromState(this.screenId);
            const recordId = pageDefinition?.queryParameters?._id;
            return actions.selectRecord(this.screenId, recordId ? String(recordId) : null, true);
        });
    }
    async emptyPage(skipDirtyCheck = false) {
        return this.wrapNavigationMethod(skipDirtyCheck, () => actions.selectRecord(this.screenId, NEW_PAGE));
    }
    async closeRecord(skipDirtyCheck = false) {
        return this.wrapNavigationMethod(skipDirtyCheck, () => actions.selectRecord(this.screenId, null));
    }
    async selectRecord(recordId, skipDirtyCheck = false) {
        return this.wrapNavigationMethod(skipDirtyCheck, () => actions.selectRecord(this.screenId, String(recordId), false));
    }
    async firstRecord(skipDirtyCheck = false) {
        return this.wrapNavigationMethod(skipDirtyCheck, () => actions.selectFirstRecord(this.screenId));
    }
    async nextRecord(skipDirtyCheck = false) {
        if (await this.hasNextRecord()) {
            return this.wrapNavigationMethod(skipDirtyCheck, () => actions.selectNextRecord(this.screenId));
        }
        return undefined;
    }
    async previousRecord(skipDirtyCheck = false) {
        if (await this.hasPreviousRecord()) {
            return this.wrapNavigationMethod(skipDirtyCheck, () => actions.selectPreviousRecord(this.screenId));
        }
        return undefined;
    }
    async hasPreviousRecord() {
        try {
            const nextRecordId = await findNavPanelRecordLocation(this.screenId, getStore().getState(), -1);
            return nextRecordId !== null;
        }
        catch {
            return false;
        }
    }
    async hasNextRecord() {
        try {
            const nextRecordId = await findNavPanelRecordLocation(this.screenId, getStore().getState(), 1);
            return nextRecordId !== null;
        }
        catch {
            return false;
        }
    }
    async hardRefresh() {
        await wipeCache();
        window.location.reload();
    }
    async goHome(skipDirtyCheck = false) {
        return this.wrapNavigationMethod(skipDirtyCheck, () => actions.goHome());
    }
}
export const getRouter = (screenId) => new Router(screenId);
//# sourceMappingURL=router.js.map