import lodash, { isEqual } from 'lodash';
import { mergeColumnPropertyDefinitions } from '../component/abstract-decorator-utils';
import { navigationPanelId } from '../component/container/navigation-panel/navigation-panel-types';
import { MENU_SEPARATOR } from '../component/container/page/page-types';
import { FieldKey } from '../component/types';
import { getStore } from '../redux';
import { refreshNavigationPanel } from '../redux/actions/navigation-panel-actions';
import { getImagePlaceholderValue } from '../utils/nested-field-utils';
import { resolveByValue } from '../utils/resolve-value-utils';
import { getNavigationPanelState, getPagePropertiesFromPageDefinition } from '../utils/state-utils';
import { getNestedFieldArrayFromCardDefinition, getNestedFieldArrayFromCardExtensionDefinition, } from '../utils/transformers';
import { CollectionValue } from './collection-data-service';
import { CollectionFieldTypes } from './collection-data-types';
import { addCustomColumnToNestedFields } from './customization-service';
import { ReadOnlyGraphQLApi } from './graphql-api';
import { getNavigationPanelOrderByDefinition } from './graphql-query-builder';
import { localize } from './i18n-service';
import { getRouter } from './router';
import { getScreenElement } from './screen-base-definition';
import * as storageService from './storage-service';
import { applyDefaultValuesOnNestedField } from '../utils/data-type-utils';
import { QUERY_PARAM_CUSTOM_MAIN_LIST_FILTER } from '../utils/constants';
const sealedStorageService = Object.seal(storageService);
export function createMySelectedDataOptionItem(pageDefinition) {
    return {
        title: localize('@sage/xtrem-ui/navigation-panel-my-view', 'My selected data'),
        graphQLFilter: JSON.parse(String(pageDefinition.queryParameters?.[QUERY_PARAM_CUSTOM_MAIN_LIST_FILTER] || {})),
    };
}
const mapMainListActionToTableRecordAction = (screenId) => (a) => {
    const originalOnClick = a.onClick;
    const onClick = async function OverriddenOnClick(...args) {
        await originalOnClick.apply(this, args);
        const dispatch = getStore().dispatch;
        if (a.refreshesMainList === 'record') {
            await getNavigationPanelState(screenId)?.value?.refreshRecord({ recordId: args[0] });
        }
        if (a.refreshesMainList === 'list') {
            await dispatch(refreshNavigationPanel(screenId));
        }
    };
    return { ...a, onClick };
};
export const resolveNavigationPanelOptionFilter = (pageDefinition) => (optionItem) => {
    if (!optionItem) {
        return undefined;
    }
    if (lodash.isFunction(optionItem.graphQLFilter)) {
        const graphQLFilter = optionItem.graphQLFilter.apply({}, [storageService, pageDefinition.queryParameters]);
        return { ...optionItem, graphQLFilter };
    }
    return optionItem;
};
export const resolveOptionsMenu = async (pageDefinition, optionsMenu, applicationContext) => {
    const options = [];
    if (pageDefinition.queryParameters?.[QUERY_PARAM_CUSTOM_MAIN_LIST_FILTER]) {
        // If the "_filter" query param is set, we parse it and use it as a filter option
        options.push(createMySelectedDataOptionItem(pageDefinition));
    }
    if (optionsMenu && lodash.isFunction(optionsMenu)) {
        const resolvedMenu = await optionsMenu(new ReadOnlyGraphQLApi(), sealedStorageService, pageDefinition.queryParameters, applicationContext?.login || '', applicationContext?.userCode || '');
        options.push(...resolvedMenu
            .map(resolveNavigationPanelOptionFilter(pageDefinition))
            .filter(v => !!v));
    }
    else if (optionsMenu) {
        options.push(...optionsMenu
            .map(resolveNavigationPanelOptionFilter(pageDefinition))
            .filter(v => !!v));
    }
    // In case there is no option menu defined for the page but we added a filter based on the query parameters, the we need to add an "All" option too
    if (pageDefinition.queryParameters?.[QUERY_PARAM_CUSTOM_MAIN_LIST_FILTER] && options.length === 1) {
        options.push({
            title: localize('@sage/xtrem-ui/navigation-panel-no-filter', 'All'),
            graphQLFilter: {},
        });
    }
    return options;
};
export const getMergedOptionMenu = async (pageDefinition, navigationPanel, applicationContext) => {
    let result = await resolveOptionsMenu(pageDefinition, navigationPanel.optionsMenu, applicationContext);
    if (navigationPanel.optionMenus) {
        // eslint-disable-next-line no-restricted-syntax
        for (const optionsMenu of navigationPanel.optionMenus) {
            // eslint-disable-next-line no-await-in-loop
            const extensionResult = await resolveOptionsMenu(pageDefinition, optionsMenu, applicationContext);
            result = [...result, ...extensionResult];
        }
    }
    return lodash.uniqWith(result, lodash.isEqual);
};
export const getInitialOptionMenuItem = (pageDefinition, navigationTableProperties) => {
    const screenId = pageDefinition.metadata.screenId;
    const optionMenu = resolveByValue({
        propertyValue: navigationTableProperties.optionsMenu,
        screenId,
        rowValue: null,
        fieldValue: null,
        skipHexFormat: true,
    });
    const savedTableViewMenuItem = pageDefinition.tableViews?.[navigationPanelId]?.$current?.level?.[0].optionsMenuItem;
    // We need to check if the previous saved table view option is still a valid option in the current optionMenu list
    // If not, we set the first optionMenu item as the default one
    const optionItem = savedTableViewMenuItem && optionMenu.find(item => isEqual(item, savedTableViewMenuItem))
        ? savedTableViewMenuItem
        : optionMenu[0];
    return resolveNavigationPanelOptionFilter(pageDefinition)(optionItem);
};
export const onNavigationPanelRowClick = (pageDefinition, navigationPanel, path) => (recordId, rowItem, isModifierKeyPushed = false) => {
    if (isModifierKeyPushed && path) {
        const url = `${path.split('/').slice(0, 3).join('/')}/${btoa(JSON.stringify({ _id: recordId }))}`;
        window.open(url, '_blank');
        return;
    }
    if (navigationPanel.onSelect) {
        // If the onSelect returns true, we consider the event consumed and do not execute the actual navigation event.
        const navigationPrevented = navigationPanel.onSelect.apply(getScreenElement(pageDefinition), [rowItem]);
        if (navigationPrevented) {
            return;
        }
    }
    getRouter(pageDefinition.metadata.screenId).selectRecord(recordId);
};
export const createNavigationTableProperties = async (screenId, pageNode, pageDefinition, nodeTypes, dataTypes, path, applicationContext) => {
    const pageProperties = getPagePropertiesFromPageDefinition(pageDefinition);
    const navigationPanel = pageProperties.navigationPanel;
    if (!navigationPanel) {
        throw new Error(`No navigation panel definition found on ${screenId}.`);
    }
    const optionsMenu = await getMergedOptionMenu(pageDefinition, navigationPanel, applicationContext);
    const listItem = { ...navigationPanel.listItem };
    if (listItem.image) {
        listItem.image.properties = {
            placeholderValue: (recordValue) => getImagePlaceholderValue(recordValue, listItem.title),
            ...listItem.image.properties,
        };
    }
    const orderBy = getNavigationPanelOrderByDefinition(listItem, navigationPanel.orderBy);
    const baseColumns = getNestedFieldArrayFromCardDefinition(listItem);
    const node = String(pageProperties.node);
    const columns = (navigationPanel.extensionListItem || []).reduce((prevValue, currentValue) => mergeColumnPropertyDefinitions(prevValue, getNestedFieldArrayFromCardExtensionDefinition(currentValue)).map(column => {
        applyDefaultValuesOnNestedField(nodeTypes, dataTypes, column, node);
        return column;
    }), baseColumns);
    columns.forEach(column => {
        applyDefaultValuesOnNestedField(nodeTypes, dataTypes, column, node);
    });
    addCustomColumnToNestedFields(node, nodeTypes, columns, navigationPanelId, FieldKey.Table, false, pageDefinition.metadata.customizations);
    const mobileCard = (navigationPanel.extensionListItem || []).reduce((prevValue, currentValue) => ({
        title: currentValue.title || prevValue.title,
        titleRight: currentValue.titleRight || prevValue.titleRight,
        line2: currentValue.line2 || prevValue.line2,
        line2Right: currentValue.line2Right || prevValue.line2Right,
        line3: currentValue.line3 || prevValue.line3,
        line3Right: currentValue.line3Right || prevValue.line3Right,
        image: currentValue.image || prevValue.image,
    }), navigationPanel.listItem);
    /**
     * INFO: Casting the result to an PageActionControlObject[] should be fine, since we make sure
     *       the create actions are arrays in the page decorator.
     */
    const createActions = resolveByValue({
        propertyValue: pageProperties.createAction,
        screenId,
        rowValue: null,
        fieldValue: null,
        skipHexFormat: true,
    });
    return {
        canExport: true,
        canSelect: (navigationPanel.bulkActions ?? []).length > 0,
        canUserHideColumns: true,
        columns,
        emptyStateClickableText: navigationPanel.emptyStateClickableText,
        emptyStateText: navigationPanel.emptyStateText,
        hasSearchBoxMobile: true,
        headerBusinessActions: createActions || undefined,
        isReadOnly: true,
        isTitleHidden: false,
        mobileCard,
        node: pageNode,
        onEmptyStateLinkClick: navigationPanel.onEmptyStateLinkClick,
        onRowClick: onNavigationPanelRowClick(pageDefinition, navigationPanel, path),
        optionsMenu: optionsMenu,
        optionsMenuType: navigationPanel.menuType,
        dropdownActions: navigationPanel.dropdownActions?.map(a => {
            if (a === MENU_SEPARATOR) {
                return a;
            }
            return mapMainListActionToTableRecordAction(screenId)(a);
        }),
        inlineActions: navigationPanel.inlineActions?.map(mapMainListActionToTableRecordAction(screenId)),
        orderBy,
        pageSize: 20,
        title: pageProperties.objectTypePlural,
        canResizeColumns: true,
        cardColor: navigationPanel.cardColor,
        startDateField: navigationPanel.startDateField,
        endDateField: navigationPanel.endDateField,
        minDate: navigationPanel.minDate,
        maxDate: navigationPanel.maxDate,
        isEventMovable: navigationPanel.isEventMovable,
    };
};
export const createNavigationPanelValue = (screenId, navigationPanelTableProperties, initialValues, nodeTypes = getStore().getState().nodeTypes, locale = 'en-US', activeOptionsMenuItem, orderBy) => new CollectionValue({
    activeOptionsMenuItem,
    columnDefinitions: [navigationPanelTableProperties.columns || []],
    elementId: navigationPanelId,
    fieldType: CollectionFieldTypes.NAVIGATION_PANEL,
    filter: [navigationPanelTableProperties.filter],
    hasNextPage: true,
    initialValues,
    isTransient: false,
    locale,
    nodes: [navigationPanelTableProperties.node],
    nodeTypes,
    orderBy: [orderBy || navigationPanelTableProperties.orderBy],
    screenId,
});
//# sourceMappingURL=navigation-panel-service.js.map