import { DateFilterPeriodType, } from '@sage/bms-dashboard';
import { MD5 } from 'crypto-js';
import { isEmpty, escape as lodashEscape, noop, sortBy, uniqBy } from 'lodash';
import * as showdown from 'showdown';
import { getImageUrlFromValue } from '../component/field/image/image-utils';
import { GENERIC_PLATFORM_WIDGET_PREFIX } from '../dashboard/generic-widgets';
import { WidgetType } from '../dashboard/widgets/abstract-widget';
import * as xtremRedux from '../redux';
import { xtremConsole } from '../utils/console';
import { DASHBOARD_SCREEN_ID, PREVIEW_WIDGET_ID } from '../utils/constants';
import { resolveDetailedIcon } from '../utils/detailed-icons-utils';
import { resolveImageUrl } from '../utils/vp-utils';
import { getPlatformStringLiteralsSchemaFragment } from './graphql-query-builder';
import { executeGraphqlQuery } from './graphql-utils';
import { localize } from './i18n-service';
import { processStringLiteralsFromResponse } from './screen-loader-service';
import { executeVisualProcessLink } from './visual-process-service';
export const DASHBOARD_DETAILS_QUERY = {
    _id: true,
    title: true,
    isTemplate: true,
    children: {
        _id: true,
        type: true,
        settings: true,
        x: true,
        y: true,
        h: true,
        w: true,
    },
};
export const fetchCurrentDashboardDefinition = async (group, shouldFetchPlatformLiterals = false) => {
    // Load source code of the page and its extensions from the server
    const query = {
        dashboard: {
            selectedDashboard: {
                ...DASHBOARD_DETAILS_QUERY,
                __args: {
                    group,
                },
            },
            availableDashboards: {
                _id: true,
                title: true,
                __args: {
                    group,
                },
            },
            canEditDashboards: true,
        },
    };
    if (shouldFetchPlatformLiterals) {
        query.strings = getPlatformStringLiteralsSchemaFragment();
    }
    const { data } = await executeGraphqlQuery({
        query,
        endpoint: '/metadata',
    });
    const dashboard = data.dashboard.selectedDashboard;
    if (!dashboard) {
        return {
            dashboard: null,
            stringLiterals: {},
            availableDashboards: null,
            canEditDashboards: data.dashboard.canEditDashboards,
        };
    }
    dashboard.children = sortBy(dashboard.children?.map((c) => ({
        ...c,
        settings: JSON.parse(c.settings),
    })) || [], ['y', 'x']);
    const stringLiterals = shouldFetchPlatformLiterals
        ? processStringLiteralsFromResponse(undefined, data)
        : {};
    const availableDashboards = data.dashboard.availableDashboards;
    return { dashboard, stringLiterals, availableDashboards, canEditDashboards: data.dashboard.canEditDashboards };
};
export const fetchWidgetCategories = async () => {
    const response = await executeGraphqlQuery({
        query: { dashboard: { getWidgetCategories: { key: true, title: true } } },
        endpoint: '/metadata',
    });
    if (Array.isArray(response.data?.dashboard?.getWidgetCategories)) {
        return response.data.dashboard.getWidgetCategories.reduce((prevValue, currentValue) => {
            return { ...prevValue, [currentValue.key]: currentValue.title };
        }, {});
    }
    throw new Error('Failed to load widget categories from the server');
};
export const fetchDashboardTemplates = async (group) => {
    const query = {
        dashboard: {
            templates: {
                __args: {
                    group,
                },
                _id: true,
                title: true,
                description: true,
                listIcon: true,
            },
        },
    };
    const { data } = await executeGraphqlQuery({
        query,
        endpoint: '/metadata',
    });
    return data.dashboard.templates;
};
export const setSelectedDashboard = async (selectedId, group) => {
    const { data } = await executeGraphqlQuery({
        query: {
            mutation: {
                dashboard: {
                    selectDashboard: {
                        __args: {
                            dashboardId: selectedId,
                            group,
                        },
                        ...DASHBOARD_DETAILS_QUERY,
                    },
                },
            },
        },
        endpoint: '/metadata',
    });
    if (data.dashboard.selectDashboard) {
        const children = data.dashboard.selectDashboard.children?.map((c) => ({
            ...c,
            settings: c.settings ? JSON.parse(c.settings) : {},
        }));
        return { ...data.dashboard.selectDashboard, children };
    }
    throw new Error('Failed to select dashboard');
};
export const cloneDashboard = async (dashboardIdToClone) => {
    const { data, error } = await executeGraphqlQuery({
        query: {
            mutation: {
                dashboard: {
                    cloneDashboard: {
                        __args: {
                            dashboardId: dashboardIdToClone,
                        },
                        _id: true,
                        title: true,
                    },
                },
            },
        },
        endpoint: '/metadata',
    });
    if (!error && data.dashboard.cloneDashboard) {
        return data.dashboard.cloneDashboard;
    }
    throw new Error('Failed to clone dashboard');
};
export const deleteDashboard = async (dashboardIdToDelete) => {
    const { error } = await executeGraphqlQuery({
        query: {
            mutation: {
                dashboard: {
                    deleteDashboard: {
                        __args: {
                            dashboardId: dashboardIdToDelete,
                        },
                    },
                },
            },
        },
        endpoint: '/metadata',
    });
    if (error) {
        xtremConsole.error(error);
        throw new Error('Failed to delete dashboard');
    }
};
export const createEmptyDashboard = async (group) => {
    const { data, error } = await executeGraphqlQuery({
        query: {
            mutation: {
                dashboard: {
                    createDashboard: {
                        ...DASHBOARD_DETAILS_QUERY,
                        __args: {
                            group,
                        },
                    },
                },
            },
        },
        endpoint: '/metadata',
    });
    if (!error && data.dashboard.createDashboard) {
        return data.dashboard.createDashboard;
    }
    throw new Error('Failed to create dashboard');
};
export const updateDashboard = async (dashboardDefinition) => {
    const { data, error } = await executeGraphqlQuery({
        query: {
            mutation: {
                dashboard: {
                    updateDashboardLayout: {
                        __args: {
                            dashboardId: dashboardDefinition._id,
                            layout: JSON.stringify(dashboardDefinition),
                        },
                        ...DASHBOARD_DETAILS_QUERY,
                    },
                },
            },
        },
        endpoint: '/metadata',
    });
    if (!error && data.dashboard.updateDashboardLayout) {
        data.dashboard.updateDashboardLayout.children = sortBy(data.dashboard.updateDashboardLayout.children?.map((c) => ({
            ...c,
            settings: JSON.parse(c.settings),
        })) || [], ['y', 'x']);
        return data.dashboard.updateDashboardLayout;
    }
    throw new Error('Failed to update dashboards.');
};
/**
 * Fetches widget list from the server and adds "Others" category to those which doesn't have one.
 * It also adds a _sortValue as the items should be categorized alphabetically, e
 * except the Others, so we cannot sort them just by their label
 * @returns
 */
export const getWidgetList = async (group) => {
    const { data } = await executeGraphqlQuery({
        query: {
            widgets: {
                __args: {
                    filter: group ? { group } : {},
                },
                key: true,
                category: true,
                categoryLabel: true,
                type: true,
                title: true,
                description: true,
            },
        },
        endpoint: '/metadata',
    });
    if (!data.widgets || !Array.isArray(data.widgets)) {
        throw new Error('Failed to fetch widget list.');
    }
    const otherCategory = {
        _id: '_OTHERS',
        title: localize('@sage/xtrem-ui/dashboard-widget-category-others', 'Others'),
        sortValue: Number.MAX_SAFE_INTEGER,
    };
    const uniqueServerCategories = uniqBy(data.widgets.filter((w) => !!w.category).map((w) => ({ _id: w.category, title: w.categoryLabel })), w => w._id);
    const sortedServerCategories = sortBy(uniqueServerCategories, w => w.title).map((category, index) => ({ ...category, sortValue: index + 1 }));
    const categories = [...sortedServerCategories, otherCategory];
    const widgets = data.widgets.map((w) => ({
        _id: w.key,
        type: w.type,
        description: w.description,
        title: w.title,
        category: w.category ? categories.find(c => c._id === w.category) || otherCategory : otherCategory,
    }));
    return { widgets, categories };
};
const updateWidgetOptions = ({ dashboardId, widgetId, group, shouldRefreshData, options = {}, skipUnset = false, }) => {
    const thunkDispatch = xtremRedux.getStore().dispatch;
    const state = xtremRedux.getStore().getState();
    if (!state.loading.widgets[widgetId]) {
        thunkDispatch(xtremRedux.actions.setWidgetOptions(dashboardId, widgetId, group, options));
        if (shouldRefreshData) {
            refreshWidget(widgetId, group, skipUnset);
        }
    }
};
export const refreshWidget = (widgetId, group, skipUnset = false) => {
    const thunkDispatch = xtremRedux.getStore().dispatch;
    thunkDispatch(xtremRedux.actions.loadWidgetData({ widgetId, forceRefetch: true, skipUnset, group }));
};
export const fetchWidgetPage = async ({ queryArgs, widgetId, group, }) => {
    const thunkDispatch = xtremRedux.getStore().dispatch;
    return thunkDispatch(xtremRedux.actions.loadWidgetData({ widgetId, forceRefetch: false, queryArgs, dataMode: 'ADD', group }));
};
const getStaticContentWidgetConfig = (widgetId, widgetDefinition, group) => {
    const staticContentProperties = widgetDefinition.properties;
    const mdContent = resolveWidgetProperty(staticContentProperties.content, group, widgetDefinition.widgetObject || widgetId) || '';
    const converter = new showdown.Converter();
    return {
        type: 'basic',
        data: {
            htmlContent: `<div style="font-size: 14px">${converter.makeHtml(lodashEscape(mdContent))}</div>`,
        },
    };
};
const getContactCardWidgetConfig = (widgetId, widgetDefinition, group) => {
    const contactCardWidgetProperties = widgetDefinition.properties;
    const data = resolveWidgetProperty(contactCardWidgetProperties.content, group, widgetDefinition.widgetObject || widgetId);
    return {
        title: resolveWidgetProperty(contactCardWidgetProperties.title, group, widgetDefinition.widgetObject || widgetId),
        type: 'contact-card',
        data: {
            ...data,
            iconSrc: data?.iconSrc ? resolveDetailedIcon(data.iconSrc) : undefined,
            contacts: data?.contacts ?? {},
            numberOfAddresses: data?.numberOfAddresses ?? 0,
            numberOfContacts: data?.numberOfContacts ?? 0,
        },
    };
};
const createAxesConfig = (widgetDefinition, group) => (s) => {
    const { onClick, tooltipContent } = s;
    return {
        ...s,
        onClick: onClick
            ? ({ record, primaryValue, secondaryValue }) => resolveWidgetProperty(onClick, group, widgetDefinition.widgetObject, record, primaryValue, secondaryValue)
            : undefined,
        tooltipContent: tooltipContent
            ? ({ record, primaryValue, secondaryValue }) => resolveWidgetProperty(tooltipContent, group, widgetDefinition.widgetObject, record, primaryValue, secondaryValue)
            : undefined,
    };
};
const getBarChartWidgetConfig = (widgetId, widgetDefinition, group) => {
    const barChartProperties = widgetDefinition.properties;
    const primaryAxis = resolveWidgetProperty(barChartProperties.primaryAxis, group, widgetDefinition.widgetObject || widgetId);
    return {
        title: resolveWidgetProperty(barChartProperties.title, group, widgetDefinition.widgetObject || widgetId),
        type: 'bar-chart',
        data: {
            areAxesStacked: resolveWidgetProperty(barChartProperties.areAxesStacked, group, widgetDefinition.widgetObject || widgetId),
            areAxesSwapped: resolveWidgetProperty(barChartProperties.areAxesSwapped, group, widgetDefinition.widgetObject || widgetId),
            data: resolveWidgetProperty(barChartProperties.content, group, widgetDefinition.widgetObject || widgetId) ??
                [],
            isHistogram: resolveWidgetProperty(barChartProperties.isHistogram, group, widgetDefinition.widgetObject || widgetId),
            primaryAxis: {
                ...primaryAxis,
                bind: primaryAxis?.bind ?? '',
                title: primaryAxis?.title
                    ? resolveWidgetProperty(barChartProperties.primaryAxis.title, group, widgetDefinition.widgetObject || widgetId)
                    : undefined,
            },
            primaryAxisLabel: resolveWidgetProperty(barChartProperties.primaryAxisLabel, group, widgetDefinition.widgetObject || widgetId),
            secondaryAxisLabel: resolveWidgetProperty(barChartProperties.secondaryAxisLabel, group, widgetDefinition.widgetObject || widgetId),
            secondaryAxes: barChartProperties.secondaryAxes.map(createAxesConfig(widgetDefinition, group)),
        },
    };
};
const getLineChartWidgetConfig = (widgetId, widgetDefinition, group) => {
    const lineChartProperties = widgetDefinition.properties;
    const primaryAxis = resolveWidgetProperty(lineChartProperties.primaryAxis, group, widgetDefinition.widgetObject || widgetId);
    return {
        title: resolveWidgetProperty(lineChartProperties.title, group, widgetDefinition.widgetObject || widgetId),
        type: 'line-chart',
        data: {
            data: resolveWidgetProperty(lineChartProperties.content, group, widgetDefinition.widgetObject || widgetId) ??
                [],
            primaryAxis: {
                ...primaryAxis,
                bind: primaryAxis?.bind ?? '',
                title: primaryAxis?.title
                    ? resolveWidgetProperty(lineChartProperties.primaryAxis.title, group, widgetDefinition.widgetObject || widgetId)
                    : undefined,
            },
            secondaryAxes: lineChartProperties.secondaryAxes.map(createAxesConfig(widgetDefinition, group)),
            primaryAxisLabel: resolveWidgetProperty(lineChartProperties.primaryAxisLabel, group, widgetDefinition.widgetObject || widgetId),
            secondaryAxisLabel: resolveWidgetProperty(lineChartProperties.secondaryAxisLabel, group, widgetDefinition.widgetObject || widgetId),
        },
    };
};
const getPieChartWidgetConfig = (widgetId, widgetDefinition, group) => {
    const pieChartWidgetProperties = widgetDefinition.properties;
    const data = (resolveWidgetProperty(pieChartWidgetProperties.content, group, widgetDefinition.widgetObject || widgetId) || []).map((r) => ({
        ...r,
        i: r._id,
    }));
    return {
        title: resolveWidgetProperty(pieChartWidgetProperties.title, group, widgetDefinition.widgetObject || widgetId),
        type: 'pie-chart',
        data: {
            pieChartData: data,
            collectionItemDefinition: getRowDefinition({
                widgetDefinition,
                widgetId,
                group,
                tableProperties: pieChartWidgetProperties,
            }),
            isDonut: pieChartWidgetProperties.isDonut,
            hasCardView: pieChartWidgetProperties.hasCardView,
            filter: pieChartWidgetProperties.filter
                ? {
                    ...pieChartWidgetProperties.filter,
                    onChange: pieChartWidgetProperties.filter.onChange
                        ? (newValue) => {
                            resolveWidgetProperty(pieChartWidgetProperties?.filter?.onChange, group, widgetDefinition.widgetObject || widgetId, newValue);
                        }
                        : noop,
                }
                : undefined,
        },
    };
};
const getIndicatorTileWidgetConfig = (widgetId, widgetDefinition, group) => {
    const indicatorTileProperties = widgetDefinition.properties;
    const icon = resolveWidgetProperty(indicatorTileProperties.icon, group, widgetDefinition.widgetObject || widgetId);
    return {
        title: undefined,
        type: 'simple-indicator',
        data: {
            description: resolveWidgetProperty(indicatorTileProperties.subtitle, group, widgetDefinition.widgetObject || widgetId),
            mainIndicator: {
                title: String(resolveWidgetProperty(indicatorTileProperties.value, group, widgetDefinition.widgetObject || widgetId)),
                subtitle: resolveWidgetProperty(indicatorTileProperties.title, group, widgetDefinition.widgetObject || widgetId),
            },
            icon: icon
                ? {
                    url: resolveDetailedIcon(icon),
                    color: resolveWidgetProperty(indicatorTileProperties.color, group, widgetDefinition.widgetObject || widgetId),
                }
                : undefined,
            type: 'text',
        },
    };
};
const getGaugeWidgetConfig = (widgetId, widgetDefinition, group) => {
    const gaugeProperties = widgetDefinition.properties;
    return {
        title: resolveWidgetProperty(gaugeProperties.title, group, widgetDefinition.widgetObject || widgetId),
        type: 'gauge',
        data: {
            data: resolveWidgetProperty(gaugeProperties.content, group, widgetDefinition.widgetObject || widgetId) ?? {
                value: 0,
            },
            color: resolveWidgetProperty(gaugeProperties.color, group, widgetDefinition.widgetObject || widgetId),
            scale: resolveWidgetProperty(gaugeProperties.scale, group, widgetDefinition.widgetObject || widgetId),
            valueUnit: resolveWidgetProperty(gaugeProperties.valueUnit, group, widgetDefinition.widgetObject || widgetId),
            evolutionUnit: resolveWidgetProperty(gaugeProperties.evolutionUnit, group, widgetDefinition.widgetObject || widgetId),
        },
    };
};
const getTableFilterMenu = ({ widgetDefinition, widgetId, group, dataDropdownMenu, }) => {
    const dropdownMenu = resolveWidgetProperty(dataDropdownMenu, group, widgetDefinition.widgetObject || widgetId) || {};
    const firstKey = Object.keys(dropdownMenu)[0] ?? '';
    return Object.keys(dropdownMenu[firstKey] ?? []).reduce((acc, key) => {
        acc[key] = dropdownMenu[firstKey][key].title;
        return acc;
    }, {});
};
const getDateFilterDefinition = ({ widgetDefinition, widgetId, group, tableProperties, }) => {
    if (!tableProperties.dateFilter) {
        return undefined;
    }
    return {
        periodTypes: tableProperties.dateFilter.periodTypes || [
            DateFilterPeriodType.DAY,
            DateFilterPeriodType.WEEK,
            DateFilterPeriodType.MONTH,
            DateFilterPeriodType.YEAR,
        ],
        defaultDate: resolveWidgetProperty(tableProperties.dateFilter.defaultDate, group, widgetDefinition.widgetObject || widgetId),
        defaultPeriodType: tableProperties.dateFilter.defaultPeriodType,
        maxDate: resolveWidgetProperty(tableProperties.dateFilter.minDate, group, widgetDefinition.widgetObject || widgetId),
        minDate: resolveWidgetProperty(tableProperties.dateFilter.minDate, group, widgetDefinition.widgetObject || widgetId),
    };
};
const getRowDefinition = ({ widgetDefinition, widgetId, group, tableProperties, 
// If set to true, all click listeners are disabled
isEditing = false, }) => {
    const rowDefinition = resolveWidgetProperty(tableProperties.rowDefinition, group, widgetDefinition.widgetObject || widgetId);
    if (isEmpty(rowDefinition)) {
        return { title: { title: '' } };
    }
    return Object.keys(rowDefinition).reduce((prevValue, k) => {
        prevValue[k] = {
            ...rowDefinition[k],
            onClick: isEditing
                ? noop
                : (i) => resolveWidgetProperty(rowDefinition[k]?.onClick, group, widgetDefinition.widgetObject || widgetId, String(i)),
            displayOptions: {
                colorVariant: (i) => resolveWidgetProperty(rowDefinition[k].displayOptions?.colorVariant, group, widgetDefinition.widgetObject || widgetId, String(i)),
                tooltipText: (i) => resolveWidgetProperty(rowDefinition[k]?.displayOptions?.tooltipText, group, widgetDefinition.widgetObject || widgetId, String(i)),
            },
            renderedAs: rowDefinition[k].renderedAs,
            valueFormat: rowDefinition[k].valueFormat,
        };
        return prevValue;
    }, {});
};
const getTableOnFilterMenuChanged = ({ widgetDefinition, widgetId, group, tableProperties, dashboardId, }) => (selectedFilter) => {
    const dropdownMenu = resolveWidgetProperty(tableProperties.dataDropdownMenu, group, widgetDefinition.widgetObject || widgetId) ||
        {};
    const firstKey = Object.keys(dropdownMenu)[0] ?? '';
    if (firstKey) {
        updateWidgetOptions({
            dashboardId,
            widgetId,
            group,
            shouldRefreshData: true,
            skipUnset: true,
            options: {
                ...widgetDefinition.options,
                dataOptions: {
                    ...widgetDefinition.options?.dataOptions,
                    [firstKey]: selectedFilter,
                },
            },
        });
    }
};
const onTableSelectionChanged = ({ widgetDefinition, widgetId, group, dashboardId, }) => (selectedItems) => {
    updateWidgetOptions({
        dashboardId,
        widgetId,
        group,
        shouldRefreshData: false,
        skipUnset: true,
        options: { ...widgetDefinition.options, selectedItems },
    });
};
const onTableRepresentationModeChanged = ({ widgetDefinition, widgetId, group, dashboardId, }) => (updateMode) => {
    updateWidgetOptions({
        dashboardId,
        widgetId,
        group,
        shouldRefreshData: false,
        skipUnset: true,
        options: {
            ...widgetDefinition.options,
            mode: updateMode,
        },
    });
};
const onTableDateFilterChanged = ({ widgetDefinition, widgetId, group, dashboardId, }) => (selectedPeriod) => {
    updateWidgetOptions({
        dashboardId,
        widgetId,
        group,
        shouldRefreshData: true,
        options: { ...widgetDefinition.options, selectedPeriod },
        skipUnset: true,
    });
};
const getTableWidgetConfig = (widgetId, widgetDefinition, group, dashboardId, 
// If set to true, all click listeners are disabled
isEditing = false) => {
    const tableProperties = widgetDefinition.properties;
    const canSelect = isEditing
        ? false
        : !!resolveWidgetProperty(tableProperties.canSelect, group, widgetDefinition.widgetObject || widgetId);
    const mode = widgetDefinition.options?.mode ||
        resolveWidgetProperty(tableProperties.displayMode, group, widgetDefinition.widgetObject || widgetId) ||
        'table';
    const canChangeViewMode = resolveWidgetProperty(tableProperties.canSwitchViewMode, group, widgetDefinition.widgetObject || widgetId);
    const data = (resolveWidgetProperty(tableProperties.content, group, widgetDefinition.widgetObject || widgetId) || []).map((r) => ({
        ...r,
        i: r._id,
        image: r.image ? getImageUrlFromValue(r.image) : undefined,
    }));
    const totalCount = resolveWidgetProperty(tableProperties.totalCount, group, widgetDefinition.widgetObject);
    const iconSrc = tableProperties.businessIcon
        ? resolveDetailedIcon(resolveWidgetProperty(tableProperties.businessIcon, group, widgetDefinition.widgetObject || widgetId) ??
            '')
        : undefined;
    const filterMenuType = tableProperties.filterMenuType;
    const rowActions = resolveWidgetProperty(tableProperties.rowActions, group, widgetDefinition.widgetObject);
    return {
        type: 'table',
        data: {
            totalCount,
            data,
            canSelect,
            mode,
            canChangeViewMode,
            selectedPeriod: widgetDefinition.options?.selectedPeriod,
            rowDefinition: getRowDefinition({ widgetDefinition, widgetId, group, tableProperties, isEditing }),
            filterMenu: getTableFilterMenu({
                widgetDefinition,
                widgetId,
                group,
                dataDropdownMenu: tableProperties.dataDropdownMenu,
            }),
            dateFilter: getDateFilterDefinition({ widgetDefinition, widgetId, group, tableProperties }),
            onFilterMenuChanged: getTableOnFilterMenuChanged({
                widgetDefinition,
                widgetId,
                group,
                tableProperties,
                dashboardId,
            }),
            onSelectionChanged: onTableSelectionChanged({ widgetDefinition, widgetId, group, dashboardId }),
            onRepresentationModeChanged: onTableRepresentationModeChanged({
                widgetDefinition,
                widgetId,
                group,
                dashboardId,
            }),
            onDateFilterChanged: onTableDateFilterChanged({ widgetDefinition, widgetId, group, dashboardId }),
            iconSrc,
            filterMenuType,
            loadMoreRows: async (loadMoreParams = {
                startIndex: 0,
                stopIndex: 20,
                pageCount: 1,
                pageSize: 20,
                first: 20,
            }) => {
                if (loadMoreParams.after && widgetDefinition._id !== PREVIEW_WIDGET_ID) {
                    return fetchWidgetPage({
                        widgetId: widgetDefinition._id,
                        queryArgs: loadMoreParams,
                        group,
                    });
                }
                return [];
            },
            pageSize: 20,
            rowActions: rowActions
                ? rowActions.map(r => ({
                    ...r,
                    onClick: (i) => {
                        resolveWidgetProperty(r.onClick, group, widgetDefinition.widgetObject || widgetId, i);
                    },
                }))
                : rowActions,
        },
    };
};
export const getVisualProcessWidgetConfig = (widgetId, widgetDefinition, group, 
// If set to true, all click listeners are disabled
isEditing = false) => {
    const visualProcessProperties = widgetDefinition.properties;
    const visualProcessContent = resolveWidgetProperty(visualProcessProperties.content, group, widgetDefinition.widgetObject || widgetId);
    if (!visualProcessContent) {
        throw new Error('No content found for visual process');
    }
    return {
        type: 'visual-process',
        data: {
            value: visualProcessContent,
            resolveImageUrl,
            onLinkClick: isEditing
                ? noop
                : (link) => executeVisualProcessLink(DASHBOARD_SCREEN_ID, link),
        },
    };
};
const getCallToActionDefinition = (widgetId, group, widgetDefinition, callToActions) => {
    return Object.keys(callToActions).reduce((prevValue, key) => {
        const action = callToActions[key];
        prevValue[key] = {
            title: resolveWidgetProperty(action.title, group, widgetDefinition.widgetObject || widgetId) ?? '',
            isDisabled: !!resolveWidgetProperty(action.isDisabled, group, widgetDefinition.widgetObject || widgetId),
            isHidden: !!resolveWidgetProperty(action.isHidden, group, widgetDefinition.widgetObject || widgetId),
            onClick: () => resolveWidgetProperty(action.onClick, group, widgetDefinition.widgetObject || widgetId),
        };
        return prevValue;
    }, {});
};
const addSettingsPageToProps = (widgetId, group, widgetDefinition, props) => {
    const isGenericWidget = widgetDefinition.artifactName.startsWith(GENERIC_PLATFORM_WIDGET_PREFIX);
    if (widgetDefinition.properties.settingsPage && props.dropdown) {
        props.dropdown.items = [
            {
                i: 'settings',
                label: localize('@sage/xtrem-ui/dashboard-widget-settings', 'Settings'),
                icon: 'settings',
                onClick: async () => {
                    try {
                        if (widgetDefinition.properties.settingsPage) {
                            const store = xtremRedux.getStore();
                            const state = store.getState();
                            const settings = state.dashboard.dashboardGroups[group].dashboardEditor.currentDashboardDefinition.children.find(c => c._id === widgetId)
                                ?.settings || {};
                            const updatedSettings = await widgetDefinition.widgetObject.$.dialog.page(widgetDefinition.properties.settingsPage, settings, { rightAligned: true });
                            const thunkDispatch = store.dispatch;
                            await thunkDispatch(xtremRedux.actions.updateWidgetSettings(widgetId, group, updatedSettings));
                        }
                    }
                    catch {
                        // Do nothing.
                    }
                },
            },
            ...(props.dropdown.items || []),
        ];
    }
    if (isGenericWidget && props.dropdown) {
        props.dropdown.items = [
            {
                i: 'edit',
                label: localize('@sage/xtrem-ui/dashboard-editor-widget-edit', 'Edit'),
                icon: 'settings',
                onClick() {
                    const store = xtremRedux.getStore();
                    const state = store.getState();
                    const settings = state.dashboard.dashboardGroups[group].dashboardEditor.currentDashboardDefinition.children.find(c => c._id === widgetId)?.settings;
                    if (settings) {
                        const thunkDispatch = store.dispatch;
                        thunkDispatch(xtremRedux.actions.openWidgetEditorDialog(group, widgetId, settings));
                    }
                },
            },
            ...(props.dropdown.items || []),
        ];
    }
};
const addHeaderActionsToProps = (widgetId, group, widgetDefinition, props) => {
    if (widgetDefinition.properties.headerActions && props.dropdown) {
        props.dropdown.items = [
            ...widgetDefinition.properties.headerActions.map(ha => ({
                i: `${ha.title}-${ha.icon}`,
                label: ha.title,
                icon: ha.icon,
                onClick: () => resolveWidgetProperty(ha.onClick, group, widgetDefinition.widgetObject || widgetId),
            })),
            { i: 'header-actions-separator', isDivider: true },
            ...(props.dropdown.items || []),
        ];
    }
};
const getDefaultProps = (widgets, group, p, isEditing, canEditDashboards, onCloseCallback) => {
    const dropDownItems = [
        {
            i: 'refresh',
            label: localize('@sage/xtrem-ui/dashboard-widget-refresh', 'Refresh'),
            icon: 'refresh',
            onClick: () => refreshWidget(p._id, group),
        },
    ];
    if (isEditing && canEditDashboards) {
        dropDownItems.push({ i: 'close-separator', isDivider: true });
        dropDownItems.push({
            i: 'close',
            label: localize('@sage/xtrem-ui/dashboard-widget-close', 'Close'),
            icon: 'cross',
            onClick: onCloseCallback ? () => onCloseCallback(p._id) : undefined,
        });
    }
    const widgetDefinition = widgets?.[p._id];
    return {
        i: String(p._id),
        type: 'basic',
        x: p.x,
        y: p.y,
        w: p.w,
        h: p.h,
        title: resolveWidgetProperty(widgetDefinition.properties?.title, group, widgetDefinition.widgetObject || p._id),
        dropdown: {
            items: dropDownItems,
        },
    };
};
// TO RENDER DASHBOARD CALL THIS
export const getWidgetsFromDashboardItems = (dashboardId, group, children, widgets, 
// If set to true, all click listeners are disabled
isEditing = false, onCloseCallback, canEditDashboards = true) => children.map(p => {
    const widgetId = String(p._id);
    const props = getDefaultProps(widgets, group, p, isEditing, canEditDashboards, onCloseCallback);
    if (!widgets?.[p._id] || !widgets?.[p._id]?.data) {
        // Set loading state of the widget
        props.type = 'preview';
        props.title = undefined;
        props.dropdown = undefined;
        props.data = {};
        return props;
    }
    const widgetDefinition = widgets[p._id];
    if (isEditing) {
        // The settings are only available in edit mode
        addSettingsPageToProps(widgetId, group, widgetDefinition, props);
    }
    if (canEditDashboards) {
        addHeaderActionsToProps(widgetId, group, widgetDefinition, props);
    }
    if (widgetDefinition.properties.callToActions) {
        props.callToActions = getCallToActionDefinition(widgetId, group, widgetDefinition, widgetDefinition.properties.callToActions);
    }
    switch (widgetDefinition.widgetType) {
        case WidgetType.barChart:
            return {
                ...props,
                ...getBarChartWidgetConfig(widgetId, widgetDefinition, group),
            };
        case WidgetType.lineChart:
            return {
                ...props,
                ...getLineChartWidgetConfig(widgetId, widgetDefinition, group),
            };
        case WidgetType.pieChart:
            return {
                ...props,
                ...getPieChartWidgetConfig(widgetId, widgetDefinition, group),
            };
        case WidgetType.indicatorTile:
            return {
                ...props,
                ...getIndicatorTileWidgetConfig(widgetId, widgetDefinition, group),
            };
        case WidgetType.gauge:
            return {
                ...props,
                ...getGaugeWidgetConfig(widgetId, widgetDefinition, group),
            };
        case WidgetType.staticContent:
            return {
                ...props,
                ...getStaticContentWidgetConfig(widgetId, widgetDefinition, group),
            };
        case WidgetType.contactCard:
            return {
                ...props,
                ...getContactCardWidgetConfig(widgetId, widgetDefinition, group),
            };
        case WidgetType.table:
            return {
                ...props,
                ...getTableWidgetConfig(widgetId, widgetDefinition, group, dashboardId, isEditing),
            };
        case WidgetType.visualProcess:
            return {
                ...props,
                ...getVisualProcessWidgetConfig(widgetId, widgetDefinition, group, isEditing),
            };
        default:
            props.data = {
                htmlContent: `<div>Unknown widget: ${widgetDefinition.widgetType}</div>`,
            };
    }
    return props;
});
export const fetchWidgetData = async ({ forceRefetch = false, locale, version, widgetDefinition, queryArgs, group, }) => {
    const query = queryArgs
        ? resolveWidgetProperty(widgetDefinition.properties.getQuery, group, widgetDefinition.widgetObject, queryArgs)
        : resolveWidgetProperty(widgetDefinition.properties.getQuery, group, widgetDefinition.widgetObject);
    // The query is hashed and added to the key
    const settingsHash = MD5(JSON.stringify(query)).toString();
    const requestArgs = {
        query,
        cacheSettings: version && locale
            ? {
                key: `${widgetDefinition.artifactName}-${settingsHash}-widget-data`,
                locale,
                version,
                shouldFetchPlatformLiterals: false,
            }
            : undefined,
        forceRefetch,
    };
    let queryResult = await executeGraphqlQuery({ ...requestArgs });
    const cacheLifespan = resolveWidgetProperty(widgetDefinition.properties.cacheLifespan, group, widgetDefinition.widgetObject || widgetDefinition._id);
    // Check whether the result is coming from the cache, validate its timestamp
    if (queryResult.cachedAt && queryResult.cachedAt < Date.now() - (cacheLifespan ?? 0)) {
        // If it is cached and it was cached outside the cache time frame, force refetch the cache for the widget.
        queryResult = await executeGraphqlQuery({ ...requestArgs, forceRefetch: true });
    }
    return queryResult.data;
};
export const resolveWidgetProperty = (property, group, 
/** The widget ID or the AbstractWidget object */
widget, ...args) => {
    if (typeof property === 'function') {
        if (typeof widget === 'string') {
            const state = xtremRedux.getStore().getState();
            const widgetDefinition = state.dashboard.dashboardGroups[group].widgets[widget];
            if (widgetDefinition) {
                return property.apply(widgetDefinition.widgetObject, args);
            }
            return undefined;
        }
        return property.apply(widget, args);
    }
    return property;
};
//# sourceMappingURL=dashboard-service.js.map