import produce from 'immer';
import { navigationPanelId } from '../../component/container/navigation-panel/navigation-panel-types';
import { SERVER_VALIDATION_RULE_PREFIX } from '../../utils/constants';
import { ActionType } from '../action-types';
export const screenDefinitions = (state = {}, action) => {
    return produce(state, (nextState) => {
        let screenDefinition;
        let pageDefinition;
        switch (action.type) {
            case ActionType.SetErrors:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition?.errors) {
                    const uiProps = screenDefinition?.metadata?.uiComponentProperties?.[action.value.elementId];
                    /**
                     * TODO: this isn't ideal but since error messages are displayed by Carbon only if a title is set
                     * then just ensure that an empty title is visible so that the tooltip is shown.
                     */
                    if (uiProps && !uiProps.title) {
                        uiProps.title = ' ';
                        uiProps.isTitleHidden = false;
                    }
                    screenDefinition.errors[action.value.elementId] = action.value.errors;
                }
                break;
            case ActionType.UpdateErrors:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const errors = action.value.validationErrors;
                    screenDefinition.errors = Object.keys(errors)
                        .filter(e => Boolean(errors[e]))
                        .reduce((prevValue, elementId) => {
                        prevValue[elementId] = errors[elementId];
                        return prevValue;
                    }, {});
                }
                break;
            case ActionType.UpdateNestedFieldErrors:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const existingErrors = screenDefinition.errors[action.value.elementId] || [];
                    const filteredErrors = existingErrors.filter(e => !(e.columnId === action.value.columnId && e.recordId === action.value.recordId));
                    screenDefinition.errors = {
                        ...screenDefinition.errors,
                        [action.value.elementId]: [...filteredErrors, ...action.value.validationErrors],
                    };
                }
                break;
            case ActionType.UpdateNestedFieldRecordErrors:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const existingErrors = screenDefinition.errors[action.value.elementId] || [];
                    const filteredErrors = existingErrors.filter(e => !(e.recordId === action.value.recordId && (e.level || 0) === (action.value.level || 0)));
                    screenDefinition.errors = {
                        ...screenDefinition.errors,
                        [action.value.elementId]: [...filteredErrors, ...action.value.validationErrors],
                    };
                }
                break;
            case ActionType.AddInternalError:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const previousErrors = screenDefinition.internalErrors || {};
                    const currentErrors = {
                        ...previousErrors,
                        [action.value.elementId]: action.value.errorMessage,
                    };
                    screenDefinition.internalErrors = currentErrors;
                }
                break;
            case ActionType.RemoveInternalError:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const previousErrors = screenDefinition.internalErrors || {};
                    const currentErrors = Object.keys(previousErrors)
                        .filter(elementId => {
                        return elementId !== action.value.elementId;
                    })
                        .reduce((reduction, elementId) => {
                        return {
                            ...reduction,
                            [elementId]: previousErrors[elementId],
                        };
                    }, {});
                    screenDefinition.internalErrors = currentErrors;
                }
                break;
            case ActionType.AddScreenDefinition:
                const addScreenValue = action.value;
                // eslint-disable-next-line no-param-reassign
                nextState = { ...nextState, [addScreenValue.metadata.screenId]: addScreenValue };
                if (addScreenValue.type === 'page') {
                    addScreenValue.selectedRecordId = addScreenValue.values._id
                        ? addScreenValue.values._id
                        : null;
                }
                break;
            case ActionType.FinishScreenLoading:
                const finishAddScreenValue = action.value.pageDefinition;
                // eslint-disable-next-line no-param-reassign
                nextState = { ...nextState, [finishAddScreenValue.metadata.screenId]: finishAddScreenValue };
                if (finishAddScreenValue.type === 'page') {
                    finishAddScreenValue.selectedRecordId = finishAddScreenValue.values._id
                        ? finishAddScreenValue.values._id
                        : null;
                }
                break;
            case ActionType.CommitTransaction:
                screenDefinition = nextState[action.value.screenId];
                Object.keys(action.value.transaction.values)
                    .filter(key => action.value.transaction.values[key].hasChangedInTransaction)
                    .forEach(key => {
                    if (screenDefinition?.values) {
                        screenDefinition.values[key] = action.value.transaction.values[key].value;
                    }
                });
                Object.keys(action.value.transaction.uiComponentProperties)
                    .filter(key => action.value.transaction.uiComponentProperties[key].hasChangedInTransaction)
                    .forEach(key => {
                    if (screenDefinition?.metadata?.uiComponentProperties) {
                        screenDefinition.metadata.uiComponentProperties[key] =
                            action.value.transaction.uiComponentProperties[key].value;
                    }
                });
                break;
            case ActionType.SetNavigationPanelValue:
                pageDefinition = nextState[action.value.screenId];
                if (pageDefinition.navigationPanel) {
                    pageDefinition.navigationPanel.value = action.value.value;
                }
                break;
            case ActionType.RemoveNonNestedError:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const existingErrors = (screenDefinition.errors[action.value.elementId] || []).filter((e) => !!e.columnId);
                    if (existingErrors.length === 0) {
                        delete screenDefinition.errors[action.value.elementId];
                    }
                    else {
                        screenDefinition.errors = {
                            ...screenDefinition.errors,
                            [action.value.elementId]: [...existingErrors],
                        };
                    }
                }
                break;
            case ActionType.RemovePhantomError:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const existingErrors = screenDefinition.errors[action.value.elementId] || [];
                    screenDefinition.errors[action.value.elementId] = existingErrors.filter((e) => e.validationRule !== 'dirtyPhantomRow');
                }
                break;
            case ActionType.RemoveScreenDefinition:
                delete nextState[action.value];
                break;
            case ActionType.SetFieldDirtyState:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition && !screenDefinition.dirtyStates[action.value.elementId]) {
                    screenDefinition.dirtyStates[action.value.elementId] = true;
                }
                break;
            case ActionType.SetFieldCleanState:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition && screenDefinition.dirtyStates[action.value.elementId]) {
                    screenDefinition.dirtyStates[action.value.elementId] = false;
                }
                break;
            case ActionType.SetFieldValue:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition &&
                    action.value.isOrganicChange &&
                    !screenDefinition.dirtyStates[action.value.elementId]) {
                    screenDefinition.dirtyStates[action.value.elementId] = true;
                }
                if (screenDefinition?.values) {
                    screenDefinition.values[action.value.elementId] = action.value.fieldValue;
                }
                break;
            // TODO Should we use it? Why not starting a transaction?
            case ActionType.SetUiComponentProperties:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition?.metadata?.uiComponentProperties) {
                    screenDefinition.metadata.uiComponentProperties[action.value.elementId] =
                        action.value.fieldProperties;
                }
                break;
            case ActionType.SetScreenDefinitionDialogId:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    screenDefinition.dialogId = action.value.dialogId;
                }
                break;
            case ActionType.SetStickerDefinitionDialogId:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    screenDefinition.dialogId = action.value.dialogId;
                    screenDefinition.onFinish = action.value.onFinish;
                }
                break;
            case ActionType.SetScreenDefinitionReady:
                pageDefinition = nextState[action.value];
                if (pageDefinition) {
                    pageDefinition.isReady = true;
                }
                break;
            case ActionType.SetValues:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    screenDefinition.values = action.value.values;
                    if (!action.value.preserveValidationState) {
                        screenDefinition.errors = {};
                    }
                    if (screenDefinition.type === 'page') {
                        screenDefinition.selectedRecordId = action.value.values._id
                            ? action.value.values._id
                            : null;
                    }
                }
                break;
            case ActionType.ResetScreenUiComponentProperties:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition?.metadata?.uiComponentProperties) {
                    screenDefinition.metadata.uiComponentProperties = {
                        ...screenDefinition?.metadata?.defaultUiComponentProperties,
                    };
                }
                break;
            case ActionType.SetNavigationPanelIsHidden:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition?.navigationPanel) {
                    screenDefinition.navigationPanel.isHidden = action.value.isHidden;
                }
                break;
            case ActionType.SetNavigationPanelIsOpened:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition?.navigationPanel) {
                    screenDefinition.navigationPanel.isOpened = action.value.isOpened;
                }
                break;
            case ActionType.SetNavigationPanelIsHeaderHidden:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition?.navigationPanel) {
                    screenDefinition.navigationPanel.isHeaderHidden = action.value.isHeaderHidden;
                }
                break;
            case ActionType.SetPageClean:
                screenDefinition = nextState[action.value];
                if (screenDefinition) {
                    screenDefinition.dirtyStates = {};
                }
                break;
            case ActionType.SetPath:
                const screenId = Object.keys(nextState).find(s => nextState[s].isMainPage);
                screenDefinition = screenId ? nextState[screenId] : undefined;
                if (screenDefinition) {
                    screenDefinition.dirtyStates = {};
                }
                break;
            case ActionType.SetQueryParameter:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    pageDefinition = screenDefinition;
                    pageDefinition.queryParameters = {
                        ...pageDefinition.queryParameters,
                        [action.value.parameterName]: action.value.value,
                    };
                    if (action.value.parameterName === '_id') {
                        pageDefinition.selectedRecordId = action.value.value ? String(action.value.value) : null;
                    }
                }
                break;
            case ActionType.RemovePageServerErrors:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    screenDefinition.errors = Object.keys(screenDefinition.errors).reduce((prevValue, elementId) => {
                        const errors = (screenDefinition?.errors[elementId] || []).filter(e => !e.validationRule.startsWith(SERVER_VALIDATION_RULE_PREFIX));
                        prevValue[elementId] = errors.length > 0 ? errors : [];
                        return prevValue;
                    }, {});
                }
                break;
            case ActionType.SetActiveSection:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    screenDefinition.activeSection = action.value.activeSection;
                }
                break;
            case ActionType.SetTableViewColumnHidden:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const tableViews = screenDefinition.tableViews?.[action.value.elementId];
                    if (tableViews?.$current?.level[action.value.level]) {
                        tableViews.$current.level[action.value.level] = {
                            ...tableViews.$current.level[action.value.level],
                            columnHidden: action.value.columnHidden,
                        };
                    }
                }
                break;
            case ActionType.SetTableViewColumnOrder:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const tableViews = screenDefinition.tableViews?.[action.value.elementId];
                    if (tableViews?.$current?.level[action.value.level]) {
                        tableViews.$current.level[action.value.level] = {
                            ...tableViews.$current.level[action.value.level],
                            columnOrder: action.value.columnOrder,
                        };
                    }
                }
                break;
            case ActionType.SetTableViewFilter:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const tableViews = screenDefinition.tableViews?.[action.value.elementId];
                    if (tableViews?.$current?.level[action.value.level]) {
                        tableViews.$current.level[action.value.level] = {
                            ...tableViews.$current.level[action.value.level],
                            filter: action.value.filter,
                        };
                    }
                }
                break;
            case ActionType.SetTableViewGrouping:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const tableViews = screenDefinition.tableViews?.[action.value.elementId];
                    if (tableViews?.$current?.level[action.value.level]) {
                        tableViews.$current.level[action.value.level] = {
                            sortOrder: [],
                            ...tableViews.$current.level[action.value.level],
                            grouping: action.value.grouping,
                        };
                    }
                }
                break;
            case ActionType.SetTableViewOptionsMenuItem:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const tableViews = screenDefinition.tableViews?.[action.value.elementId];
                    if (tableViews?.$current?.level[action.value.level]) {
                        tableViews.$current.level[action.value.level] = {
                            ...tableViews.$current.level[action.value.level],
                            optionsMenuItem: action.value.optionsMenuItem,
                        };
                    }
                }
                break;
            case ActionType.SetTableViewOptionsMenuItemAndViewFilter:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const tableViews = screenDefinition.tableViews?.[action.value.elementId];
                    if (tableViews?.$current?.level[action.value.level]) {
                        tableViews.$current.level[action.value.level] = {
                            ...tableViews.$current.level[action.value.level],
                            filter: action.value.filter,
                            optionsMenuItem: action.value.optionsMenuItem,
                        };
                    }
                }
                break;
            case ActionType.SetTableViewSearchText:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const tableViews = screenDefinition.tableViews?.[action.value.elementId];
                    if (tableViews?.$current?.level[action.value.level]) {
                        tableViews.$current.level[action.value.level] = {
                            ...tableViews.$current.level[action.value.level],
                            searchText: action.value.searchText,
                        };
                    }
                }
                break;
            case ActionType.ClearNavigationPanelSearchText:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const tableViews = screenDefinition.tableViews?.[navigationPanelId];
                    if (tableViews?.$current?.level[0]) {
                        tableViews.$current.level[0] = {
                            ...tableViews.$current.level[0],
                            searchText: '',
                        };
                    }
                }
                break;
            case ActionType.SetTableViewSortOrder:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    const tableViews = screenDefinition.tableViews?.[action.value.elementId];
                    if (tableViews?.$current?.level[action.value.level]) {
                        tableViews.$current.level[action.value.level] = {
                            ...tableViews.$current.level[action.value.level],
                            sortOrder: action.value.sortOrder,
                        };
                    }
                }
                break;
            case ActionType.Set360ViewState:
                screenDefinition = nextState[action.value.screenId];
                if (screenDefinition) {
                    screenDefinition.is360ViewOn = action.value.state;
                }
                break;
            default:
            // Intentionally left empty.
        }
        return nextState;
    });
};
//# sourceMappingURL=screen-definitions.js.map