import { isEmpty, isNil } from 'lodash';
import { openDirtyPageConfirmationDialog } from '../../service/dirty-state-service';
import { commitTransaction } from '../../service/transactions-service';
import { resolveByValue } from '../../utils/resolve-value-utils';
import { getPageDefinitionFromState, getSidebarNestedFields, getSidebarTableProperties, isSidebarDirty, } from '../../utils/state-utils';
import { cleanMetadataFromRecord } from '../../utils/transformers';
import { ActionType } from '../action-types';
const handleOnSidebarOpenEvent = async ({ sidebarState, screenId, pageDefinition, recordValue, level, }) => {
    if (sidebarState?.onRecordOpened) {
        // Trigger on record open event if the functional developer declared one
        await sidebarState.onRecordOpened.apply(pageDefinition.page, [
            recordValue._id,
            cleanMetadataFromRecord(recordValue),
            level ?? 0,
        ]);
        commitTransaction(screenId);
    }
};
export const closeTableSidebar = (suppressDiscardEvent = false) => async (dispatch, getState) => {
    const state = getState();
    const sidebarState = state.sidebar;
    if (!suppressDiscardEvent) {
        if (!sidebarState) {
            throw new Error('Cannot close the sidebar if it is not even defined.');
        }
        if (isSidebarDirty(state)) {
            try {
                await openDirtyPageConfirmationDialog(sidebarState.screenId);
            }
            catch {
                // If the user rejects the dialog, we won't continue
                return;
            }
        }
        const pageDefinition = getPageDefinitionFromState(sidebarState.screenId, state);
        const tableProperties = getSidebarTableProperties({
            pageDefinition,
            elementId: sidebarState.elementId,
            level: sidebarState.level,
        });
        if (tableProperties.sidebar.onRecordDiscarded) {
            await tableProperties.sidebar.onRecordDiscarded.apply(pageDefinition.page);
            commitTransaction(sidebarState.screenId);
        }
        const tableValue = pageDefinition.values[sidebarState.elementId];
        tableValue.cancelRecordTransaction({
            recordId: sidebarState.recordId || '-1',
            recordLevel: sidebarState.level,
        });
    }
    dispatch({
        type: ActionType.CloseTableSidebar,
        value: undefined,
    });
};
export const openTableSidebar = ({ screenId, elementId, recordId, parentId, level, }) => async (dispatch, getState) => {
    dispatch({
        type: ActionType.OpenTableSidebar,
        value: {
            elementId,
            screenId,
            parentId,
            level,
        },
    });
    let pageDefinition = getPageDefinitionFromState(screenId, getState());
    const tableProperties = getSidebarTableProperties({ pageDefinition, elementId, level });
    const sidebarLayoutDefinition = resolveByValue({
        screenId,
        propertyValue: tableProperties.sidebar.layout,
        rowValue: null,
        fieldValue: null,
        skipHexFormat: true,
    });
    if (!sidebarLayoutDefinition) {
        throw new Error(`No sidebar layout is defined for ${elementId}`);
    }
    pageDefinition = getPageDefinitionFromState(screenId, getState());
    const tableValue = pageDefinition.values[elementId];
    const recordValue = recordId
        ? tableValue.getRecordByIdAndLevel({ id: recordId, level }) // Modify the current record
        : await tableValue.createNewPhantomRow({ isUncommitted: true, level, parentId }); // Create a new record
    // We need fresh values here in case of the callback made any changes
    pageDefinition = getPageDefinitionFromState(screenId, getState());
    let prevRecord;
    let nextRecord;
    if (recordId) {
        // We don't need to start a record transaction for new record because the createNewPhantomRow is already called with the isUncommitted flag.
        tableValue.startRecordTransaction({ recordId, recordLevel: level });
        prevRecord = await tableValue.getPreviousRecord({ recordId, recordLevel: level });
        nextRecord = await tableValue.getNextRecord({ recordId, tableProperties, recordLevel: level });
    }
    getState().applicationContext?.onTelemetryEvent?.(`tableSidebarOpened-${elementId}-${level || 0}`, {
        elementId,
        screenId,
        level,
    });
    await handleOnSidebarOpenEvent({
        sidebarState: tableProperties.sidebar,
        screenId,
        pageDefinition,
        recordValue,
        level,
    });
    dispatch({
        type: ActionType.SetTableSidebarRecord,
        value: {
            _id: recordId || recordValue._id,
            isNewRecord: isNil(recordId),
            hasNextRecord: !!nextRecord,
            hasPreviousRecord: !!prevRecord,
            level,
        },
    });
};
export const confirmTableSidebar = (addNewRecord = false) => async (dispatch, getState) => {
    const state = getState();
    const sidebarState = state.sidebar;
    if (!sidebarState || !sidebarState.screenId) {
        throw new Error('Cannot confirm the sidebar if it is not even defined.');
    }
    const pageDefinition = getPageDefinitionFromState(sidebarState.screenId, state);
    const tableProperties = getSidebarTableProperties({
        elementId: sidebarState.elementId,
        pageDefinition,
        level: sidebarState.level,
    });
    const collectionValue = pageDefinition.values[sidebarState.elementId];
    const recordId = sidebarState.recordId;
    if (recordId) {
        const rawRecord = collectionValue.getRawRecord({
            id: recordId,
            isUncommitted: true,
            level: sidebarState.level,
            cleanMetadata: false,
        });
        const result = await collectionValue.runValidationOnRecord({
            recordData: rawRecord,
            columnsToRevalidate: getSidebarNestedFields(state),
            isUncommitted: true,
        });
        if (!isEmpty(result)) {
            // If validations fail, we won't proceed
            return;
        }
        const updateRecord = collectionValue.commitRecord({ recordId, recordLevel: sidebarState.level });
        if (tableProperties.sidebar.onRecordConfirmed) {
            await tableProperties.sidebar.onRecordConfirmed.apply(pageDefinition.page, [
                recordId,
                cleanMetadataFromRecord(updateRecord),
            ]);
        }
    }
    getState().applicationContext?.onTelemetryEvent?.(`tableSidebarConfirmed-${sidebarState.elementId}-${sidebarState.level || 0}`, {
        elementId: sidebarState.elementId,
        screenId: sidebarState.screenId,
        level: sidebarState.level,
    });
    if (!addNewRecord) {
        dispatch(closeTableSidebar(true));
    }
    else {
        dispatch(openTableSidebar({
            screenId: sidebarState.screenId,
            elementId: sidebarState.elementId,
            level: sidebarState.level,
        }));
    }
};
const sidebarNextOrPrevRecord = async ({ dispatch, getState, isNext, }) => {
    dispatch({
        type: ActionType.SetTableSidebarLoading,
        value: true,
    });
    const sidebarState = getState().sidebar;
    if (!sidebarState || !sidebarState.screenId) {
        throw new Error('Cannot select the next record if the sidebar is not even defined.');
    }
    const pageDefinition = getPageDefinitionFromState(sidebarState.screenId, getState());
    const tableProperties = getSidebarTableProperties({
        pageDefinition,
        elementId: sidebarState.elementId,
        level: sidebarState.level,
    });
    const recordId = sidebarState.recordId;
    if (!recordId) {
        return;
    }
    const collectionValue = pageDefinition.values[sidebarState.elementId];
    if (isNext) {
        getState().applicationContext?.onTelemetryEvent?.(`tableSidebarNextRecordButtonClicked-${sidebarState.elementId}-${sidebarState.level || 0}`, {
            elementId: sidebarState.elementId,
            screenId: sidebarState.screenId,
            level: sidebarState.level,
        });
        const nextRecord = await collectionValue.getNextRecord({
            recordId,
            tableProperties,
            recordLevel: sidebarState.level,
        });
        if (nextRecord) {
            collectionValue.cancelRecordTransaction({ recordId, recordLevel: sidebarState.level });
            collectionValue.startRecordTransaction({ recordId: nextRecord._id, recordLevel: sidebarState.level });
            await handleOnSidebarOpenEvent({
                sidebarState: tableProperties.sidebar,
                screenId: sidebarState.screenId,
                pageDefinition,
                recordValue: nextRecord,
                level: sidebarState.level,
            });
            const hasNextRecord = await collectionValue.getNextRecord({
                recordId: nextRecord._id,
                tableProperties,
                recordLevel: sidebarState.level,
            });
            dispatch({
                type: ActionType.SetTableSidebarRecord,
                value: {
                    _id: nextRecord._id,
                    hasNextRecord: !!hasNextRecord,
                    hasPreviousRecord: true,
                    level: sidebarState.level,
                },
            });
        }
    }
    else {
        getState().applicationContext?.onTelemetryEvent?.(`tableSidebarPreviousRecordButtonClicked-${sidebarState.elementId}-${sidebarState.level || 0}`, {
            elementId: sidebarState.elementId,
            screenId: sidebarState.screenId,
            level: sidebarState.level,
        });
        const previousRecord = await collectionValue.getPreviousRecord({
            recordId,
            recordLevel: sidebarState.level,
        });
        if (previousRecord) {
            collectionValue.cancelRecordTransaction({ recordId, recordLevel: sidebarState.level });
            collectionValue.startRecordTransaction({ recordId: previousRecord._id, recordLevel: sidebarState.level });
            await handleOnSidebarOpenEvent({
                sidebarState: tableProperties.sidebar,
                screenId: sidebarState.screenId,
                pageDefinition,
                recordValue: previousRecord,
                level: sidebarState.level,
            });
            const hasPreviousRecord = await collectionValue.getPreviousRecord({
                recordId: previousRecord._id,
                recordLevel: sidebarState.level,
            });
            dispatch({
                type: ActionType.SetTableSidebarRecord,
                value: {
                    _id: previousRecord._id,
                    hasNextRecord: true,
                    hasPreviousRecord: !!hasPreviousRecord,
                    level: sidebarState.level,
                },
            });
        }
    }
};
export const selectSidebarNextRecord = () => async (dispatch, getState) => {
    return sidebarNextOrPrevRecord({ dispatch, getState, isNext: true });
};
export const selectSidebarPreviousRecord = () => async (dispatch, getState) => {
    return sidebarNextOrPrevRecord({ dispatch, getState, isNext: false });
};
export const setSidebarDirty = () => ({ type: ActionType.SetTableSidebarDirty });
//# sourceMappingURL=table-sidebar-actions.js.map