import { getFieldTitle, getPageSubtitle } from '../../component/field/carbon-helpers';
import { errorDialog } from '../../service/dialog-service';
import { notifyConsumerAboutDirtyStatus } from '../../service/dirty-state-service';
import { fetchDefaultValues, fetchRecordData } from '../../service/graphql-service';
import { destroyScreenCollections } from '../../service/loki';
import { fetchPageDefinition } from '../../service/screen-loader-service';
import { notifyConsumerOnError } from '../../service/telemetry-service';
import { rollbackTransaction } from '../../service/transactions-service';
import { formatScreenValues } from '../../service/value-formatter-service';
import * as wrapperService from '../../service/wrapper-service';
import { asyncForEach } from '../../utils/async';
import { xtremConsole } from '../../utils/console';
import { NEW_PAGE, DASHBOARD_SCREEN_ID } from '../../utils/constants';
import { triggerScreenEvent } from '../../utils/events';
import { getNavigationPanelState, getNavigationPanelTablePropertiesFromPageDefinition, getPageDefinitionFromState, getPagePropertiesFromPageDefinition, getPagePropertiesFromState, } from '../../utils/state-utils';
import { getLocalStorage } from '../../utils/window';
import { ActionType } from '../action-types';
import { addScreenDefinition, clearWidgetOptions, removeScreenDefinition, setGlobalLoading, setPath, setScreenDefinitionReady, setUserCustomizations, } from './common-actions';
import { closeDialog } from './dialog-actions';
import { setNavigationPanelIsOpened } from './navigation-panel-actions';
import { getPageTitlesFromPageDefinition } from '../../utils/page-utils';
const addQueryParametersToPath = (path, queryParameters) => {
    const params = Object.keys(queryParameters).length > 0 ? `/${btoa(JSON.stringify(queryParameters))}` : '';
    return path.split('/').slice(0, 3).join('/') + params;
};
const getQueryParametersFromPath = (urlPart) => {
    try {
        return JSON.parse(atob(urlPart));
    }
    catch {
        // If we provide a deep link (e.g. /@sage/em-sales/SalesOrder/SalesOrder:QTEFR0110040) the atob call will fail
        // In order to support accessing resources in a REST fashion, we set the last part of the url pathname as _id by default
        return {
            _id: urlPart,
        };
    }
};
export const getPageValues = async (screenDefinition, recordId, clean = false, nodeTypes) => {
    const screenId = screenDefinition.metadata.screenId;
    const pageProperties = getPagePropertiesFromPageDefinition(screenDefinition);
    if (pageProperties.isTransient || !pageProperties.node) {
        return {};
    }
    if (recordId && recordId !== NEW_PAGE) {
        return fetchRecordData(screenId, recordId, screenDefinition);
    }
    return fetchDefaultValues(screenDefinition, String(pageProperties.node), undefined, clean, nodeTypes);
};
export const triggerPostNavigationToasts = (state, path, queryParameters) => {
    if (state.applicationContext?.handleNavigation) {
        state.applicationContext?.handleNavigation(path, queryParameters);
    }
    return notifyConsumerAboutDirtyStatus(state);
};
const handleNavigationPanelError = (dispatch, screenId) => (error) => {
    dispatch(setGlobalLoading(false));
    errorDialog(screenId, 'Error', error);
    notifyConsumerOnError(error);
    xtremConsole.error('Error on loading record', error);
};
const closeCurrentPage = () => async (dispatch, getState) => {
    const state = getState();
    Object.keys(state.activeDialogs).forEach(dialogId => {
        dispatch(closeDialog(parseInt(dialogId, 10)));
    });
    const currentScreenDefinitions = state.screenDefinitions;
    const closeScreenDefinition = async (screenId) => {
        /**
         * We need to check if the screen definition is still on the state. It could be that the `closeDialog` event closed and removed a page dialog
         * so trying to close it again would throw an exception
         *  */
        if (getState().screenDefinitions[screenId]) {
            await triggerScreenEvent(screenId, 'onClose');
            wrapperService.onBlur();
            dispatch(removeScreenDefinition(screenId));
        }
    };
    const screenIds = Object.keys(state.screenDefinitions).filter(screenId => currentScreenDefinitions[screenId].type === 'page' && screenId !== DASHBOARD_SCREEN_ID);
    await asyncForEach(screenIds, closeScreenDefinition);
};
/**
 * Be aware that base64 encoded parameters in the path will override the queryParameters object
 * @param path Be
 * @param queryParameters
 */
export const navigate = (path, queryParameters = {}) => async (dispatch, getState) => {
    dispatch(setGlobalLoading(true));
    const pathParts = path.split('/');
    if (pathParts.length > 3) {
        // eslint-disable-next-line no-param-reassign
        queryParameters = getQueryParametersFromPath(pathParts[3]);
    }
    const state = getState();
    if (!state.path && path) {
        dispatch(clearWidgetOptions());
    }
    await dispatch(closeCurrentPage());
    if (path) {
        try {
            const pageDefinition = await fetchPageDefinition({
                getState,
                dispatch,
                path,
                isMainPage: true,
                queryParameters,
            });
            if (Object.keys(queryParameters).length > 0) {
                // The path needs to be updated here because fetchPageDefinition might modify queryParameters
                // eslint-disable-next-line no-param-reassign
                path = addQueryParametersToPath(path, queryParameters);
            }
            dispatch(setPath(path));
            // It has to be called with a "fresh" state so the previous actions are already applied to the state value.
            await triggerPostNavigationToasts(getState(), path, queryParameters);
            if (pageDefinition) {
                const screenId = pageDefinition.metadata.screenId;
                const storedCustomizations = getLocalStorage().getItem('userCustomizations');
                if (storedCustomizations) {
                    const userCustomizations = JSON.parse(storedCustomizations);
                    dispatch(setUserCustomizations(userCustomizations));
                    const screenCustomizations = userCustomizations[screenId];
                    if (screenCustomizations) {
                        Object.keys(screenCustomizations).forEach(key => {
                            pageDefinition.metadata.uiComponentProperties[key] = {
                                ...pageDefinition.metadata.uiComponentProperties[key],
                                ...screenCustomizations[key],
                            };
                        });
                    }
                }
                dispatch(addScreenDefinition(pageDefinition));
                const pageProperties = getPagePropertiesFromPageDefinition(pageDefinition);
                if (state.applicationContext && typeof state.applicationContext.onPageTitleChange === 'function') {
                    state.applicationContext.onPageTitleChange(getFieldTitle(screenId, pageProperties, null) || null, getPageSubtitle(screenId, pageProperties, null));
                }
                const hasNavigationPanel = !!pageProperties.navigationPanel;
                if (hasNavigationPanel && !pageDefinition.selectedRecordId) {
                    dispatch(setNavigationPanelIsOpened(true, screenId));
                }
                else {
                    dispatch(setNavigationPanelIsOpened(false, screenId));
                }
                try {
                    pageDefinition.page.$standardDeleteAction.isDisabled =
                        !queryParameters._id || queryParameters._id === NEW_PAGE;
                    pageDefinition.page.$standardDuplicateAction.isDisabled =
                        !queryParameters._id || queryParameters._id === NEW_PAGE;
                    await triggerScreenEvent(screenId, 'onLoad');
                }
                catch (error) {
                    handleNavigationPanelError(dispatch, screenId)(error);
                }
                dispatch(setScreenDefinitionReady(screenId));
            }
        }
        catch (error) {
            errorDialog(pathParts[2], 'Error', error);
            notifyConsumerOnError(error);
            xtremConsole.error('Error on navigating', error);
        }
    }
    else {
        state.applicationContext?.onPageTitleChange?.(null, null);
        dispatch(setPath(path));
    }
    dispatch(setGlobalLoading(false));
};
export const selectRecord = (screenId, recordId, clean = false) => async (dispatch, getState) => {
    rollbackTransaction(screenId);
    dispatch(setGlobalLoading(true));
    destroyScreenCollections(screenId, true);
    const state = getState();
    const pageDefinition = getPageDefinitionFromState(screenId, state);
    const pageProperties = getPagePropertiesFromState(screenId, state);
    if (!pageDefinition || !pageProperties) {
        return;
    }
    if (recordId) {
        pageDefinition.queryParameters._id = recordId;
        pageDefinition.selectedRecordId = recordId;
    }
    else {
        delete pageDefinition.queryParameters._id;
        pageDefinition.selectedRecordId = recordId;
    }
    const path = addQueryParametersToPath(state.path, pageDefinition.queryParameters);
    dispatch(setPath(path));
    // It has to be called with a "fresh" state so the previous actions are already applied to the state value.
    await triggerPostNavigationToasts(getState(), path, pageDefinition.queryParameters);
    try {
        const values = await getPageValues(pageDefinition, recordId, clean, state.nodeTypes);
        const formattedValues = formatScreenValues({
            controlObjects: pageDefinition.metadata.controlObjects,
            nodeTypes: state.nodeTypes,
            parentNode: String(pageProperties.node),
            plugins: state.plugins,
            screenId,
            values,
        });
        dispatch({ type: ActionType.ResetScreenUiComponentProperties, value: { screenId } });
        dispatch({ type: ActionType.SetValues, value: { screenId, values: formattedValues } });
        dispatch({ type: ActionType.ClearNavigationPanelSearchText, value: { screenId } });
        pageDefinition.page.$standardDeleteAction.isDisabled =
            !pageDefinition.queryParameters._id || pageDefinition.queryParameters._id === NEW_PAGE;
        pageDefinition.page.$standardDuplicateAction.isDisabled =
            !pageDefinition.queryParameters._id || pageDefinition.queryParameters._id === NEW_PAGE;
        await triggerScreenEvent(screenId, 'onLoad');
        if (!recordId && state.browser.lessThan.m) {
            dispatch(setNavigationPanelIsOpened(true, screenId));
        }
        else if (!recordId || state.browser.lessThan.m) {
            dispatch(setNavigationPanelIsOpened(!state.browser.lessThan.m, screenId));
        }
        const { title, subtitle } = getPageTitlesFromPageDefinition(pageDefinition, state.applicationContext?.locale || 'en-US');
        state.applicationContext?.onPageTitleChange?.(title, subtitle || null);
    }
    catch (error) {
        handleNavigationPanelError(dispatch, screenId)(error);
    }
};
export const findNavPanelRecordLocation = async (screenId, state, offset) => {
    const navigationPanelState = getNavigationPanelState(screenId, state);
    const pageDefinition = getPageDefinitionFromState(screenId, state);
    // If the nav panel state OR the current record ID is not found, cannot identify the current position
    if (!navigationPanelState || !pageDefinition || !pageDefinition.queryParameters._id) {
        return null;
    }
    let navigationPanelItems = navigationPanelState.value.getData({
        cleanMetadata: false,
        limit: Number.MAX_SAFE_INTEGER,
    });
    let pageNumber = 0;
    do {
        pageNumber += 1;
        let currentItemIndex = navigationPanelItems.findIndex((e) => e._id === pageDefinition.queryParameters._id);
        const isCurrentLastItem = currentItemIndex === navigationPanelItems.length - 1;
        const isCurrentNotInTheSet = currentItemIndex === -1;
        // If the current item is the last one in the set OR not found at all, try to fetch next page
        if ((isCurrentLastItem || isCurrentNotInTheSet) && navigationPanelState.value.hasNextPage) {
            // eslint-disable-next-line no-await-in-loop
            const result = await navigationPanelState.value.getPageWithCurrentQueryArguments({
                tableFieldProperties: getNavigationPanelTablePropertiesFromPageDefinition(pageDefinition),
                pageNumber,
                pageSize: 20,
                cursor: [...navigationPanelItems].reverse()[0].__cursor,
                cleanMetadata: false,
            });
            navigationPanelItems = [...navigationPanelItems, ...result];
            currentItemIndex = navigationPanelItems.findIndex((e) => e._id === pageDefinition.queryParameters._id);
        }
        if (currentItemIndex > -1) {
            return navigationPanelItems[currentItemIndex + offset]._id;
        }
    } while (pageDefinition.queryParameters._id !== NEW_PAGE && navigationPanelState.value.hasNextPage);
    // We try to find the current item until we read the full collection and no more pages are left
    return null;
};
export const selectNextRecord = (screenId) => async (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('pageNextRecordButtonClicked', { screenId });
    try {
        const nextRecordId = await findNavPanelRecordLocation(screenId, getState(), 1);
        if (nextRecordId) {
            await dispatch(selectRecord(screenId, nextRecordId));
        }
    }
    catch {
        handleNavigationPanelError(dispatch, screenId);
    }
};
export const selectFirstRecord = (screenId) => async (dispatch, getState) => {
    const state = getState();
    const pageDefinition = getPageDefinitionFromState(screenId, state);
    const navigationPanelState = getNavigationPanelState(screenId, state);
    if (!pageDefinition || !navigationPanelState) {
        throw new Error(`No navigation panel definition was found for ${screenId}`);
    }
    const navigationPanelItems = navigationPanelState.value.getData({
        cleanMetadata: false,
        limit: Number.MAX_SAFE_INTEGER,
    });
    if (navigationPanelItems.length === 0) {
        throw new Error(`The navigation panel does not have any items for ${screenId}`);
    }
    await dispatch(selectRecord(screenId, navigationPanelItems[0]._id));
};
export const selectPreviousRecord = (screenId) => async (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('pageNextRecordButtonClicked', { screenId });
    try {
        const previousRecordId = await findNavPanelRecordLocation(screenId, getState(), -1);
        if (previousRecordId) {
            await dispatch(selectRecord(screenId, previousRecordId));
        }
    }
    catch {
        handleNavigationPanelError(dispatch, screenId);
    }
};
export const goHome = () => async (dispatch, getState) => {
    dispatch(clearWidgetOptions());
    await dispatch(closeCurrentPage());
    getState().applicationContext?.handleNavigation('', {});
};
export const setIdToQueryParameters = (screenId, _id) => ({
    type: ActionType.SetQueryParameter,
    value: { parameterName: '_id', screenId, value: _id },
});
//# sourceMappingURL=router-actions.js.map