import { navigationPanelId } from '../../component/container/navigation-panel/navigation-panel-types';
import { fetchNavigationPanelData } from '../../service/graphql-service';
import { createNavigationPanelValue } from '../../service/navigation-panel-service';
import { getNavigationPanelDefinitionFromState, getNavigationPanelState, getPageDefinitionFromState, getPagePropertiesFromState, } from '../../utils/state-utils';
import { ActionType } from '../action-types';
import { setStoredTableViewsByElement } from '../../service/table-view-storage-service';
import { getOrderByFromSortModel } from '../../utils/table-component-utils';
import { getGraphQLFilter, getTypedNestedFields } from '../../service/filter-service';
import { getNestedFieldsFromProperties } from '../../utils/nested-field-utils';
import { mapAgGridFilterToXtremFilters } from '../../utils/ag-grid/ag-grid-table-utils';
import { camelCase } from 'lodash';
export const refreshNavigationPanel = (screenId, recordAdded = false) => async (dispatch, getState) => {
    const state = getState();
    const pageDefinition = getPageDefinitionFromState(screenId, state);
    if (!pageDefinition) {
        throw new Error(`No page definition found for ${screenId} page.`);
    }
    const pageProperties = getPagePropertiesFromState(screenId, state);
    if (!pageProperties) {
        throw new Error(`No page properties found for ${screenId} page.`);
    }
    const navigationPanelDef = getNavigationPanelDefinitionFromState(screenId, state);
    if (!navigationPanelDef) {
        throw new Error(`Navigation panel couldn't be refreshed as it is not defined for ${screenId} page`);
    }
    const navigationPanelState = getNavigationPanelState(screenId, state);
    if (!navigationPanelState) {
        throw new Error(`Navigation panel couldn't be refreshed as it is not defined for ${screenId} page`);
    }
    const rootNode = pageProperties.node;
    if (!rootNode) {
        throw new Error(`Navigation panel couldn't be refreshed because no root node was found on ${screenId} page.`);
    }
    const navigationPanelProperties = pageDefinition.metadata.uiComponentProperties[navigationPanelId];
    const currentRecords = navigationPanelState.value.getData({
        cleanMetadata: true,
        limit: Number.MAX_SAFE_INTEGER,
    });
    const pageSize = Math.max(30, recordAdded ? currentRecords.length + 1 : currentRecords.length);
    const navigationPanelTableViews = pageDefinition.tableViews[navigationPanelId];
    const currentView = navigationPanelTableViews?.$current?.level[0];
    const selectedOptionsMenuItem = currentView?.optionsMenuItem || navigationPanelProperties?.optionsMenu?.[0];
    const orderBy = currentView?.sortOrder
        ? getOrderByFromSortModel(currentView?.sortOrder, navigationPanelProperties.columns || [])
        : navigationPanelProperties.orderBy;
    const filter = currentView?.filter
        ? getGraphQLFilter(Object.keys(currentView.filter).map(mapAgGridFilterToXtremFilters(currentView.filter)), getTypedNestedFields(screenId, rootNode, getNestedFieldsFromProperties(navigationPanelProperties), state.nodeTypes))
        : {};
    const navigationPanelData = await fetchNavigationPanelData(screenId, pageDefinition, navigationPanelProperties, orderBy, filter, pageSize, state.nodeTypes, selectedOptionsMenuItem);
    const value = createNavigationPanelValue(screenId, navigationPanelProperties, navigationPanelData, state.nodeTypes, state.applicationContext?.locale, selectedOptionsMenuItem, orderBy);
    dispatch(setNavigationPanelValue(screenId, value));
};
const setNavigationPanelValue = (screenId, value) => (dispatch) => {
    dispatch({
        type: ActionType.SetNavigationPanelValue,
        value: { screenId, value },
    });
};
const storeUserCustomization = (getState, screenId, elementId) => {
    const state = getState();
    if (state.applicationContext?.login && state.screenDefinitions[screenId]?.tableViews?.[elementId]) {
        setStoredTableViewsByElement(state.applicationContext.login, screenId, elementId, state.screenDefinitions[screenId].tableViews[elementId]);
    }
};
export const setNavigationPanelIsHidden = (isHidden, screenId) => (dispatch) => {
    dispatch({ type: ActionType.SetNavigationPanelIsHidden, value: { isHidden, screenId } });
};
export const setNavigationPanelIsOpened = (isOpened, screenId) => (dispatch) => {
    dispatch({ type: ActionType.SetNavigationPanelIsOpened, value: { isOpened, screenId } });
};
export const setNavigationPanelIsHeaderHidden = (isHeaderHidden, screenId) => (dispatch) => {
    dispatch({ type: ActionType.SetNavigationPanelIsHeaderHidden, value: { isHeaderHidden, screenId } });
};
export const setTableViewColumnHidden = (screenId, elementId, level, columnHidden) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.(`tableVisibilityChange-${elementId}`, {
        screenId,
        elementId,
    });
    dispatch({
        type: ActionType.SetTableViewColumnHidden,
        value: { screenId, elementId, level, columnHidden },
    });
    storeUserCustomization(getState, screenId, elementId);
};
export const setTableViewColumnOrder = (screenId, elementId, level, columnOrder) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.(`tableColumnOrderChange-${elementId}`, {
        screenId,
        elementId,
    });
    dispatch({
        type: ActionType.SetTableViewColumnOrder,
        value: { screenId, elementId, level, columnOrder },
    });
    storeUserCustomization(getState, screenId, elementId);
};
export const setTableViewFilter = (screenId, elementId, level, filter) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.(`tableFiltered-${elementId}`, {
        elementId,
        screenId,
    });
    dispatch({
        type: ActionType.SetTableViewFilter,
        value: { screenId, elementId, level, filter },
    });
    storeUserCustomization(getState, screenId, elementId);
};
export const setTableViewGrouping = (screenId, elementId, level, grouping) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.(`tableGroupingChange-${elementId}-${grouping?.key}`, {
        elementId,
        screenId,
        item: grouping?.key,
    });
    dispatch({
        type: ActionType.SetTableViewGrouping,
        value: { screenId, elementId, level, grouping },
    });
    storeUserCustomization(getState, screenId, elementId);
};
export const setTableViewOptionMenuItem = (screenId, elementId, level, optionsMenuItem) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.(`tableOptionsMenuChange-${elementId}-${camelCase(optionsMenuItem?.title)}`, {
        elementId,
        item: optionsMenuItem?.title,
        screenId,
    });
    dispatch({
        type: ActionType.SetTableViewOptionsMenuItem,
        value: { screenId, elementId, level, optionsMenuItem },
    });
    storeUserCustomization(getState, screenId, elementId);
};
export const setTableViewOptionsMenuItemAndViewFilter = (screenId, elementId, level, optionsMenuItem, filter) => (dispatch, getState) => {
    dispatch({
        type: ActionType.SetTableViewOptionsMenuItem,
        value: { screenId, elementId, level, optionsMenuItem, filter },
    });
    storeUserCustomization(getState, screenId, elementId);
};
export const setTableViewSearchText = (screenId, elementId, level, searchText) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.(`tableQuickTextFiltered-${elementId}`, {
        elementId,
        screenId,
    });
    dispatch({
        type: ActionType.SetTableViewSearchText,
        value: { screenId, elementId, level, searchText },
    });
};
export const clearNavigationPanelSearchText = (screenId) => (dispatch) => {
    dispatch({
        type: ActionType.ClearNavigationPanelSearchText,
        value: { screenId },
    });
};
export const setTableViewSortOrder = (screenId, elementId, level, sortOrder) => (dispatch, getState) => {
    dispatch({
        type: ActionType.SetTableViewSortOrder,
        value: { screenId, elementId, level, sortOrder },
    });
    storeUserCustomization(getState, screenId, elementId);
};
//# sourceMappingURL=navigation-panel-actions.js.map