import { closeDialog } from '../../service/dialog-service';
import { dispatchContainerValidation } from '../../service/dispatch-service';
import { processNodeDetailsResponse, queryInitialMetadata } from '../../service/metadata-service';
import { showContainerValidationToast } from '../../service/toast-service';
import { WIZARD_FINISHED } from '../../utils/constants';
import { triggerFieldEvent, triggerScreenEvent } from '../../utils/events';
import { getPageDefinitionFromState, getVisibleSectionsFromPage } from '../../utils/state-utils';
import { parseScreenValues } from '../../utils/transformers';
import { ActionType } from '../action-types';
import { clearOutOfDateEntries, getArtifactCacheDatabase, getCachedRawNodeDetails, } from '../../utils/artifact-cache-utils';
import { isDevMode } from '../../utils/window';
export const setPath = (path) => ({
    type: ActionType.SetPath,
    value: path,
});
export const setComponentLoading = (screenId, elementId, isLoading) => ({
    type: ActionType.SetComponentLoading,
    value: { screenId, elementId, isLoading },
});
export const setApplicationContext = (applicationContext) => ({
    type: ActionType.SetApplicationContext,
    value: applicationContext,
});
export const updateUserCustomization = (screenId, userCustomizations) => {
    return { type: ActionType.UpdateUserCustomization, value: { screenId, userCustomizations } };
};
export const finishScreenLoading = (pageDefinition, nodeTypes, dataTypes, enumTypes, locale) => ({
    type: ActionType.FinishScreenLoading,
    value: { pageDefinition, nodeTypes, dataTypes, enumTypes, locale },
});
export const addSchemaInfo = (nodeTypes, dataTypes, enumTypes, locale) => ({
    type: ActionType.AddSchemaInfo,
    value: { nodeTypes, dataTypes, enumTypes, locale },
});
export const addScreenDefinition = (pageDefinition) => ({
    type: ActionType.AddScreenDefinition,
    value: pageDefinition,
});
export const addPlugin = (pluginPackage, pluginDefinition) => ({
    type: ActionType.AddPlugin,
    value: {
        pluginPackage,
        pluginDefinition,
    },
});
export const setScreenDefinitionReady = (screenId) => ({
    type: ActionType.SetScreenDefinitionReady,
    value: screenId,
});
export const setScreenDefinitionDialogId = (screenId, dialogId, pageControlObject, title, subtitle) => ({
    type: ActionType.SetScreenDefinitionDialogId,
    value: { screenId, dialogId, pageControlObject, title, subtitle },
});
export const removeScreenDefinition = (screenId) => ({
    type: ActionType.RemoveScreenDefinition,
    value: screenId,
});
export const setFieldProperties = (screenId, elementId, fieldProperties) => ({
    type: ActionType.SetUiComponentProperties,
    value: {
        screenId,
        elementId,
        fieldProperties,
    },
});
export const setGlobalLoading = (loaderState) => ({
    type: ActionType.SetGlobalLoading,
    value: loaderState,
});
export const setUserCustomizations = (userCustomizations) => ({
    type: ActionType.SetUserCustomizations,
    value: userCustomizations,
});
export const actionStub = () => {
    throw new Error('Action stub called, did you forget to override the the action?');
};
export const addTranslations = (locale, literals) => ({
    type: ActionType.AddTranslations,
    value: {
        locale,
        literals,
    },
});
export const setInitialMetaData = (applicationContext) => async (dispatch, getState) => {
    const state = getState();
    if (!state.applicationPackages) {
        let processedNodeDetails = null;
        const initialMetaData = await queryInitialMetadata();
        if (!isDevMode()) {
            const db = await getArtifactCacheDatabase();
            const rawNodeDetails = await getCachedRawNodeDetails({
                db,
                installedPackages: initialMetaData.installedPackages,
                passphrase: applicationContext.cacheEncryptionKey,
            });
            processedNodeDetails = processNodeDetailsResponse(rawNodeDetails);
            // Clear up the out of date artifacts from the cache
            await clearOutOfDateEntries({ db, installedPackages: initialMetaData.installedPackages });
        }
        dispatch({
            type: ActionType.SetInitialMetaData,
            value: {
                ...initialMetaData,
                nodeTypes: processedNodeDetails?.nodeTypes || {},
                dataTypes: processedNodeDetails?.dataTypes || {},
                enumTypes: processedNodeDetails?.enumTypes || {},
                locale: applicationContext.locale || 'en-US',
            },
        });
    }
};
export const clearWidgetOptions = () => ({
    type: ActionType.ClearWidgetOptions,
    value: null,
});
export const set360ViewState = (screenId, state) => {
    triggerScreenEvent(screenId, 'on360ViewSwitched', state);
    return {
        type: ActionType.Set360ViewState,
        value: { screenId, state },
    };
};
/**
 * Sets the active section in the state
 * @param screenId
 * @param activeSection the ID of the active section OR `WIZARD_FINISHED` token if the user is stepping to close the widget.
 * @param isInitialLoad
 * @returns
 */
export const setActiveSection = (screenId, activeSection, isInitialLoad = false) => async (dispatch, getState) => {
    const state = getState();
    const pageDefinition = getPageDefinitionFromState(screenId, state);
    const sections = getVisibleSectionsFromPage(screenId, pageDefinition);
    getState().applicationContext?.onTelemetryEvent?.(`sectionActivated-${activeSection}`, {
        screenId,
        activeSection,
    });
    const currentlyActiveSection = pageDefinition.activeSection || getVisibleSectionsFromPage(screenId, pageDefinition)[0]?.id;
    if (currentlyActiveSection === activeSection) {
        // If the current active section is the same that we are about to change to, we should not proceed
        return;
    }
    if (currentlyActiveSection && pageDefinition.page.$.mode === 'wizard') {
        // In case of wizard mode we need to run the validations before we can change the tabs
        // We need to check if we are stepping forward or backwards in the wizard, we only validate on forward steps
        const indexOfCurrentSection = sections.findIndex(s => s.id === pageDefinition.activeSection);
        const indexOfNextSection = sections.findIndex(s => s.id === activeSection);
        if (indexOfCurrentSection < indexOfNextSection) {
            const result = await dispatchContainerValidation(pageDefinition.page._pageMetadata.screenId, currentlyActiveSection);
            // If the validation fails, we display the page error dialog
            if (result.blockingErrors.length > 0) {
                showContainerValidationToast(pageDefinition.page._pageMetadata, result);
                return;
            }
        }
    }
    if (!isInitialLoad && currentlyActiveSection) {
        await triggerFieldEvent(screenId, currentlyActiveSection, 'onInactive');
    }
    if (activeSection === WIZARD_FINISHED) {
        if (pageDefinition.dialogId) {
            await closeDialog(pageDefinition.dialogId, parseScreenValues(screenId, true, false, state), true);
        }
        else {
            triggerScreenEvent(screenId, 'onClose', true);
        }
        return;
    }
    await dispatch({
        type: ActionType.SetActiveSection,
        value: { activeSection, screenId },
    });
    if (!isInitialLoad && activeSection) {
        await triggerFieldEvent(screenId, activeSection, 'onActive');
    }
};
export const stepOneSection = (screenId, direction) => async (dispatch, getState) => {
    const state = getState();
    const screenDefinitions = state.screenDefinitions;
    // This function is only called for wizard page dialogs, so here we don't need to handle the case of custom dialogs
    const sections = getVisibleSectionsFromPage(screenId, screenDefinitions[screenId]);
    const activeSection = screenDefinitions[screenId].activeSection || sections[0]?.id;
    const currentIndex = sections.findIndex(s => screenId && s.id === activeSection);
    const nextIndex = currentIndex + direction;
    if (nextIndex === sections.length) {
        // If we step out of the range of sections, that means that the wizard process is now complete.
        return dispatch(setActiveSection(screenId, WIZARD_FINISHED));
    }
    return dispatch(setActiveSection(screenId, sections[nextIndex].id));
};
//# sourceMappingURL=common-actions.js.map