import * as React from 'react';
import { connect } from 'react-redux';
import { isChart } from './widget-editor-utils';
import { FilterableSelect, Option } from 'carbon-react/esm/components/select';
import { GridColumn, GridRow, SelectionCard } from '@sage/xtrem-ui-components';
import { getGutterSize } from '../../utils/responsive-utils';
import Textbox from 'carbon-react/esm/components/textbox';
import { localize } from '../../service/i18n-service';
import Typography from 'carbon-react/esm/components/typography';
import { resolveDetailedIcon } from '../../utils/detailed-icons-utils';
import { isEmpty, memoize } from 'lodash';
const widgetTypes = memoize(() => ({
    INDICATOR_TILE: {
        title: localize('@sage/xtrem-ui/widget-editor-type-indicator-tile', 'Indicator'),
        description: localize('@sage/xtrem-ui/widget-editor-type-indicator-tile-description', 'Display major record updates.'),
        icon: 'card',
    },
    TABLE: {
        title: localize('@sage/xtrem-ui/widget-editor-type-table', 'Table'),
        description: localize('@sage/xtrem-ui/widget-editor-type-table-description', 'Display data in rows and columns.'),
        icon: 'table',
    },
    BAR_CHART: {
        title: localize('@sage/xtrem-ui/widget-editor-type-bar-chart', 'Bar chart'),
        description: localize('@sage/xtrem-ui/widget-editor-type-bar-chart-description', 'Display data visually with bars.'),
        icon: 'chart-bar',
    },
    LINE_CHART: {
        title: localize('@sage/xtrem-ui/widget-editor-type-line-chart', 'Line chart'),
        description: localize('@sage/xtrem-ui/widget-editor-type-line-chart-description', 'Display data visually with lines.'),
        icon: 'financials',
    },
}));
export function BasicsStep({ nodeNames, widgetCategories, onWidgetDefinitionUpdated, widgetDefinition, browserIs, isNewWidget, stepIndex, }) {
    const [title, setTitle] = React.useState(widgetDefinition.title || '');
    const [titleValidationMessage, setTitleValidationMessage] = React.useState(undefined);
    const [nodeValidationMessage, setNodeValidationMessage] = React.useState(undefined);
    const hasWidgetSelection = React.useMemo(() => isNewWidget || isChart(widgetDefinition), [isNewWidget, widgetDefinition]);
    React.useEffect(() => {
        setTitle(widgetDefinition.title || '');
    }, [widgetDefinition.title, setTitle]);
    const selectableWidgetsKeys = React.useMemo(() => (!isNewWidget && isChart(widgetDefinition) ? ['BAR_CHART', 'LINE_CHART'] : Object.keys(widgetTypes())), [widgetDefinition, isNewWidget]);
    const gridGutter = getGutterSize(browserIs);
    const onNodeChanged = React.useCallback((ev) => {
        const node = ev.target.value;
        const isDifferentNode = node !== widgetDefinition.node;
        if (isDifferentNode) {
            // Only keep a few basic properties because if the node type changes, we reset all other settings
            onWidgetDefinitionUpdated({
                node,
                title: widgetDefinition.title,
                category: widgetDefinition.category,
                type: widgetDefinition.type,
            });
        }
        if (!node) {
            setNodeValidationMessage(localize('@sage/xtrem-ui/widget-editor-basic-step-missing-node', 'You need to add a node'));
        }
        else {
            setNodeValidationMessage(undefined);
        }
    }, [onWidgetDefinitionUpdated, widgetDefinition, setNodeValidationMessage]);
    const onCategoryChanged = React.useCallback((ev) => {
        onWidgetDefinitionUpdated({ ...widgetDefinition, category: ev.target.value });
    }, [onWidgetDefinitionUpdated, widgetDefinition]);
    const onTypeSelected = React.useCallback((type) => {
        // Only keep a few basic properties because if the widget type changes, we reset all other settings
        onWidgetDefinitionUpdated({
            type,
            title: widgetDefinition.title,
            category: widgetDefinition.category,
            node: widgetDefinition.node,
        });
    }, [onWidgetDefinitionUpdated, widgetDefinition]);
    const onTitleChanged = React.useCallback((ev) => {
        setTitle(ev.target.value);
    }, [setTitle]);
    const onWidgetTitleBlur = React.useCallback(() => {
        onWidgetDefinitionUpdated({ ...widgetDefinition, title });
        if (!title) {
            setTitleValidationMessage(localize('@sage/xtrem-ui/widget-editor-basic-step-missing-title', 'You need to add a title'));
        }
        else {
            setTitleValidationMessage(undefined);
        }
    }, [title, widgetDefinition, onWidgetDefinitionUpdated, setTitleValidationMessage]);
    const resolvedWidgetTypes = widgetTypes();
    return (React.createElement("div", null,
        React.createElement(Typography, { variant: "h2", "data-testid": "e-widget-editor-step-title" }, localize('@sage/xtrem-ui/widget-editor-basic-step-title', '{{stepIndex}}. Select a widget to get started', { stepIndex })),
        React.createElement("div", { className: "e-widget-editor-section" },
            React.createElement(GridRow, { columns: 8, gutter: gridGutter, margin: 0, verticalMargin: 0 },
                React.createElement(GridColumn, { columnSpan: 4 },
                    React.createElement(Textbox, { "data-testid": "e-widget-editor-basic-title", required: true, validationOnLabel: true, error: titleValidationMessage, label: localize('@sage/xtrem-ui/widget-editor-widget-title', 'Widget title'), value: title, onChange: onTitleChanged, onBlur: onWidgetTitleBlur })),
                !isEmpty(widgetCategories) && (React.createElement(GridColumn, { columnSpan: 4 },
                    React.createElement(FilterableSelect, { "data-testid": "e-widget-editor-basic-category", name: "widgetCategory", id: "widgetCategory", label: localize('@sage/xtrem-ui/widget-editor-widget-category', 'Widget category'), onChange: onCategoryChanged, value: widgetDefinition.category || '' }, Object.keys(widgetCategories).map(k => (React.createElement(Option, { text: widgetCategories[k], value: k, key: k })))))),
                isNewWidget && (React.createElement(GridColumn, { columnSpan: 4 },
                    React.createElement(FilterableSelect, { "data-testid": "e-widget-editor-basic-node", required: true, validationOnLabel: true, error: nodeValidationMessage, name: "node", id: "node", label: localize('@sage/xtrem-ui/widget-editor-entry-node', 'Entry module'), onChange: onNodeChanged, value: widgetDefinition.node || '' }, Object.keys(nodeNames).map(k => (React.createElement(Option, { text: nodeNames[k], value: k, key: k })))))))),
        hasWidgetSelection && (React.createElement(React.Fragment, null,
            React.createElement(Typography, { variant: "segment-header-small", mt: "16px" }, "Widgets"),
            React.createElement("div", { className: "e-widget-editor-section" },
                React.createElement(GridRow, { columns: 8, gutter: gridGutter, margin: 0, verticalMargin: 0 }, selectableWidgetsKeys.map(key => (React.createElement(GridColumn, { columnSpan: 4, key: key },
                    React.createElement(SelectionCard, { _id: key, isSelected: widgetDefinition.type === key, title: resolvedWidgetTypes[key].title, icon: resolveDetailedIcon(resolvedWidgetTypes[key].icon), description: resolvedWidgetTypes[key].description, onClick: () => onTypeSelected(key) }))))))))));
}
const mapStateToProps = (state, props) => {
    const { widgetEditor } = state.dashboard.dashboardGroups[props.group];
    return {
        ...props,
        browserIs: state.browser.is,
        isNewWidget: !widgetEditor.widgetId,
        widgetCategories: state.dashboard.widgetCategories,
    };
};
export const ConnectedBasicsStep = connect(mapStateToProps)(BasicsStep);
//# sourceMappingURL=basics-step.js.map