import { get, isNil } from 'lodash';
import { AbstractWidgetQueryBuilder } from '.';
import { calculateDeepPaths } from '../../service/customization-service';
import { formatDateToCurrentLocale, localize, localizeEnumMember } from '../../service/i18n-service';
const presentationMapping = {
    Checkbox: 'normal',
    Date: 'normal',
    Label: 'pill',
    Numeric: 'normal',
    Progress: 'normal',
    Text: 'normal',
};
export class TableWidgetQueryBuilder extends AbstractWidgetQueryBuilder {
    constructor(userWidgetDefinition) {
        super(userWidgetDefinition);
        this.columns = ['title', 'titleRight', 'line2', 'line2Right', 'line3', 'line3Right'];
        // We need a unique object key for the dashboard library for each column
        for (let i = 4; i < 100; i += 1) {
            this.columns.push(`line${i}`);
        }
        this.rowDefinition = this.columns.reduce((acc, key, index) => {
            const column = this.userWidgetDefinition.columns?.[index];
            if (column) {
                acc[key] = {
                    title: column.title,
                    renderedAs: presentationMapping[column.presentation],
                    displayOptions: {},
                    valueFormat: column.presentation === 'Numeric' ? 'number' : 'string',
                    decimalDigits: column.presentation === 'Numeric' && !isNil(column.formatting) ? column.formatting : 2,
                };
            }
            return acc;
        }, {});
    }
    build(queryArgs) {
        return this.buildQuery(queryArgs).buildFilter().buildOrderBy().query;
    }
    getValue(data) {
        const edges = get(data, `${this.nodePath}.query.edges`);
        if (!edges) {
            return [];
        }
        return edges.map(({ node, cursor }) => {
            return this.columns.reduce((acc, k, index) => {
                const column = this.userWidgetDefinition.columns?.[index];
                if (column) {
                    const columnId = column.id ?? column.path;
                    const { subpath, selectorSegments } = calculateDeepPaths(columnId);
                    let value = get(node, subpath.join('.'), '');
                    const isJsonSubfield = selectorSegments.length > 0;
                    if (isJsonSubfield) {
                        value = get(JSON.parse(value || '{}'), selectorSegments.join('.'), '');
                    }
                    acc[k] = this.formatValue({ value, column, isJsonSubfield });
                }
                return acc;
            }, { _id: cursor });
        });
    }
    formatValue({ column: { data: { type, enumValues, node }, formatting, divisor, }, value, isJsonSubfield, }) {
        if (!type) {
            return String(value);
        }
        switch (type) {
            case 'Boolean':
                return value === true
                    ? localize('@sage/xtrem-ui/true', 'True')
                    : localize('@sage/xtrem-ui/false', 'False');
            case 'Date':
            case 'DateTime':
                return value ? formatDateToCurrentLocale(value) : '';
            case 'Decimal':
            case 'Float':
            case 'Int':
            case 'IntReference':
                const num = Number(value);
                if (Number.isNaN(num)) {
                    return '0';
                }
                return (num / (divisor ?? 1)).toFixed(formatting);
            case 'Enum':
                const localizedOptions = this.userWidgetDefinition.node && enumValues && node
                    ? enumValues.reduce((enumValueDict, key) => {
                        const enumType = node.indexOf('/') === -1
                            ? `${this.userWidgetDefinition.node?.split('/').slice(0, -1).join('/')}/${node}`
                            : node;
                        enumValueDict[key] = localizeEnumMember(enumType, key);
                        return enumValueDict;
                    }, {})
                    : undefined;
                return localizedOptions?.[value] ?? String(value);
            case 'ExternalReference':
            case 'ID':
            case 'InputStream':
            case 'IntOrString':
            case 'Json':
            case 'String':
            default:
                break;
        }
        return value === null && isJsonSubfield ? '' : String(value);
    }
}
//# sourceMappingURL=table-widget-query-builder.js.map