import Button from 'carbon-react/esm/components/button';
import ButtonMinor from 'carbon-react/esm/components/button-minor';
import { camelCase, noop } from 'lodash';
import React from 'react';
import { connect } from 'react-redux';
import styled from 'styled-components';
import { calculateActionMenu } from '../../../utils/action-menu-utils';
import { TABLE_INLINE_ACTION_CSS_CLASS } from '../../../utils/constants';
import { triggerHandledEvent } from '../../../utils/events';
import { getPageDefinitionFromState } from '../../../utils/state-utils';
import { splitValueToMergedValue } from '../../../utils/transformers';
import { navigationPanelId } from '../../container/navigation-panel/navigation-panel-types';
const getTableInlineActionTestId = (title, id) => {
    const uniqueId = camelCase(id || title);
    return `${TABLE_INLINE_ACTION_CSS_CLASS} e-inline-action e-inline-action--${uniqueId} e-action e-action--${uniqueId}`;
};
const StyledMinorButton = styled(ButtonMinor) `
    ${(props) => (props.isDisplayed ? '' : 'display: none;')}
    ${(props) => props.isHidden ? '.e-card-content && { display: none; }; visibility: hidden;' : 'visibility: visible;'}
`;
const StyledButton = styled(Button) `
    ${(props) => (props.isDisplayed ? '' : 'display: none;')}
    ${(props) => props.isHidden ? '.e-card-content && { display: none; }; visibility: hidden;' : 'visibility: visible;'}
`;
const TableInlineAction = React.memo(({ id, buttonType, focusOnLoad, icon, isDestructive, isDisabled, isDisplayed, isHidden, isMajor, key, onClick, onKeyDown, pendoId, title, }) => {
    const buttonRef = React.useRef(null);
    React.useEffect(() => {
        if (focusOnLoad && document.activeElement?.classList?.contains?.('ag-cell')) {
            buttonRef.current?.focus();
        }
    }, [focusOnLoad]);
    const buttonProps = {
        tabIndex: isHidden || isDisabled ? '-1' : '0',
        className: TABLE_INLINE_ACTION_CSS_CLASS,
        'data-testid': getTableInlineActionTestId(title, id),
        'data-pendoid': pendoId,
        buttonType: buttonType ?? 'tertiary',
        size: 'small',
        iconType: icon,
        'aria-label': title ?? '',
        iconTooltipMessage: title ?? '',
        onClick,
        onKeyDown,
        disabled: isDisabled,
        destructive: isDestructive ?? false,
        isHidden: isHidden ?? false,
        isDisplayed: isDisplayed ?? true,
    };
    if (isMajor) {
        return React.createElement(StyledButton, { ref: buttonRef, key: key, ...buttonProps });
    }
    return (React.createElement(StyledMinorButton, { ref: r => {
            buttonRef.current = r;
        }, key: key, ...buttonProps }));
}, (prev, next) => {
    return (prev.isHidden === next.isHidden &&
        prev.isDisabled === next.isDisabled &&
        prev.buttonType === next.buttonType &&
        prev.icon === next.icon &&
        prev.title === next.title &&
        prev.isDestructive === next.isDestructive &&
        prev.isDisplayed === next.isDisplayed);
});
TableInlineAction.displayName = 'TableInlineAction';
function TableInlineActions({ inlineActions, isEditing }) {
    return (React.createElement(React.Fragment, null, inlineActions.map((inlineAction, index) => {
        return (React.createElement(TableInlineAction, { focusOnLoad: Boolean(index === 0 && isEditing), ...inlineAction, key: inlineAction.key, onKeyDown: noop }));
    })));
}
const mapStateToProps = (state, props) => {
    const pageDefinition = getPageDefinitionFromState(props.screenId, state);
    const fieldProperties = pageDefinition.metadata.uiComponentProperties[props.fieldId];
    const tableData = props.fieldId === navigationPanelId
        ? pageDefinition.navigationPanel?.value
        : pageDefinition.values[props.fieldId];
    const isNestedGrid = Object.prototype.hasOwnProperty.call(fieldProperties, 'levels');
    const inlineActions = calculateActionMenu({
        actions: props.actions,
        actionType: 'table-inline-action',
        screenId: props.screenId,
        rowValue: props.rowValue,
        pendoId: `tableInlineAction-${props.fieldId}-${props.level}`,
        accessBindings: pageDefinition.accessBindings,
        onTriggerMenuItem: (context, onClick, onError) => {
            state.applicationContext?.onTelemetryEvent?.(`tableInlineActionTriggered-${context.uniqueId}`, {
                screenId: props.screenId,
                elementId: props.fieldId,
                recordId: props.recordId,
                id: context.id,
                uniqueId: context.uniqueId,
            });
            if (!onClick) {
                return;
            }
            const args = [
                ...[
                    props.screenId,
                    props.fieldId,
                    { onClick, onError: onError || fieldProperties.onError },
                    props.recordId,
                    splitValueToMergedValue(props.rowValue),
                ],
                ...(isNestedGrid
                    ? [
                        props.level,
                        tableData.getAncestorIds({
                            id: props.recordId,
                            level: props.level,
                        }),
                    ]
                    : []),
            ];
            triggerHandledEvent(...args);
        },
    });
    return {
        elementId: props.fieldId,
        level: props.level,
        inlineActions,
        isEditing: props.isEditing,
        stopEditing: props.stopEditing,
        column: props.column,
        api: props.api,
        rowIndex: props.rowIndex,
        hasDropdownActions: props.hasDropdownActions,
    };
};
export default connect(mapStateToProps)(TableInlineActions);
//# sourceMappingURL=table-inline-actions.js.map