import React from 'react';
import Sidebar from 'carbon-react/esm/components/sidebar';
import { DraggableContainer, DraggableItem } from 'carbon-react/esm/components/draggable';
import { Checkbox } from 'carbon-react/esm/components/checkbox';
import Typography from 'carbon-react/esm/components/typography';
import Icon from 'carbon-react/esm/components/icon';
import { TableSearchBox } from '../table-search-box-component';
import { localize } from '../../../../service/i18n-service';
export function TableConfigurationDialogColumnItem({ c, onVisibilityChanged, }) {
    return (React.createElement("div", { className: "e-table-configuration-dialog-line", "data-testid": `e-table-configuration-dialog-line-col-${c.colId}` },
        React.createElement(Checkbox, { label: c.title, value: c.colId, mb: 0, checked: !c.isHidden, disabled: c.isMandatory, onChange: onVisibilityChanged }),
        c.isMandatory && React.createElement(Icon, { type: "locked" })));
}
export function TableConfigurationDialog({ isDialogOpen, onDialogClose, columnPanelColumnStates, onChangeColumnHiddenState, onOrderChange, }) {
    const [columnFilter, setColumnFilter] = React.useState('');
    const sidebarRef = React.useRef(null);
    const canCloseRef = React.useRef(true);
    const onVisibilityChanged = React.useCallback((ev) => {
        const colId = ev.target.value;
        const isHidden = ev.target.checked;
        onChangeColumnHiddenState(colId, isHidden);
    }, [onChangeColumnHiddenState]);
    const shouldDisplayDraggableItem = React.useCallback((c) => {
        // The items are visible if no filter applied or the title contains the filter value
        return (!columnFilter || c.title.toLowerCase().includes(columnFilter.toLowerCase())) && !c.isSpecialColumn;
    }, [columnFilter]);
    const shouldDisplaySpecialItem = React.useCallback((c) => {
        // The items are visible if no filter applied or the title contains the filter value
        return (!columnFilter || c.title.toLowerCase().includes(columnFilter.toLowerCase())) && !!c.isSpecialColumn;
    }, [columnFilter]);
    const onItemsReordered = React.useCallback((ids, itemWasJustMoved) => {
        if (ids.length === columnPanelColumnStates.length) {
            const newIdOrder = ids.map(String);
            return onOrderChange(newIdOrder);
        }
        return onOrderChange(getNewOrder(columnPanelColumnStates, ids, itemWasJustMoved));
    }, [onOrderChange, columnPanelColumnStates]);
    const areElementsOverlapping = React.useCallback((element1, element2) => {
        const rect1 = element1.getBoundingClientRect();
        const rect2 = element2.getBoundingClientRect();
        if (!(rect1.right < rect2.left ||
            rect1.left > rect2.right ||
            rect1.bottom < rect2.top ||
            rect1.top > rect2.bottom)) {
            return true;
        }
        return false;
    }, []);
    React.useEffect(() => {
        if (isDialogOpen) {
            canCloseRef.current = false;
            // eslint-disable-next-line no-return-assign
            setTimeout(() => (canCloseRef.current = true), 500); // Prevents the dialog from closing immediately after opening
        }
    }, [isDialogOpen]);
    const onClickListener = React.useCallback((ev) => {
        // If the user clicks outside the dialog, we automatically close it
        if (sidebarRef.current &&
            ev.target &&
            ev.target.isConnected &&
            !sidebarRef.current.contains(ev.target) &&
            (ev.target.contains(sidebarRef.current) ||
                !areElementsOverlapping(sidebarRef.current, ev.target)) &&
            canCloseRef.current) {
            onDialogClose();
        }
    }, [onDialogClose, areElementsOverlapping]);
    React.useEffect(() => {
        // Add the event listener when the component mounts or onClickListener changes
        window.document.addEventListener('click', onClickListener);
        return () => {
            // Remove the event listener when the component unmounts or before a new listener is added
            window.document.removeEventListener('click', onClickListener);
        };
    }, [onClickListener]);
    const specialColumns = columnPanelColumnStates.filter(shouldDisplaySpecialItem);
    return (React.createElement(Sidebar, { ref: sidebarRef, "aria-label": "sidebar", open: !!isDialogOpen, onCancel: onDialogClose, enableBackgroundUI: true, width: "464px", header: React.createElement(Typography, { variant: "h3", mb: "16px" }, localize('@sage/xtrem-ui/table-column-settings', 'Column settings')) },
        React.createElement("div", { className: "e-table-configuration-dialog" },
            React.createElement(TableSearchBox, { disableAutoFocus: true, onSearchBoxValueChange: setColumnFilter }),
            isDialogOpen && (React.createElement("div", { className: "e-table-configuration-dialog-columns" },
                specialColumns?.length > 0 && (React.createElement("div", { className: "e-table-configuration-dialog-special-columns" }, specialColumns.map((c) => (React.createElement(TableConfigurationDialogColumnItem, { key: c.colId, c: c, onVisibilityChanged: onVisibilityChanged }))))),
                React.createElement(DraggableContainer, { getOrder: onItemsReordered }, columnPanelColumnStates
                    .filter(shouldDisplayDraggableItem)
                    .map((c) => (React.createElement(DraggableItem, { key: c.colId, id: c.colId },
                    React.createElement(TableConfigurationDialogColumnItem, { c: c, onVisibilityChanged: onVisibilityChanged }))))))))));
}
// Function that passing one full list, one filtered list and one moved item can return the new order of the list
export const getNewOrder = (columnState, filteredColumnState, movedItem) => {
    const mappedToIdsColumnState = columnState.map(c => c.colId);
    const indexOfMovedItemInState = mappedToIdsColumnState.findIndex(c => c === movedItem);
    const indexOfMovedItemInFiltered = filteredColumnState.findIndex(c => c === movedItem);
    let newOrder;
    // If the element is moved down
    if (indexOfMovedItemInState < indexOfMovedItemInFiltered) {
        const elementBeforeMovedItem = filteredColumnState[indexOfMovedItemInFiltered - 1];
        const indexOfElementBeforeMovedItem = mappedToIdsColumnState.findIndex(c => c === elementBeforeMovedItem);
        newOrder = mappedToIdsColumnState.filter(c => c !== movedItem);
        newOrder.splice(indexOfElementBeforeMovedItem, 0, movedItem);
        return newOrder;
    }
    // If the element is moved up
    const elementAfterMovedItem = filteredColumnState[indexOfMovedItemInFiltered + 1];
    const indexOfElementAfterMovedItem = mappedToIdsColumnState.findIndex(c => c === elementAfterMovedItem);
    newOrder = mappedToIdsColumnState.filter(c => c !== movedItem);
    newOrder.splice(indexOfElementAfterMovedItem, 0, movedItem);
    return newOrder;
};
//# sourceMappingURL=table-configuration-dialog.js.map