import * as React from 'react';
import { getDataTestIdAttribute } from '../../../utils/dom';
import { ButtonComponent } from '../../field/button/button-component';
import { localize } from '../../../service/i18n-service';
import { camelCase, isEmpty, noop } from 'lodash';
import { useSelector, useStore } from 'react-redux';
import { confirmationDialog, pageDialog } from '../../../service/dialog-service';
import { GroupState } from '../../field/table/table-component-types';
import { getGroupFilterValue } from '../../../service/collection-data-utils';
import { getGraphFilter } from '../../../utils/table-component-utils';
import { showToast } from '../../../service/toast-service';
import { getNavigationPanelState, getNavigationPanelTablePropertiesFromPageDefinition, getPageDefinitionFromState, } from '../../../utils/state-utils';
import { BULK_ACTION_ASYNC_EXPORT } from '../../../utils/constants';
import { getAllUnselectedIds, getFilterModel } from '../../../utils/ag-grid/ag-grid-table-utils';
export function DesktopTableBulkActionBar({ bulkActions, gridApi, selectedGroupState, isSelectAllChecked, selectedRowCount, screenId, onClearSelection, groupByColumnField, onTelemetryEvent, }) {
    const store = useStore();
    const contextPage = useSelector(s => getPageDefinitionFromState(screenId, s).path);
    const exportTemplatesByNode = useSelector(s => getPageDefinitionFromState(screenId, s).metadata.exportTemplatesByNode);
    const graphApi = useSelector(s => getPageDefinitionFromState(screenId, s).page.$.graph);
    const locale = useSelector(s => s.applicationContext?.locale);
    const tableFieldProperties = useSelector(s => getNavigationPanelTablePropertiesFromPageDefinition(getPageDefinitionFromState(screenId, s)));
    const activeOptionsMenuItem = useSelector(s => getNavigationPanelState(screenId, s)?.value?.getActiveOptionsMenuItem());
    const getBulkActionParameters = React.useCallback(async (bulkAction) => {
        const node = String(tableFieldProperties.node);
        if (exportTemplatesByNode &&
            bulkAction.mutation === BULK_ACTION_ASYNC_EXPORT &&
            exportTemplatesByNode[node] &&
            Object.keys(exportTemplatesByNode[node]).length === 1) {
            // If the bulk action is an async export action and the page has exactly one template, then we skip the dialog
            return { id: Object.keys(exportTemplatesByNode[node])[0] };
        }
        if (bulkAction.configurationPage) {
            try {
                const result = await pageDialog(store, bulkAction.configurationPage, { contextPage, node }, { size: 'medium', height: 250 });
                if (!isEmpty(result)) {
                    return result;
                }
                return false;
            }
            catch {
                // If the custom dialog is rejected we stop the mutation process gracefully
                return false;
            }
        }
        return {};
    }, [contextPage, exportTemplatesByNode, store, tableFieldProperties.node]);
    const getSelectionFilter = React.useCallback(() => {
        const groupFilter = [];
        if (selectedGroupState) {
            Object.keys(selectedGroupState).forEach(key => {
                Object.keys(selectedGroupState[key]).forEach(key2 => {
                    if (selectedGroupState[key][key2].state !== GroupState.NOT_SELECTED) {
                        groupFilter.push(getGroupFilterValue({
                            group: {
                                key,
                                value: key2,
                                aggFunc: selectedGroupState[key][key2].aggFunc,
                            },
                            mode: 'server',
                        }));
                    }
                });
            });
        }
        return {
            ...(isSelectAllChecked &&
                getGraphFilter({
                    filterModel: getFilterModel(gridApi, groupByColumnField),
                    screenId,
                    tableFieldProperties,
                    activeOptionsMenuItem,
                })),
            _id: isSelectAllChecked || groupFilter.length > 0
                ? {
                    _nin: getAllUnselectedIds(gridApi),
                }
                : {
                    _in: gridApi.getSelectedNodes().map(row => row.data._id),
                },
            _or: groupFilter.length > 0 ? groupFilter : undefined,
        };
    }, [
        activeOptionsMenuItem,
        gridApi,
        groupByColumnField,
        isSelectAllChecked,
        screenId,
        selectedGroupState,
        tableFieldProperties,
    ]);
    const handleBulkAction = React.useCallback(async (bulkAction) => {
        try {
            if (selectedRowCount <= 0) {
                return;
            }
            onTelemetryEvent?.(`tableBulkActionTriggered-${bulkAction.id || camelCase(bulkAction.title)}`, {
                screenId,
                id: bulkAction.id || bulkAction.title,
            });
            const parameters = await getBulkActionParameters(bulkAction);
            if (parameters === false) {
                // The user rejected the dialog
                return;
            }
            await confirmationDialog(screenId, 'warn', bulkAction.title, localize('@sage/xtrem-ui/bulk-action-dialog-content', 'Perform this action on the selected items: {{itemCount}}', { itemCount: selectedRowCount }), {
                acceptButton: {
                    text: localize('@sage/xtrem-ui/ok', 'OK'),
                },
                cancelButton: {
                    text: localize('@sage/xtrem-ui/cancel', 'Cancel'),
                },
                size: 'small',
            });
            const filter = getSelectionFilter();
            const targetNode = graphApi.node(String(tableFieldProperties.node));
            try {
                await targetNode.asyncOperations?.[bulkAction.mutation]
                    ?.start({ trackingId: true }, {
                    filter: JSON.stringify(filter),
                    ...parameters,
                })
                    .execute();
                onClearSelection();
                showToast(localize('@sage/xtrem-ui/bulk-action-started', 'Action started on the selected items.'), {
                    type: 'success',
                });
            }
            catch (_err) {
                showToast(localize('@sage/xtrem-ui/bulk-action-error', 'Action could not be started, please try again.'), { type: 'error' });
            }
        }
        catch (error) {
            // User dismissed dialog, intentionally left empty
        }
    }, [
        selectedRowCount,
        getBulkActionParameters,
        screenId,
        getSelectionFilter,
        graphApi,
        tableFieldProperties.node,
        onClearSelection,
        onTelemetryEvent,
    ]);
    return (React.createElement("div", { className: "e-page-navigation-panel-bulk-actions-bar-wrapper" },
        React.createElement("div", { className: "e-page-navigation-panel-bulk-actions-bar" },
            React.createElement("span", { className: "e-page-navigation-panel-bulk-actions-bar-selected-items", "data-testid": "e-table-field-bulk-actions-bar-selected-items" }, localize('@sage/xtrem-ui/bulk-actions-bar-selected', 'Items selected: {{count}}', {
                count: selectedRowCount,
            })),
            bulkActions?.map(bulkAction => {
                const handleButtonClick = () => {
                    handleBulkAction(bulkAction);
                };
                const { mutation, title, buttonType, icon, isDestructive } = bulkAction;
                return (React.createElement("div", { key: mutation, "data-testid": getDataTestIdAttribute('table', title, mutation, 'bulk-action') },
                    React.createElement(ButtonComponent, { screenId: screenId, elementId: mutation, locale: locale, onFocus: noop, fieldProperties: {
                            title,
                            isTitleHidden: true,
                        }, isNested: true, onClick: handleButtonClick, buttonType: buttonType, icon: icon, isDestructive: isDestructive, value: title })));
            }),
            React.createElement("button", { className: "e-page-navigation-panel-bulk-actions-bar-clear-selection", "data-testid": "e-table-field-bulk-actions-bar-clear-selection", type: "button", onClick: onClearSelection }, localize('@sage/xtrem-ui/clear-selection', 'Clear selection')))));
}
//# sourceMappingURL=desktop-table-bulk-action-bar.js.map